/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.businessobject.options;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.krad.keyvalues.KeyValuesBase;
import org.kuali.kfs.module.ld.LaborPropertyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.rice.core.api.util.ConcreteKeyValue;
import org.kuali.rice.core.api.util.KeyValue;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

/**
 * Field Finder for Labor Origin Entry.
 */
public class LaborOriginEntryFieldFinder extends KeyValuesBase {

    /**
     * @see org.kuali.rice.kns.lookup.keyvalues.KeyValuesFinder#getKeyValues()
     */
    public List getKeyValues() {
        List activeLabels = new ArrayList();
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, "Fiscal Year"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, "Chart Code"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.ACCOUNT_NUMBER, "Account Number"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.SUB_ACCOUNT_NUMBER, "Sub-Account Number"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, "Object Code"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE, "Sub-Object Code"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, "Balance Type"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE, "Object Type"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.UNIVERSITY_FISCAL_PERIOD_CODE, "Fiscal Period"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.FINANCIAL_DOCUMENT_TYPE_CODE, "Document Type"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.FINANCIAL_SYSTEM_ORIGINATION_CODE, "Origin code"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.DOCUMENT_NUMBER, "Document Number"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.TRANSACTION_ENTRY_SEQUENCE_NUMBER, "Sequence Number"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.POSITION_NUMBER, "Position Number"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.PROJECT_CODE, "Project Code"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.TRANSACTION_LEDGER_ENTRY_DESC, "Description"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.TRANSACTION_LEDGER_ENTRY_AMOUNT, "Amount"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.TRANSACTION_DEBIT_CREDIT_CODE, "Debit Credit Indicator"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.TRANSACTION_DATE, "Transaction Date"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.ORGANIZATION_DOCUMENT_NUMBER, "Org Doc Number"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.ORGANIZATION_REFERENCE_ID, "Org Ref ID"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.REFERENCE_FIN_DOCUMENT_TYPE_CODE, "Ref Doc Type"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.FIN_SYSTEM_REF_ORIGINATION_CODE, "Ref Origin code"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.FINANCIAL_DOCUMENT_REFERENCE_NBR, "Ref Doc Number"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.FINANCIAL_DOCUMENT_REVERSAL_DATE, "Reversal Date"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.TRANSACTION_ENCUMBRANCE_UPDT_CD, "Enc Update Code"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.TRANSACTION_POSTING_DATE, "Transaction Posting Date"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.PAY_PERIOD_END_DATE, "Pay Period End Date"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.TRANSACTION_TOTAL_HOURS, "Trn Total Hours"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.PAYROLL_END_DATE_FISCAL_YEAR, "Payroll EndDate Fiscal Year"));
        activeLabels.add(new ConcreteKeyValue(LaborPropertyConstants.PAYROLL_END_DATE_FISCAL_PERIOD_CODE, "Payroll EndDate Fiscal Period Code"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.EMPLID, "Empl Id"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.EMPLOYEE_RECORD, "Empl Record"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.EARN_CODE, "Earn Code"));
        activeLabels.add(new ConcreteKeyValue(KFSPropertyConstants.PAY_GROUP, "Pay Group"));
        activeLabels.add(new ConcreteKeyValue(LaborPropertyConstants.SALARY_ADMINISTRATION_PLAN, "Salary Admin Plan"));
        activeLabels.add(new ConcreteKeyValue(LaborPropertyConstants.GRADE, "Grade"));
        activeLabels.add(new ConcreteKeyValue(LaborPropertyConstants.RUN_IDENTIFIER, "Run Id"));
        activeLabels.add(new ConcreteKeyValue(LaborPropertyConstants.LABORLEDGER_ORIGINAL_CHART_OF_ACCOUNTS_CODE, "Original Chart Code"));
        activeLabels.add(new ConcreteKeyValue(LaborPropertyConstants.LABORLEDGER_ORIGINAL_ACCOUNT_NUMBER, "Original Account Number"));
        activeLabels.add(new ConcreteKeyValue(LaborPropertyConstants.LABORLEDGER_ORIGINAL_SUB_ACCOUNT_NUMBER, "Original Sub-Account Numbere"));
        activeLabels.add(new ConcreteKeyValue(LaborPropertyConstants.LABORLEDGER_ORIGINAL_FINANCIAL_OBJECT_CODE, "Original Object Code"));
        activeLabels.add(new ConcreteKeyValue(LaborPropertyConstants.LABORLEDGER_ORIGINAL_FINANCIAL_SUB_OBJECT_CODE, "Original Sub-Object Code"));
        activeLabels.add(new ConcreteKeyValue(LaborPropertyConstants.HRMS_COMPANY, "Company"));
        activeLabels.add(new ConcreteKeyValue(LaborPropertyConstants.SET_ID, "SetId"));

        return activeLabels;
    }

    /**
     * Get field display name.
     *
     * @param fieldName
     * @return Returns the label
     */
    public String getFieldDisplayName(String fieldName) {
        for (Iterator iter = getKeyValues().iterator(); iter.hasNext(); ) {
            KeyValue klp = (KeyValue) iter.next();
            if (klp.getKey().equals(fieldName)) {
                return klp.getValue();
            }
        }

        return "Error";
    }

    /**
     * Get field name
     *
     * @param fieldDisplayName
     * @return Returns the key
     */
    public String getFieldName(String fieldDisplayName) {
        for (Iterator iter = getKeyValues().iterator(); iter.hasNext(); ) {
            KeyValue klp = (KeyValue) iter.next();
            if (klp.getValue().equals(fieldDisplayName)) {
                return (String) klp.getKey();
            }
        }

        return "Error";
    }

    /**
     * Check each field for valid value.
     *
     * @param fieldName
     * @param value
     * @return Returns a boolean
     */
    public boolean isValidValue(String fieldName, String value) {
        if (StringUtils.isBlank(fieldName)) {
            return false;
        }
        String fieldType = getFieldType(fieldName);
        int fieldLength = getFieldLength(fieldName);

        if (allowNull(fieldName) && (value == null || value.length() == 0)) {
            return true;
        }
        if (!allowNull(fieldName) && (value == null || value.length() == 0)) {
            return false;
        }
        if (value.length() > fieldLength) {
            return false;
        }
        if ("BigDecimal".equals(fieldType)) {
            try {
                BigDecimal d = new BigDecimal(value);
                return true;
            } catch (NumberFormatException nfe) {
                return false;
            }
        }
        if ("KualiDecimal".equals(fieldType)) {
            try {
                KualiDecimal d = new KualiDecimal(value);
                return true;
            } catch (NumberFormatException nfe) {
                return false;
            }
        } else if ("Integer".equals(fieldType)) {
            try {
                Integer d = new Integer(value);
                return true;
            } catch (NumberFormatException nfe) {
                return false;
            }
        } else if ("Date".equals(fieldType)) {
            SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
            try {
                Date d = df.parse(value);
                return true;
            } catch (ParseException e) {
                return false;
            }
        }

        return true;
    }

    /**
     * Get field type.
     *
     * @param fieldName
     * @return Returns the fieldType
     */
    public String getFieldType(String fieldName) {
        if ("universityFiscalYear".equals(fieldName)) {
            return "Integer";
        }
        if ("transactionLedgerEntrySequenceNumber".equals(fieldName)) {
            return "Integer";
        }
        if ("transactionLedgerEntryAmount".equals(fieldName)) {
            return "KualiDecimal";
        }
        if ("transactionDate".equals(fieldName)) {
            return "Date";
        }
        if ("financialDocumentReversalDate".equals(fieldName)) {
            return "Date";
        }
        if ("transactionPostingDate".equals(fieldName)) {
            return "Date";
        }
        if ("payPeriodEndDate".equals(fieldName)) {
            return "Date";
        }
        if ("transactionTotalHours".equals(fieldName)) {
            return "BigDecimal";
        }
        if ("payrollEndDateFiscalYear".equals(fieldName)) {
            return "Integer";
        }
        if ("transactionEntryProcessedTimestamp".equals(fieldName)) {
            return "Date";
        }
        if ("employeeRecord".equals(fieldName)) {
            return "Integer";
        }
        if ("transactionDateTimeStamp".equals(fieldName)) {
            return "Date";
        }
        return "String";
    }

    /**
     * Check if field allows null value
     *
     * @param fieldName
     * @return Returns a boolean
     */
    public boolean allowNull(String fieldName) {
        if ("transactionLedgerEntryAmount".equals(fieldName)) {
            return false;
        }
        return true;
    }

    /**
     * Get length of field from fieldName
     *
     * @param fieldName
     * @return Returns an int
     */
    public int getFieldLength(String fieldName) {
        if ("universityFiscalYear".equals(fieldName)) {
            return 4;
        } else if ("transactionLedgerEntrySequenceNumber".equals(fieldName)) {
            return 5;
        } else if ("transactionLedgerEntryAmount".equals(fieldName)) {
            return 20;
        } else if ("transactionDate".equals(fieldName)) {
            return 10;
        } else if ("financialDocumentReversalDate".equals(fieldName)) {
            return 10;
        } else if ("chartOfAccountsCode".equals(fieldName)) {
            return 2;
        } else if ("accountNumber".equals(fieldName)) {
            return 7;
        } else if ("subAccountNumber".equals(fieldName)) {
            return 5;
        } else if ("financialObjectCode".equals(fieldName)) {
            return 4;
        } else if ("financialSubObjectCode".equals(fieldName)) {
            return 3;
        } else if ("financialBalanceTypeCode".equals(fieldName)) {
            return 2;
        } else if ("financialObjectTypeCode".equals(fieldName)) {
            return 2;
        } else if ("universityFiscalPeriodCode".equals(fieldName)) {
            return 2;
        } else if ("financialDocumentTypeCode".equals(fieldName)) {
            return 4;
        } else if ("financialSystemOriginationCode".equals(fieldName)) {
            return 2;
        } else if (KFSPropertyConstants.DOCUMENT_NUMBER.equals(fieldName)) {
            return 14;
        } else if ("transactionLedgerEntryDescription".equals(fieldName)) {
            return 40;
        } else if ("transactionDebitCreditCode".equals(fieldName)) {
            return 1;
        } else if ("organizationDocumentNumber".equals(fieldName)) {
            return 10;
        } else if ("projectCode".equals(fieldName)) {
            return 10;
        } else if ("organizationReferenceId".equals(fieldName)) {
            return 8;
        } else if ("referenceFinancialDocumentTypeCode".equals(fieldName)) {
            return 4;
        } else if ("referenceFinancialSystemOriginationCode".equals(fieldName)) {
            return 2;
        } else if ("referenceFinancialDocumentNumber".equals(fieldName)) {
            return 14;
        } else if ("transactionEncumbranceUpdateCode".equals(fieldName)) {
            return 1;
        } else if ("positionNumber".equals(fieldName)) {
            return 8;
        } else if ("transactionPostingDate".equals(fieldName)) {
            return 10;
        } else if ("payPeriodEndDate".equals(fieldName)) {
            return 10;
        } else if ("transactionTotalHours".equals(fieldName)) {
            return 22;
        } else if ("payrollEndDateFiscalYear".equals(fieldName)) {
            return 4;
        } else if ("payrollEndDateFiscalPeriodCode".equals(fieldName)) {
            return 2;
        } else if ("emplid".equals(fieldName)) {
            return 11;
        } else if ("employeeRecord".equals(fieldName)) {
            return 3;
        } else if ("earnCode".equals(fieldName)) {
            return 3;
        } else if ("payGroup".equals(fieldName)) {
            return 3;
        } else if ("salaryAdministrationPlan".equals(fieldName)) {
            return 4;
        } else if ("grade".equals(fieldName)) {
            return 3;
        } else if ("runIdentifier".equals(fieldName)) {
            return 10;
        } else if ("laborLedgerOriginalChartOfAccountsCode".equals(fieldName)) {
            return 2;
        } else if ("laborLedgerOriginalAccountNumber".equals(fieldName)) {
            return 7;
        } else if ("laborLedgerOriginalSubAccountNumber".equals(fieldName)) {
            return 5;
        } else if ("laborLedgerOriginalFinancialObjectCode".equals(fieldName)) {
            return 4;
        } else if ("laborLedgerOriginalFinancialSubObjectCode".equals(fieldName)) {
            return 3;
        } else if ("hrmsCompany".equals(fieldName)) {
            return 3;
        } else if ("setid".equals(fieldName)) {
            return 5;
        }

        return 0;
    }
}
