/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.businessobject;

/**
 * Labor Base class for parsing serialized AccountingLines for Labor LedgerJournal Voucher
 */

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.service.BalanceTypeService;
import org.kuali.kfs.module.ld.LaborPropertyConstants;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.AccountingLineParserBase;
import org.kuali.kfs.sys.businessobject.SourceAccountingLine;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.exception.AccountingLineParserException;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.util.Map;

public class LaborJournalVoucherAccountingLineParser extends AccountingLineParserBase {
    private String balanceTypeCode;
    protected static final String[] LABOR_JV_FORMAT = {KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, KFSPropertyConstants.ACCOUNT_NUMBER, KFSPropertyConstants.SUB_ACCOUNT_NUMBER, KFSPropertyConstants.FINANCIAL_OBJECT_CODE,
        KFSPropertyConstants.OBJECT_TYPE_CODE, KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE, KFSPropertyConstants.PROJECT_CODE, KFSPropertyConstants.ORGANIZATION_REFERENCE_ID, KFSPropertyConstants.POSITION_NUMBER, KFSPropertyConstants.EMPLID, KFSPropertyConstants.EMPLOYEE_RECORD,
        LaborPropertyConstants.EARN_CODE, LaborPropertyConstants.PAY_GROUP, LaborPropertyConstants.SALARY_ADMINISTRATION_PLAN, LaborPropertyConstants.GRADE, LaborPropertyConstants.RUN_IDENTIFIER, LaborPropertyConstants.PAY_PERIOD_END_DATE, LaborPropertyConstants.PAYROLL_END_DATE_FISCAL_YEAR,
        LaborPropertyConstants.PAYROLL_END_DATE_FISCAL_PERIOD_CODE, LaborPropertyConstants.TRANSACTION_TOTAL_HOURS, LaborPropertyConstants.LABORLEDGER_ORIGINAL_CHART_OF_ACCOUNTS_CODE, LaborPropertyConstants.LABORLEDGER_ORIGINAL_ACCOUNT_NUMBER,
        LaborPropertyConstants.LABORLEDGER_ORIGINAL_SUB_ACCOUNT_NUMBER, LaborPropertyConstants.LABORLEDGER_ORIGINAL_FINANCIAL_OBJECT_CODE, LaborPropertyConstants.LABORLEDGER_ORIGINAL_FINANCIAL_SUB_OBJECT_CODE,
        LaborPropertyConstants.HRMS_COMPANY, KFSPropertyConstants.ENCUMBRANCE_UPDATE_CODE, LaborPropertyConstants.SET_ID, KFSPropertyConstants.DEBIT, KFSPropertyConstants.CREDIT};

    /**
     * Constructs a JournalVoucherAccountingLineParser.java.
     *
     * @param balanceTypeCode
     */
    public LaborJournalVoucherAccountingLineParser(String balanceTypeCode) {
        super();
        this.balanceTypeCode = balanceTypeCode;
    }

    /**
     * @see org.kuali.rice.krad.bo.AccountingLineParserBase#performCustomSourceAccountingLinePopulation(java.util.Map,
     * org.kuali.rice.krad.bo.SourceAccountingLine, java.lang.String)
     */
    @Override
    protected void performCustomSourceAccountingLinePopulation(Map<String, String> attributeValueMap, SourceAccountingLine sourceAccountingLine, String accountingLineAsString) {
        // chose debit/credit
        String debitValue = attributeValueMap.remove(KFSPropertyConstants.DEBIT);
        String creditValue = attributeValueMap.remove(KFSPropertyConstants.CREDIT);
        KualiDecimal debitAmount = null;
        try {
            if (StringUtils.isNotBlank(debitValue)) {
                debitAmount = new KualiDecimal(debitValue);
            }
        } catch (NumberFormatException e) {
            String[] errorParameters = {debitValue, retrieveAttributeLabel(sourceAccountingLine.getClass(), KFSPropertyConstants.DEBIT), accountingLineAsString};
            throw new AccountingLineParserException("invalid (NaN) '" + KFSPropertyConstants.DEBIT + "=" + debitValue + " for " + accountingLineAsString, KFSKeyConstants.AccountingLineParser.ERROR_INVALID_PROPERTY_VALUE, errorParameters);
        }
        KualiDecimal creditAmount = null;
        try {
            if (StringUtils.isNotBlank(creditValue)) {
                creditAmount = new KualiDecimal(creditValue);
            }
        } catch (NumberFormatException e) {
            String[] errorParameters = {creditValue, retrieveAttributeLabel(sourceAccountingLine.getClass(), KFSPropertyConstants.CREDIT), accountingLineAsString};
            throw new AccountingLineParserException("invalid (NaN) '" + KFSPropertyConstants.CREDIT + "=" + creditValue + " for " + accountingLineAsString, KFSKeyConstants.AccountingLineParser.ERROR_INVALID_PROPERTY_VALUE, errorParameters);
        }

        KualiDecimal amount = null;
        String debitCreditCode = null;
        if (debitAmount != null && debitAmount.isNonZero()) {
            amount = debitAmount;
            debitCreditCode = KFSConstants.GL_DEBIT_CODE;
        }

        if (creditAmount != null && creditAmount.isNonZero()) {
            amount = creditAmount;
            debitCreditCode = KFSConstants.GL_CREDIT_CODE;
        }

        sourceAccountingLine.setAmount(amount);
        sourceAccountingLine.setDebitCreditCode(debitCreditCode);

        boolean isFinancialOffsetGeneration = SpringContext.getBean(BalanceTypeService.class).getBalanceTypeByCode(balanceTypeCode).isFinancialOffsetGenerationIndicator();
        if (isFinancialOffsetGeneration || StringUtils.equals(balanceTypeCode, KFSConstants.BALANCE_TYPE_EXTERNAL_ENCUMBRANCE)) {
            super.performCustomSourceAccountingLinePopulation(attributeValueMap, sourceAccountingLine, accountingLineAsString);
        }
        sourceAccountingLine.setBalanceTypeCode(balanceTypeCode);
    }

    /**
     * @see org.kuali.rice.krad.bo.AccountingLineParserBase#getSourceAccountingLineFormat()
     */
    @Override
    public String[] getSourceAccountingLineFormat() {
        return removeChartFromFormatIfNeeded(selectFormat());
    }

    /**
     * chooses line format based on balance type code
     *
     * @return String []
     */
    private String[] selectFormat() {
        return LABOR_JV_FORMAT;
    }
}
