/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.dataaccess;

import org.kuali.kfs.module.ld.businessobject.LedgerEntry;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This is the data access object for ledger entry.
 *
 * @see org.kuali.kfs.module.ld.businessobject.LedgerEntry
 */
public interface LaborLedgerEntryDao {

    /**
     * The sequence number is one of the primary keys of ledger entry. The entries can be grouped by other keys. This
     * method is used to get the maximum sequence number in the group of entries.
     *
     * @param ledgerEntry the given ledger entry
     * @return the maximum sequence number in a group of entries. If the group doesn't exist, return 0.
     */
    Integer getMaxSquenceNumber(LedgerEntry ledgerEntry);

    /**
     * Find the ledger entries that satisfy the all entries in the given field-value pair
     *
     * @param fieldValues the given field-value pair
     * @return the ledger entries that satisfy the all entries in the given field-value pair
     */
    Iterator<LedgerEntry> find(Map<String, String> fieldValues);

    /**
     * find the employees who were paid based on a set of specified pay type within the given report periods. Here, a
     * pay type can be determined by earn code and pay group.
     *
     * @param payPeriods          the given pay periods
     * @param balanceTypes        the specified balance type codes
     * @param earnCodePayGroupMap the combination of earn codes and pay groups, where pay group is the key and earn
     *                            code set is the value
     * @return the employees who were paid based on a set of specified pay type within the given report periods
     */
    List<String> findEmployeesWithPayType(Map<Integer, Set<String>> payPeriods, List<String> balanceTypes,
            Map<String, Set<String>> earnCodePayGroupMap);

    /**
     * get the ledger entries for the given employee based on a set of specified pay type within the given report
     * periods. Here, a pay type can be determined by earn code and pay group.
     *
     * @param emplid              the given employee id
     * @param payPeriods          the given pay periods
     * @param balanceTypes        the specified balance type codes
     * @param earnCodePayGroupMap the combination of earn codes and pay groups, where pay group is the key and earn
     *                            code set is the value
     * @return true if the given employee was paid based on a set of specified pay type within the given report
     *         periods; otherwise, false
     */
    Collection<LedgerEntry> getLedgerEntriesForEmployeeWithPayType(String emplid, Map<Integer, Set<String>> payPeriods,
            List<String> balanceTypes, Map<String, Set<String>> earnCodePayGroupMap);

    /**
     * determine if the given employee is associated with a set of specified pay type within the given report periods.
     * Here, a pay type can be determined by earn code and pay group.
     *
     * @param emplid              the given employee id
     * @param payPeriods          the given pay periods
     * @param balanceTypes        the specified balance type codes
     * @param earnCodePayGroupMap the combination of earn codes and pay groups, where pay group is the key and earn
     *                            code set is the value
     * @return true if the given employee was paid based on a set of specified pay type within the given report
     *         periods; otherwise, false
     */
    boolean isEmployeeWithPayType(String emplid, Map<Integer, Set<String>> payPeriods, List<String> balanceTypes,
            Map<String, Set<String>> earnCodePayGroupMap);

    /**
     * delete the ledger entry records that were posted prior to the given fiscal year
     *
     * @param fiscalYear          the given fiscal year
     * @param chartOfAccountsCode the given chart of account code
     */
    void deleteLedgerEntriesPriorToYear(Integer fiscalYear, String chartOfAccountsCode);
}
