/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.gl.businessobject.OriginEntryFull;
import org.kuali.kfs.datadictionary.legacy.DataDictionaryService;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntry;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Global constants for labor distribution.
 */
public final class LaborConstants {

    public static final String BASE_FUNDS_LOOKUP_HELPER_SERVICE_NAME = "BaseFundsLookupableHelperService";
    public static final String CURRENT_FUNDS_LOOKUP_HELPER_SERVICE_NAME = "CurrentFundsLookupableHelperService";
    public static final String DASHES_DELETE_CODE = "-";

    public static final String DOCUMENT_EMPLOYEE_ID_ERRORS = "document.emplid";
    public static final String EMPLOYEE_LOOKUP_ERRORS = "document.employeeLookups";
    public static final String LONG_ROW_TABLE_INQUIRY_ACTION = "laborLongRowTableInquiry.do";
    public static final Map<String, String> periodCodeMapping = new HashMap<>();

    public static final String ANNUAL_CLOSING_DOCUMENT_TYPE_CODE = getAnnualClosingDocumentType();
    public static final String[] ACCOUNT_FIELDS = {KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE,
        KFSPropertyConstants.ACCOUNT_NUMBER};
    public static final String LABOR_MODULE_CODE = "KFS-LD";

    public static final String LABOR_ENTERPRISE_FEEDER_FILE_SET_TYPE_IDENTIFIER = "laborEnterpriseFeederFileSetType";

    private static String SPACE_TRANSACTION_DATE = null;
    private static String DASH_POSITION_NUMBER = null;
    private static String DASH_EMPLID = null;

    /**
     * Private Constructor since this is a constants class that should never be instantiated.
     */
    private LaborConstants() {
    }

    static {
        periodCodeMapping.put(KFSPropertyConstants.MONTH1_AMOUNT, KFSConstants.MONTH1);
        periodCodeMapping.put(KFSPropertyConstants.MONTH2_AMOUNT, KFSConstants.MONTH2);
        periodCodeMapping.put(KFSPropertyConstants.MONTH3_AMOUNT, KFSConstants.MONTH3);
        periodCodeMapping.put(KFSPropertyConstants.MONTH4_AMOUNT, KFSConstants.MONTH4);
        periodCodeMapping.put(KFSPropertyConstants.MONTH5_AMOUNT, KFSConstants.MONTH5);
        periodCodeMapping.put(KFSPropertyConstants.MONTH6_AMOUNT, KFSConstants.MONTH6);
        periodCodeMapping.put(KFSPropertyConstants.MONTH7_AMOUNT, KFSConstants.MONTH7);
        periodCodeMapping.put(KFSPropertyConstants.MONTH8_AMOUNT, KFSConstants.MONTH8);
        periodCodeMapping.put(KFSPropertyConstants.MONTH9_AMOUNT, KFSConstants.MONTH9);
        periodCodeMapping.put(KFSPropertyConstants.MONTH10_AMOUNT, KFSConstants.MONTH10);
        periodCodeMapping.put(KFSPropertyConstants.MONTH11_AMOUNT, KFSConstants.MONTH11);
        periodCodeMapping.put(KFSPropertyConstants.MONTH12_AMOUNT, KFSConstants.MONTH12);
        periodCodeMapping.put(KFSPropertyConstants.MONTH13_AMOUNT, KFSConstants.MONTH13);
    }

    public static List<String> consolidationAttributesOfOriginEntry() {
        List<String> consolidationAttributes = new ArrayList<>();

        consolidationAttributes.add(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR);
        consolidationAttributes.add(KFSPropertyConstants.UNIVERSITY_FISCAL_PERIOD_CODE);

        consolidationAttributes.add(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        consolidationAttributes.add(KFSPropertyConstants.ACCOUNT_NUMBER);
        consolidationAttributes.add(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);

        consolidationAttributes.add(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE);
        consolidationAttributes.add(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
        consolidationAttributes.add(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);
        consolidationAttributes.add(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE);

        consolidationAttributes.add(KFSPropertyConstants.FINANCIAL_DOCUMENT_TYPE_CODE);
        consolidationAttributes.add(KFSPropertyConstants.DOCUMENT_NUMBER);
        consolidationAttributes.add(KFSPropertyConstants.ORGANIZATION_DOCUMENT_NUMBER);

        consolidationAttributes.add(KFSPropertyConstants.FINANCIAL_SYSTEM_ORIGINATION_CODE);
        consolidationAttributes.add(KFSPropertyConstants.PROJECT_CODE);
        consolidationAttributes.add(KFSPropertyConstants.ORGANIZATION_REFERENCE_ID);
        consolidationAttributes.add(KFSPropertyConstants.TRANSACTION_LEDGER_ENTRY_DESC);

        // Reference Document Type, Reference Origin Code, Reference Document Number, Reversal Date, and Encumbrance
        // Update Code.
        consolidationAttributes.add(KFSPropertyConstants.REFERENCE_FINANCIAL_DOCUMENT_TYPE_CODE);
        consolidationAttributes.add(KFSPropertyConstants.REFERENCE_FINANCIAL_SYSTEM_ORIGINATION_CODE);
        consolidationAttributes.add(KFSPropertyConstants.REFERENCE_FINANCIAL_DOCUMENT_NUMBER);
        consolidationAttributes.add(KFSPropertyConstants.FINANCIAL_DOCUMENT_REVERSAL_DATE);
        consolidationAttributes.add(KFSPropertyConstants.TRANSACTION_ENCUMBRANCE_UPDT_CD);

        consolidationAttributes.add(KFSPropertyConstants.TRANSACTION_DEBIT_CREDIT_CODE);
        consolidationAttributes.add(KFSPropertyConstants.TRANSACTION_LEDGER_ENTRY_AMOUNT);

        return consolidationAttributes;
    }

    public static String getSpaceTransactionDate() {
        if (SPACE_TRANSACTION_DATE == null) {
            SPACE_TRANSACTION_DATE = StringUtils.rightPad("", SpringContext.getBean(DataDictionaryService.class)
                    .getAttributeSize(OriginEntryFull.class, KFSPropertyConstants.TRANSACTION_DATE), ' ');
        }
        return SPACE_TRANSACTION_DATE;
    }

    private static String getAnnualClosingDocumentType() {
        return SpringContext.getBean(ParameterService.class).getParameterValueAsString(
                KfsParameterConstants.GENERAL_LEDGER_BATCH.class,
                KFSConstants.SystemGroupParameterNames.GL_ANNUAL_CLOSING_DOC_TYPE);
    }

    public static String getDashPositionNumber() {
        if (DASH_POSITION_NUMBER == null) {
            DASH_POSITION_NUMBER = StringUtils.rightPad("", SpringContext.getBean(DataDictionaryService.class)
                    .getAttributeMaxLength(LaborOriginEntry.class, KFSPropertyConstants.POSITION_NUMBER), '-');
        }
        return DASH_POSITION_NUMBER;
    }

    public static String getDashEmplId() {
        if (DASH_EMPLID == null) {
            DASH_EMPLID = StringUtils.rightPad("", SpringContext.getBean(DataDictionaryService.class)
                    .getAttributeMaxLength(LaborOriginEntry.class, KFSPropertyConstants.EMPLID), '-');
        }
        return DASH_EMPLID;
    }

    public static class BalanceInquiries {
        public static final String ANNUAL_BALANCE = "ACLN_ANNL_BAL_AMT";
        public static final String BALANCE_TYPE_AC_AND_A21 = "AC&A2";
        public static final String CONTRACT_GRANT_BB_AMOUNT = "CONTR_GR_BB_AC_AMT";
        public static final String EMPLOYEE_FUNDING_EXPENSE_OBJECT_TYPE_CODE = "ES";
        public static final String EMPLOYEE_FUNDING_NORMAL_OP_EXPENSE_OBJECT_TYPE_CODE = "EX";
        public static final String LEDGER_ENTRY_LOOKUPABLE_FOR_EXPENSE_TRANSFER =
                "laborLedgerEntryForExpenseTransferLookupable";
        public static final String UnknownPersonName = "* Unknown Name *";
    }

    public static class ColumnNames {
        public static final String UNIVERSITY_FISCAL_YEAR = "UNIV_FISCAL_YR";
        public static final String TRANSACTION_LEDGER_ENTRY_AMOUNT = "TRN_LDGR_ENTR_AMT";
        public static final String POSITION_NUMBER = "POSITION_NBR";
        public static final String EMPLOYEE_IDENTIFIER = "EMPLID";
    }

    public static class BenefitExpenseTransfer {
        public static final String LABOR_LEDGER_BENEFIT_CODE = "F";
    }

    public static class DestinationNames {
        public static final String LABOR_GL_ENTRY = "LD_LBR_GL_ENTRY_T";
        public static final String LEDGER_BALANCE = "LD_LDGR_BAL_T";
        public static final String LEDGER_ENTRY = "LD_LDGR_ENTR_T";
        public static final String ORIGIN_ENTRY = "LD_LBR_ORIGIN_ENTRY_T";
    }

    public enum JournalVoucherOffsetType {
        ACCRUAL("LJVA", "Accrual", "LLJV ACCRUALS OFFSET"),
        CASH("LJVC", "Cash", "LLJV CASH OFFSET"),
        ENCUMBRANCE("LJVE", "Encumbrance", "LLJV ENCUMBRANCE OFFSET"),
        NO_OFFSET("LLJV", "No Offset", "LABOR LEDGER JOUNAL VOUCHER");

        public String description;
        public String longDescription;
        public String typeCode;

        JournalVoucherOffsetType(String typeCode, String description, String longDescription) {
            this.typeCode = typeCode;
            this.description = description;
            this.longDescription = longDescription;
        }
    }

    public static class LookupableBeanKeys {
        public static final String CSF_TRACKER = "laborCalculatedSalaryFoundationTrackerLookupable";
    }

    public static class SalaryExpenseTransfer {
        public static final String LABOR_LEDGER_SALARY_CODE = "S";
        public static final String EFFORT_VALIDATION_OVERRIDE_QUESTION = "EffortValidationOverrideQuestion";
    }

    public static class BatchFileSystem {
        public static final String EXTENSION = ".data";
        public static final String DONE_FILE_EXTENSION = ".done";

        public static final String NIGHTLY_OUT_FILE = "ld_labentry_kfs";
        public static final String BACKUP_FILE = "ld_ldbackup";
        public static final String PRE_SCRUBBER_FILE = "ld_prescrub";
        public static final String SCRUBBER_INPUT_FILE = "ld_sortscrb";

        public static final String SCRUBBER_VALID_OUTPUT_FILE = "ld_scrbout1";
        public static final String SCRUBBER_ERROR_OUTPUT_FILE = "ld_scrberr1";
        public static final String SCRUBBER_EXPIRED_OUTPUT_FILE = "ld_expaccts";

        public static final String SCRUBBER_ERROR_SORTED_FILE = "ld_sorterr1";
        public static final String DEMERGER_VALID_OUTPUT_FILE = "ld_scrbout2";
        public static final String DEMERGER_ERROR_OUTPUT_FILE = "ld_scrberr2";

        public static final String POSTER_INPUT_FILE = "ld_sortpost";
        public static final String POSTER_VALID_OUTPUT_FILE = "ld_postout";
        public static final String POSTER_ERROR_OUTPUT_FILE = "ld_posterrs";

        public static final String LABOR_GL_ENTRY_FILE = "gl_glentry_lab";
        public static final String BALANCE_FORWARDS_FILE = "ld_balance_forwards";
        public static final String LABOR_ENTERPRISE_FEED = "ld_ldentry_entp";
    }

    public static class PermissionNames {
        public static final String OVERRIDE_TRANSFER_IMPACTING_EFFORT_CERTIFICATION =
                "Override Transfer Impacting Open Effort Certification";
    }

    public static class BenefitCalculation {
        public static final String ENABLE_FRINGE_BENEFIT_CALC_BY_BENEFIT_RATE_CATEGORY_PARAMETER =
                "ENABLE_FRINGE_BENEFIT_CALC_BY_BENEFIT_RATE_CATEGORY_IND";
        public static final String DEFAULT_BENEFIT_RATE_CATEGORY_CODE_PARAMETER = "DEFAULT_BENEFIT_RATE_CATEGORY_CODE";
        public static final String ACCOUNT_CODE_OFFSET_PROPERTY_NAME = "accountCodeOffset";
        public static final String OBJECT_CODE_OFFSET_PROPERTY_NAME = "objectCodeOffset";
    }

    public static class LateAdjustment {
        public static final String GENERATE_LATE_ADJUSTMENT_STMT_ID = "GenerateLateAdjustStmt";
        public static final int NUM_LATE_ADJUST_FIELDS = 4;
        public static final Integer FISCAL_PERIODS_PER_YEAR = 12;
    }
}
