/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2016 The Kuali Foundation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.tem.document.web.struts;

import org.apache.commons.lang.StringUtils;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.service.DataDictionaryService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.tem.document.TravelDocument;
import org.kuali.kfs.module.tem.document.service.TravelDocumentService;
import org.kuali.rice.core.api.config.property.ConfigurationService;

import static org.kuali.kfs.module.tem.TemConstants.CONFIRM_AMENDMENT_QUESTION;
import static org.kuali.kfs.module.tem.TemConstants.RETURN_TO_FO_NOTE_PREFIX;
import static org.kuali.kfs.module.tem.TemConstants.RETURN_TO_FO_QUESTION;
import static org.kuali.kfs.module.tem.TemConstants.RETURN_TO_FO_TEXT;
import static org.kuali.kfs.module.tem.TemKeyConstants.ERROR_TA_REASON_PASTLIMIT;
import static org.kuali.kfs.module.tem.TemKeyConstants.ERROR_TA_REASON_REQUIRED;
import static org.kuali.kfs.module.tem.TemKeyConstants.TA_QUESTION_DOCUMENT;
import static org.kuali.kfs.sys.KFSConstants.BLANK_SPACE;
import static org.kuali.kfs.sys.KFSConstants.NOTE_TEXT_PROPERTY_NAME;
import static org.kuali.kfs.sys.KFSConstants.QUESTION_REASON_ATTRIBUTE_NAME;

public class ReturnToFiscalOfficerQuestionHandler implements QuestionHandler<TravelDocument> {
    private ConfigurationService ConfigurationService;
    private DataDictionaryService dataDictionaryService;
    private TravelDocumentService travelDocumentService;

    @Override
    public <T> T handleResponse(final Inquisitive<TravelDocument, ?> asker) throws Exception {
        if (asker.denied(RETURN_TO_FO_QUESTION)) {
            return (T) asker.back();
        } else if (asker.confirmed(CONFIRM_AMENDMENT_QUESTION)) {
            return (T) asker.end();
            // This is the case when the user clicks on "OK" in the end.
            // After we inform the user that the close has been rerouted, we'll redirect to the portal page.
        }

        final String noteStr = getReturnToFiscalOfficerNote(RETURN_TO_FO_NOTE_PREFIX, asker.getReason());
        final String message = getReturnToFiscalOfficerQuestion(RETURN_TO_FO_TEXT);

        int noteTextMaxLength = getDataDictionaryService().getAttributeMaxLength(Note.class, NOTE_TEXT_PROPERTY_NAME).intValue();
        String question = StringUtils.replace(message, "{0}", RETURN_TO_FO_TEXT);
        if (StringUtils.isBlank(asker.getReason())) {
            return (T) asker.confirm(RETURN_TO_FO_QUESTION, question, true, ERROR_TA_REASON_REQUIRED, QUESTION_REASON_ATTRIBUTE_NAME, RETURN_TO_FO_TEXT);
        } else if (noteStr.length() > noteTextMaxLength) {
            return (T) asker.confirm(RETURN_TO_FO_QUESTION, question, true, ERROR_TA_REASON_PASTLIMIT, QUESTION_REASON_ATTRIBUTE_NAME, new Integer(noteStr.length() - noteTextMaxLength).toString());
        }

        final TravelDocument document = asker.getDocument();
        getTravelDocumentService().routeToFiscalOfficer(document, noteStr);
        return (T) asker.finish();
    }

    @Override
    public <T> T askQuestion(final Inquisitive<TravelDocument, ?> asker) throws Exception {
        final String message = getReturnToFiscalOfficerQuestion(RETURN_TO_FO_TEXT);

        T retval = (T) asker.confirm(RETURN_TO_FO_QUESTION, message, true);
        return retval;

    }

    public String getReturnToFiscalOfficerQuestion(final String operation) {
        String message = "";
        final String key = getConfigurationService().getPropertyValueAsString(TA_QUESTION_DOCUMENT);
        message = StringUtils.replace(key, "{0}", operation);
        // Ask question if not already asked.
        return message;
    }

    public String getMessageFrom(final String messageType) {
        return getConfigurationService().getPropertyValueAsString(messageType);
    }

    public String getReturnToFiscalOfficerNote(final String notePrefix, String reason) {
        String noteText = "";
        // Have to check length on value entered.
        final String introNoteMessage = notePrefix + BLANK_SPACE;

        // Build out full message.
        noteText = introNoteMessage + reason;
        final int noteTextLength = noteText.length();

        // Get note text max length from DD.
        final int noteTextMaxLength = getDataDictionaryService().getAttributeMaxLength(Note.class, NOTE_TEXT_PROPERTY_NAME).intValue();

        if (StringUtils.isBlank(reason) || (noteTextLength > noteTextMaxLength)) {
            // Figure out exact number of characters that the user can enter.
            int reasonLimit = noteTextMaxLength - noteTextLength;

            if (ObjectUtils.isNull(reason)) {
                // Prevent a NPE by setting the reason to a blank string.
                reason = "";
            }
        }
        return noteText;
    }

    /**
     * Sets the ConfigurationService attribute.
     *
     * @return Returns the ConfigurationService.
     */
    public void setConfigurationService(final ConfigurationService ConfigurationService) {
        this.ConfigurationService = ConfigurationService;
    }

    /**
     * Gets the ConfigurationService attribute.
     *
     * @return Returns the ConfigurationService.
     */
    protected ConfigurationService getConfigurationService() {
        return ConfigurationService;
    }

    /**
     * Sets the travelDocumentService attribute.
     *
     * @return Returns the travelDocumentService.
     */
    public void setTravelDocumentService(final TravelDocumentService travelDocumentService) {
        this.travelDocumentService = travelDocumentService;
    }

    /**
     * Gets the travelDocumentService attribute.
     *
     * @return Returns the travelDocumentService.
     */
    protected TravelDocumentService getTravelDocumentService() {
        return travelDocumentService;
    }

    /**
     * Sets the dataDictionaryService attribute.
     *
     * @return Returns the dataDictionaryService.
     */
    public void setDataDictionaryService(final DataDictionaryService dataDictionaryService) {
        this.dataDictionaryService = dataDictionaryService;
    }

    /**
     * Gets the dataDictionaryService attribute.
     *
     * @return Returns the dataDictionaryService.
     */
    protected DataDictionaryService getDataDictionaryService() {
        return dataDictionaryService;
    }
}
