<%--

    The Kuali Financial System, a comprehensive financial management system for higher education.

    Copyright 2005-2020 Kuali, Inc.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

--%>
<div id="lookupModal"></div>
<script type="text/javascript">
    /**
     * This block of javascript is the glue that brings together maintenance documents and the new lookup framework.
     * Using the data-* properties on the lookup input icons, we can send the appropriate data to the lookup
     * react components providing the same functionality as the old lookups but using a modal and the new pages
     * to create a faster and easier experience for the user
     */

    function documentPathToSelector(path) {
        return '[name="' + path + '"]'
    }

    function getElementWithSelector(selector) {
        var namedElement = $(documentPathToSelector(selector))
        if (namedElement.length > 0) {
            return namedElement
        } else {
            var idElement = document.getElementById(selector);
            if (idElement) {
                return $(idElement)
            }
        }
    }

    /**
     * Processes an attribute string for field conversions. Converts a string of data to a map where the keys are
     * the path of the document and the value is the key of the fieldName
     */
    function getParameterMap(paramString) {
        var parameterMap = {}
        var params = paramString.split(',')
        params.forEach(function(param) {
            if (param.length > 0) {
                var paramParts = param.split(':')
                var documentPath = paramParts[0]
                var resultKey = paramParts[1]
                parameterMap[documentPath] = resultKey
            }
        })
        return parameterMap
    }

    /**
     * Takes a map of selectors:formFieldName
     * returns: a map with keys being the formFieldName and values being the element value for the provided selector
     */
    function getDocumentValuesFromParamMap(paramMap) {
        var documentValues = {}
        for (var fieldName in paramMap) {
            var documentPath = paramMap[fieldName]
            var selector = documentPathToSelector(documentPath)
            var element = $(selector)
            if (element.length > 0) {
                documentValues[fieldName] = element.val()
            }
        }
        return documentValues
    }

    /**
     * Takes a map of document path keys that are paired with formFieldNames. Using those document path keys we
     * create a selector to populate any relationships that can be inferred from the returned value of the lookup
     */
    function setDocumentValuesFromParamMap(paramMap, values) {
        for (var documentPath in paramMap) {
            var element = getElementWithSelector(documentPath)
            if (element && element.length > 0) {
                var resultKey = paramMap[documentPath]
                var value = values[resultKey]
                if (element.is('input')) {
                    element.val(value)
                } else {
                    element.text(value)
                }
            }
        }
    }

    function propertiesFromButton(button) {
        var propertyMap = {
            // businessObjectName is the name of the business object we are looking up
            businessObjectName: 'data-business-object-name',

            // lookupParameters are the fields that should be populated as a result of chosen row in a lookup
            lookupParameters: 'data-lookup-parameters',

            // fieldConversions are fields to be pre-populated on a lookup if they are already filled out on the doc
            fieldConversions: 'data-field-conversions',

            // formFieldPropertyName is the name of the input field associated with the clicked lookup
            formFieldPropertyName: 'data-field-name',

            // readOnlyFields are fields to be read-only on a lookup
            readOnlyFields: 'data-read-only-fields'
        }

        var buttonProps = {}
        var attributes = button.attributes
        for (var propName in propertyMap) {
            var attributeName = propertyMap[propName]
            if (attributes[attributeName]) {
                buttonProps[propName] = attributes[attributeName].value
            }
        }

        return buttonProps
    }

    function newLookupClicked(e) {
        e.preventDefault();
        var buttonProps = propertiesFromButton(e.target)
        var conversionMap = getParameterMap(buttonProps.fieldConversions)

        // We don't want to populate the lookup with the corresponding input field value
        // If you are looking up this value and you have something typed into it, its doubtful that's what you
        // want to search on
        var conversionMapFields = Object.keys(conversionMap)
        conversionMapFields.forEach(function(field) {
            if (conversionMap[field] === buttonProps.formFieldPropertyName)  {
                delete conversionMap[field];
            }
        })
        var documentValues = getDocumentValuesFromParamMap(conversionMap)

        var businessObjectNameParts = buttonProps.businessObjectName.split(".");
        var readOnlyFields = (buttonProps.readOnlyFields && buttonProps.readOnlyFields.length > 0) ? buttonProps.readOnlyFields.split(",") : []
        if (businessObjectNameParts.length > 0) {
            var options = {
                businessObjectName: businessObjectNameParts[businessObjectNameParts.length-1],
                formData: documentValues,
                readOnlyFields: readOnlyFields
            }

            window.showLookupAdapter(options, function(returnedValue) {
                var lookupParameterMap = getParameterMap(buttonProps.lookupParameters)
                setDocumentValuesFromParamMap(lookupParameterMap, returnedValue)
            })
        }
        return false;
    }

    function multipleReturnLookupClicked(e) {
        e.preventDefault()
        var buttonProps = propertiesFromButton(e.target)
        var businessObjectNameParts = buttonProps.businessObjectName.split(".");
        if (businessObjectNameParts.length > 0) {
            var options = {
                businessObjectName: businessObjectNameParts[businessObjectNameParts.length-1],
                multiReturnEnabled: true
            }

            window.showLookupAdapter(options, function(returnedValue) {
                var resultsType
                if (Array.isArray(returnedValue)) {
                    resultsType = 'objects'
                } else if (typeof returnedValue === 'object') {
                    resultsType = 'query'
                }

                document.KualiForm['multiValueLookupResultsType'].value = resultsType
                document.KualiForm['multiValueLookupResults'].value = JSON.stringify(returnedValue)
                var submit = document.querySelector('[value="multivalue-lookup"]')
                if (submit) {
                    submit.click();
                }
            })
        }
        return false;
    }

    document.addEventListener("DOMContentLoaded", function() {
        var newLookups = document.querySelectorAll('[data-lookup-type="single"]')
        newLookups.forEach(function(lookupButton) {
            lookupButton.addEventListener('click', newLookupClicked);
        });

        var multipleReturnLookups = document.querySelectorAll('[data-lookup-type="multiple"]')
        multipleReturnLookups.forEach(function(lookupButton) {
            lookupButton.addEventListener('click', multipleReturnLookupClicked);
        });
    })
</script>
