/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kcb.service;

import java.lang.reflect.Proxy;

import org.kuali.rice.kcb.util.BeanFactoryInvocationHandler;
import org.springframework.beans.factory.BeanFactory;

/**
 * Class that holds a singleton reference to KCBServiceLocator
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class GlobalKCBServiceLocator {

    /**
     * The KCBServiceLocator singleton
     */
    private static KCBServiceLocator locator;

    /**
     * The global initializer that constructs the KCBServiceLocator singleton
     * @param beanFactory the beanFactory from which to construct the KCBServiceLocator
     */
    public static synchronized void init(BeanFactory beanFactory) {
        if (locator != null) {
            throw new IllegalStateException("GlobalKCBServiceLocator already initialized");
        }

        locator = (KCBServiceLocator) Proxy.newProxyInstance(GlobalKCBServiceLocator.class.getClassLoader(),
                new Class[] { KCBServiceLocator.class },
                new BeanFactoryInvocationHandler(beanFactory));
    }

    /**
     * Returns whether the GlobalKCBServiceLocator has already been initialized (classloader scoped)
     * @return whether the GlobalKCBServiceLocator has already been initialized (classloader scoped)
     */
    public static synchronized boolean isInitialized() {
        return locator != null;
    }

    /**
     * Un-sets the KCBServiceLocator singleton, in order to fulfill a "lifecycle"
     * contract (whereby init may be called again in the same class loader), specifically for
     * unit tests.
     */
    public static synchronized void destroy() {
        locator = null;
    }

    /**
     * Returns the KCBServiceLocator singleton
     * @return the KCBServiceLocator singleton
     */
    public static synchronized KCBServiceLocator getInstance() {
        return locator;
    }
}
