/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ken.util;

import javax.xml.XMLConstants;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * Utility base class for Entity and LSResource Resolvers that should resolve
 * arguments to resources in the ClassLoader.
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class ClassLoaderResourceResolver {
    protected final Logger LOG = LogManager.getLogger(getClass());

    /**
     * This contains definitions for items in the core "xml" schema, i.e. base, id, lang, and space attributes. 
     */
    private static final String XML_NAMESPACE_SCHEMA = "http://www.w3.org/2001/xml.xsd";
    private static final String XSD_NAMESPACE_SCHEMA = XMLConstants.W3C_XML_SCHEMA_NS_URI;

    /**
     * Root path in class loader.  Defaults to "schema".
     */
    protected final String base;
    /**
     * Prefix of resources to honor.  Defaults to "" (no specific prefix).
     */
    protected final String prefix;

    public ClassLoaderResourceResolver() {
        this.base = "schema";
        this.prefix = "";
    }

    public ClassLoaderResourceResolver(String base, String prefix) {
        this.base = base;
        this.prefix = prefix;
    }

    protected String resolveSystemId(String systemId) {
        if (systemId.equals(XML_NAMESPACE_SCHEMA)) {
            return base + "/xml.xsd";
        } else if (systemId.equals(XSD_NAMESPACE_SCHEMA)) {
            return base + "/XMLSchema.xsd";
        } else if (systemId.startsWith("resource:" + prefix +"/")) {
            /* It turns out that the stock XMLSchema.xsd refers to XMLSchema.dtd in a relative
               fashion which results in the parser qualifying it to some local file:// path
               which breaks our detection here.
               So I have made a small mod to the stock XMLSchema.xsd so that it instead refers to
               resource:XMLSchema.dtd which can be looked up locally.
               The same is true for XMLSchema.dtd with regard to datatypes.dtd, so I have also
               modified XMLSchema.dtd to refer to resource:datatypes.dtd.
               An alternative would be to rely on publicId, however that would essentially hard code
               the lookup to always be in the classpath and rule out being able to redirect the location
               of the physical resource through the systemId, which is useful.
            */

            String path = base + "/" + systemId.substring(("resource:" + prefix + "/").length());
            // ok, if the path does not itself end in .xsd or .dtd, it is bare/abstract
            // so realize it by appending .xsd
            // this allows us to support looking up files ending with ".dtd" through resource: without
            // having extra logic to attempt to look up both suffixes for every single resource:
            // (all of which except XMLSchema.dtd and datatypes.dtd at this point are .xsd files)
            if (!(systemId.endsWith(".xsd") || systemId.endsWith(".dtd"))) {
                path += ".xsd";
            }
            return path;
        } else {
            return null;
        }
    }
}
