/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ken.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;

import javax.xml.namespace.NamespaceContext;

/**
 * XPath NamespaceContext that is configured with a predefined prefix->NS map.
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class ConfiguredNamespaceContext implements NamespaceContext {

    private Map<String, String> prefixToNamespace = new HashMap<String, String>();
    private Map<String, HashSet<String>> namespaceToPrefix = new HashMap<String, HashSet<String>>();

    public ConfiguredNamespaceContext(Map<String, String> prefixToNamespace) {
        this.prefixToNamespace = prefixToNamespace;
        // create a reverse namespace to prefix(es) map
        for (Map.Entry<String, String> entry: prefixToNamespace.entrySet()) {
            String namespace = entry.getValue();
            String prefix = entry.getKey();
            HashSet<String> prefixes = namespaceToPrefix.get(namespace);
            if (prefixes == null) {
                prefixes = new HashSet<String>(4);
                namespaceToPrefix.put(namespace, prefixes);
            }
            prefixes.add(prefix);
        }
    }

    @Override
    public String getNamespaceURI(String prefix) {
        //LOG.trace("getNamespaceURI(" + prefix + ")");
        if (prefix == null) {
            throw new IllegalArgumentException("The prefix cannot be null.");
        }

        return prefixToNamespace.get(prefix);
    }

    @Override
    public String getPrefix(String namespaceURI) {
        //LOG.trace("getPrefix(" + namespaceURI + ")");
        if (namespaceURI == null) {
            throw new IllegalArgumentException("The namespace uri cannot be null.");
        }
        Iterator<String> prefixes = getPrefixes(namespaceURI);
        if (prefixes != null) {
            return prefixes.next();
        } else {
            return null;
        }
    }

    @Override
    public Iterator<String> getPrefixes(String namespaceURI) {
        //LOG.trace("getPrefixes(" + namespaceURI + ")");
        if (namespaceURI == null) {
            throw new IllegalArgumentException("The namespace uri cannot be null.");
        }

        HashSet<String> prefixes = namespaceToPrefix.get(namespaceURI);
        if (prefixes != null && prefixes.size() > 0) {
            return prefixes.iterator();
        } else {
            return null;
        }
    }
}
