/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ken.util;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Iterator;

import javax.xml.transform.ErrorListener;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.core.api.util.xml.SafeXmlUtils;

/**
 * This class handles XSLT transformations.
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class ContentTransformer {
    private static final Logger LOG = LogManager.getLogger(ContentTransformer.class);

    private static final class LoggingErrorListener implements ErrorListener {
        private final ErrorListener delegate;
        public LoggingErrorListener(ErrorListener delegate) {
            this.delegate = delegate;
        }

        @Override
        public void error(TransformerException exception) {
            LOG.error("Error transforming document", exception);
        }

        @Override
        public void fatalError(TransformerException exception) throws TransformerException {
            if (delegate != null) {
                delegate.fatalError(exception);
            } else {
                throw exception;
            }
            
        }

        @Override
        public void warning(TransformerException exception) {
            LOG.warn("Error transforming document", exception);
        }
        
    };

    private Transformer transformer;

    public ContentTransformer(StreamSource aStyleSheet) throws Exception {
        // create transformer        
        TransformerFactory factory = SafeXmlUtils.safeTransformerFactory();
        transformer = factory.newTransformer( aStyleSheet );
    }

    public ContentTransformer(StreamSource aStyleSheet, HashMap parametermap) throws Exception {
       // create transformer
       TransformerFactory factory = SafeXmlUtils.safeTransformerFactory();
       transformer = factory.newTransformer( aStyleSheet );
       Iterator iter = parametermap.keySet().iterator();
       while (iter.hasNext()) {
          Object o = iter.next();
          String param = o.toString();
          String value = (String) parametermap.get(param);
          transformer.setParameter(param, value);
       }
       transformer.setErrorListener(new LoggingErrorListener(transformer.getErrorListener()));
    }

    /**
     * This method performs the actual transformation.
     */
    public String transform(String xml) throws Exception {

        // perform transformation
        Source xmlsource = new StreamSource(new StringReader(xml));
        StringWriter sout = new StringWriter();
         
        transformer.transform(xmlsource, new StreamResult(sout));

        // return resulting document
        return sout.toString();
    }
}
