/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.batch;

import org.apache.commons.io.IOUtils;
import org.kuali.rice.core.api.CoreApiServiceLocator;
import org.kuali.rice.core.api.config.ConfigurationException;
import org.kuali.rice.core.api.impex.xml.FileXmlDocCollection;
import org.kuali.rice.core.api.impex.xml.XmlDoc;
import org.kuali.rice.core.api.impex.xml.XmlDocCollection;
import org.kuali.rice.core.api.util.ClasspathOrFileResourceLoader;
import org.kuali.rice.kew.api.WorkflowRuntimeException;
import org.springframework.core.io.Resource;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

public class KEWXmlDataLoader {

    /**
     * Loads the XML specified by the resource string, which should be in Spring resource notation
     * @param resource resource string in Spring resource notation
     * @throws Exception 
     */
    public static void loadXmlResource(String resource) throws Exception {
        Resource res = new ClasspathOrFileResourceLoader().getResource(resource);
        InputStream xmlFile = res.getInputStream();
        try (xmlFile) {
            if (xmlFile == null) {
                throw new ConfigurationException("Didn't find resource " + resource);
            }
            loadXmlStream(xmlFile);
        }

    }

    /**
     * Loads the XML resource from the classloader, from the package of the specified class
     * if the class appears relative, or from the root of the classloader if it contains a slash
     * @param clazz the class whose package should be used to qualify the path
     * @param path the package-relative path of the XML resource
     * @throws Exception
     */
    public static void loadXmlClassLoaderResource(Class clazz, String path) throws Exception {
        if (path.indexOf('/') < 0) {
            loadXmlPackageResource(clazz, path);
        } else {
            loadXmlClassLoaderResource(clazz.getClassLoader(), path);
        }
    }

    /**
     * Loads the XML resource from the classloader, from the package of the specified class.
     * @param clazz the class whose package should be used to qualify the path
     * @param path the package-relative path of the XML resource
     * @throws Exception
     */
    public static void loadXmlPackageResource(Class clazz, String path) throws Exception {
        InputStream xmlFile = clazz.getResourceAsStream(path);
        try (xmlFile) {
            if (xmlFile == null) {
                throw new WorkflowRuntimeException("Didn't find resource " + path);
            }
            loadXmlStream(xmlFile);
        }
    }

    /**
     * Loads the XML resource from the specified classloader
     * @param classloader the classloader from which to load the resource
     * @param path the classloader path of the XML resource
     * @throws Exception
     */
    public static void loadXmlClassLoaderResource(ClassLoader classloader, String path) throws Exception {
        InputStream xmlFile = classloader.getResourceAsStream(path);
        try (xmlFile) {
            if (xmlFile == null) {
                throw new WorkflowRuntimeException("Didn't find resource " + path);
            }
            loadXmlStream(xmlFile);
        }
    }

    /**
     * Load the XML file from the file system.
     * 
     * @param fileName the path to the XML file
     * @throws Exception
     */
    public static void loadXmlFile(String fileName) throws Exception {
        try (FileInputStream fis = new FileInputStream(fileName)) {
            loadXmlStream(fis);
        }
    }

    /**
     * Loads XML from a stream
     * @param xmlStream the XML byte stream
     * @throws Exception
     */
    public static void loadXmlStream(InputStream xmlStream) throws Exception {
       List<XmlDocCollection> xmlFiles = new ArrayList<XmlDocCollection>();
        XmlDocCollection docCollection = getFileXmlDocCollection(xmlStream, "UnitTestTemp");
        //XmlDocCollection docCollection = new StreamXmlDocCollection(xmlStream);
        xmlFiles.add(docCollection);
        CoreApiServiceLocator.getXmlIngesterService().ingest(xmlFiles);
        for (XmlDoc doc: docCollection.getXmlDocs()) {
            if (!doc.isProcessed()) {
                throw new RuntimeException("Failed to ingest xml doc: " + doc.getName());
            }
        }
    }

    /**
     * Helper method that turns a stream into a FileXmlDocCollection by first making a copy on the file system.
     * @param xmlFile
     * @param tempFileName
     * @return
     * @throws IOException
     */
    public static FileXmlDocCollection getFileXmlDocCollection(InputStream stream, String tempFileName) throws IOException {
        if (stream == null) {
            throw new RuntimeException("Stream is null!");
        }

        File temp = File.createTempFile(tempFileName, ".xml");
        temp.deleteOnExit();

        try (FileOutputStream fos = new FileOutputStream(temp)) {
            IOUtils.copy(stream, fos);
        }

        return new FileXmlDocCollection(temp);
    }
}
