/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.routemodule;

import java.util.List;

import org.kuali.rice.kew.actionrequest.ActionRequestValue;
import org.kuali.rice.kew.api.exception.WorkflowException;
import org.kuali.rice.kew.engine.RouteContext;
import org.kuali.rice.kew.util.ResponsibleParty;


/**
 * A RouteModule is responsible for generating Action Requests for a given Route Header document.
 * Implementations of this class are potentially remotable, so this interface uses value objects.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public interface RouteModule {

    /**
     * Generate action requests for the given RouteContext.
     *
     * <p>The list of requests returned should only contain root action requests (those with no parent).</p>
     *
     * @return A List of the generated ActionRequestValue objects.
     */
    List<ActionRequestValue> findActionRequests(RouteContext context) throws Exception;

    /**
     * The route module will re-resolve the given responsibilityId and return an object that contains the key to either
     * a user, group, or role.  The original responsibility associated with the given id will then be swapped with the
     * new ResponsibleParty when routing is resolved.
     *
     * <p>This method is intended to allow for the calling code to be able to "switch-out" one responsibility for
     * another (in the case of a change to an assigned reponsibility).</p>
     *
     * <p>Returning null from this method will indicate that the original responsibile party should remain unchanged.
     * This is typically the recommended default for implementation of this method and covers most use cases.</p>
     *
     * @param responsibilityId the id of the responsibility to be resolved
     * @return the responsibility party with which to replace the original responsibility, or null if the responsibility
     * should remain unchanged.
     */
    ResponsibleParty resolveResponsibilityId(String responsibilityId) throws WorkflowException;

    /**
     * Returns true if this RouteModule has more requests available which it can generate.  This method will only be
     * called after the first set of action requests returned by the first invocation of findActionRequests have been
     * fully activated.  If this method returns true, findActionRequests will be invoked again and those requests will
     * all be activated before the cycle continues until this method returns false;
     *
     * @param context the route context
     * @return true if this route module has more requests it can generate, false otherwise
     */
    boolean isMoreRequestsAvailable(RouteContext context);

}
