/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.xml.export;

import org.jdom2.Document;
import org.jdom2.Element;
import org.kuali.rice.core.api.impex.ExportDataSet;
import org.kuali.rice.core.api.util.xml.SafeXmlUtils;
import org.kuali.rice.core.api.util.xml.XmlHelper;
import org.kuali.rice.core.api.util.xml.XmlRenderer;
import org.kuali.rice.core.framework.impex.xml.XmlExporter;
import org.kuali.rice.kew.api.WorkflowRuntimeException;
import org.kuali.rice.kew.export.KewExportDataSet;
import org.kuali.rice.kew.rule.bo.RuleAttribute;

import java.io.StringReader;
import java.util.Iterator;

import static org.kuali.rice.core.api.impex.xml.XmlConstants.*;

/**
 * Exports {@link org.kuali.rice.kew.rule.bo.RuleAttribute}s to XML.
 * 
 * @see org.kuali.rice.kew.rule.bo.RuleAttribute
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class RuleAttributeXmlExporter implements XmlExporter {

    protected final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(getClass());
    
    private XmlRenderer renderer = new XmlRenderer(RULE_ATTRIBUTE_NAMESPACE);
    
	@Override
	public boolean supportPrettyPrint() {
		return true;
	}

    @Override
    public Element export(ExportDataSet exportDataSet) {
    	KewExportDataSet dataSet = KewExportDataSet.fromExportDataSet(exportDataSet);
        if (!dataSet.getRuleAttributes().isEmpty()) {
            Element rootElement = renderer.renderElement(null, RULE_ATTRIBUTES);
            rootElement.setAttribute(SCHEMA_LOCATION_ATTR, RULE_ATTRIBUTE_SCHEMA_LOCATION, SCHEMA_NAMESPACE);
            for (Iterator iterator = dataSet.getRuleAttributes().iterator(); iterator.hasNext();) {
                RuleAttribute template = (RuleAttribute)iterator.next();
                exportRuleAttribute(rootElement, template);
            }
            return rootElement;
        }
        return null;
    }
    
    private void exportRuleAttribute(Element parent, RuleAttribute ruleAttribute) {
        Element attributeElement = renderer.renderElement(parent, RULE_ATTRIBUTE);
        renderer.renderTextElement(attributeElement, NAME, ruleAttribute.getName());
        renderer.renderTextElement(attributeElement, CLASS_NAME, ruleAttribute.getResourceDescriptor());
        renderer.renderTextElement(attributeElement, LABEL, ruleAttribute.getLabel());
        renderer.renderTextElement(attributeElement, DESCRIPTION, ruleAttribute.getDescription());
        renderer.renderTextElement(attributeElement, TYPE, ruleAttribute.getType());
        renderer.renderTextElement(attributeElement, APPLICATION_ID, ruleAttribute.getApplicationId());
        if (!org.apache.commons.lang.StringUtils.isEmpty(ruleAttribute.getXmlConfigData())) {
            try {
                Document configDoc = SafeXmlUtils.safeSAXBuilder().build(new StringReader(ruleAttribute.getXmlConfigData()));
                XmlHelper.propagateNamespace(configDoc.getRootElement(), RULE_ATTRIBUTE_NAMESPACE);
                attributeElement.addContent(configDoc.getRootElement().detach());
            } catch (Exception e) {
            	LOG.error("Error parsing attribute XML configuration.", e);
                throw new WorkflowRuntimeException("Error parsing attribute XML configuration.", e);
            }
        }
    }
    
}
