/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kcb.deliverer.impl;

import java.util.*;

import org.apache.commons.validator.EmailValidator;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.kcb.bo.MessageDelivery;
import org.kuali.rice.kcb.deliverer.MessageDeliverer;
import org.kuali.rice.kcb.exception.ErrorList;
import org.kuali.rice.kcb.api.exception.MessageDeliveryException;
import org.kuali.rice.kcb.service.EmailService;
import org.kuali.rice.kcb.service.GlobalKCBServiceLocator;
import org.kuali.rice.kcb.service.RecipientPreferenceService;

/**
 * This class is responsible for describing the email delivery mechanism for
 * the system.
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class EmailMessageDeliverer implements MessageDeliverer {
    private static Logger LOG = LogManager.getLogger(EmailMessageDeliverer.class);

    private EmailService emailService;
    private RecipientPreferenceService recipientPreferenceService;

    public static final String NAME = "Email";
    public static final String EMAIL_ADDR_PREF_KEY = "email_address";
    public static final String EMAIL_DELIV_FRMT_PREF_KEY = "email_delivery_format";

    /**
     * This implementation uses the email service to deliver a notification.
     * @see org.kuali.rice.kcb.deliverer.MessageDeliverer#deliver(MessageDelivery)
     */
    @Override
    public void deliver(MessageDelivery messageDelivery) throws MessageDeliveryException {
        try {
            // figure out the proper recipient email address
            String recipientEmailAddressPrefKey = getName()+"."+EMAIL_ADDR_PREF_KEY;
            String recipientEmailFormatPrefKey = getName()+"."+EMAIL_DELIV_FRMT_PREF_KEY;

            String recipientEmailAddress = getRecipientPreferenceService().getRecipientPreference(messageDelivery.getMessage().getRecipient(), recipientEmailAddressPrefKey).getValue();
            String recipientEmailFormat = getRecipientPreferenceService().getRecipientPreference(messageDelivery.getMessage().getRecipient(), recipientEmailFormatPrefKey).getValue();

            Long emailMessageId = getEmailService().sendEmail(messageDelivery, recipientEmailAddress, recipientEmailFormat);

            String deliverySystemId = null;
            if (emailMessageId != null) {
                deliverySystemId = emailMessageId.toString();
            }
            messageDelivery.setDelivererSystemId(deliverySystemId);
        } catch (Exception we) {
            LOG.error("Error delivering email notification", we);
            throw new MessageDeliveryException("Error delivering email notification", we);
        }
    }

    @Override
    public void dismiss(MessageDelivery messageDelivery, String user, String cause) {
        // we can't remove an email message once it has been sent
    }


    @Override
    public String getDescription() {
        return "Enter an Email Address and Email Delivery Format below and select the channels for which you would like email delivery " +
               "notifications. Select \"None\" in the channel list to remove a delivery type for all channels.  " +
               "Only one Email Address and Email Delivery Format may be specified. Any data entered and " +
               "saved will override prior Delivery Type selections.";
    }


    @Override
    public String getName() {
        return NAME;
    }


    @Override
    public String getTitle() {
        return "Email Message Delivery";
    }

    /**
     * This implementation returns an address field.
     */
    @Override
    public LinkedHashMap<String, String> getPreferenceKeys() {
        LinkedHashMap<String, String> prefKeys = new LinkedHashMap<>();
        prefKeys.put(EMAIL_ADDR_PREF_KEY, "Email Address (\"abc@def.edu\")");
        prefKeys.put(EMAIL_DELIV_FRMT_PREF_KEY, "Email Delivery Format (text or html)");
        return prefKeys;
    }

    @Override
    public void validatePreferenceValues(HashMap<String, String> prefs) throws ErrorList {
        boolean error = false;
        ErrorList errorList = new ErrorList();
        String[] validformats = {"text","html"};

        if (!prefs.containsKey(getName()+"."+EMAIL_ADDR_PREF_KEY)) {
            errorList.addError("Email Address is a required field.");
            error = true;
        } else {
            String addressValue = prefs.get(getName()+"."+EMAIL_ADDR_PREF_KEY);
            EmailValidator validator = EmailValidator.getInstance();
            if (!validator.isValid(addressValue)) {
                errorList.addError("Email Address is required and must be properly formatted - \"abc@def.edu\".");
                error = true;
            }
        }

        // validate format
        if (!prefs.containsKey(getName()+"."+EMAIL_DELIV_FRMT_PREF_KEY)) {
            errorList.addError("Email Delivery Format is required.");
            error = true;
        } else {
            String formatValue = prefs.get(getName()+"."+EMAIL_DELIV_FRMT_PREF_KEY);
            Set<String> formats = new HashSet<>();
            Collections.addAll(formats, validformats);

            if (!formats.contains(formatValue)) {
                errorList.addError("Email Delivery Format is required and must be entered as \"text\" or \"html\".");
                error = true;
            }
        }

        if (error) throw errorList;
    }

    private EmailService getEmailService() {
        if (emailService == null) {
            this.emailService = GlobalKCBServiceLocator.getInstance().getEmailService();
        }

        return emailService;
    }

    private RecipientPreferenceService getRecipientPreferenceService() {
        if (recipientPreferenceService == null) {
            this.recipientPreferenceService = GlobalKCBServiceLocator.getInstance().getRecipientPreferenceService();
        }

        return recipientPreferenceService;
    }
}
