/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kcb.service.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import java.util.Map.Entry;

import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.core.api.exception.RiceIllegalArgumentException;
import org.kuali.rice.kcb.bo.MessageDelivery;
import org.kuali.rice.kcb.deliverer.MessageDeliverer;
import org.kuali.rice.kcb.deliverer.impl.EmailMessageDeliverer;
import org.kuali.rice.kcb.deliverer.impl.MockMessageDeliverer;
import org.kuali.rice.kcb.service.MessageDelivererRegistryService;
import org.springframework.beans.factory.InitializingBean;

/**
 * MessageDelivererRegistryService implementation - for now we use a HashMap to do this registration, in the future we'll use resource loading.
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class MessageDelivererRegistryServiceImpl implements MessageDelivererRegistryService, InitializingBean {
    private static Logger LOG = LogManager.getLogger(MessageDelivererRegistryServiceImpl.class);

    // holds information about the registered deliverer types
    private HashMap<String, Class<? extends MessageDeliverer>> messageDelivererTypes;

    @Override
    public Collection<String> getAllDelivererTypes() {
        Collection<MessageDeliverer> deliverers = getAllDeliverers();
        Set<String> types = new HashSet<>(deliverers.size());
        for (MessageDeliverer deliverer: deliverers) {
            types.add(deliverer.getName().toLowerCase());
        }
        return types;
    }

    /**
     * Implements by constructing instances of each registered class and adding to an ArrayList that
     * gets passed back to the calling method.
     */
    @Override
    public Collection<MessageDeliverer> getAllDeliverers() {
        ArrayList<MessageDeliverer>  delivererTypes = new ArrayList<>();

        Set<Entry<String, Class<? extends MessageDeliverer>>> registeredTypes = messageDelivererTypes.entrySet();

        // iterate over each type and add an instance of each to the returning ArrayList
        for(Entry<String, Class<? extends MessageDeliverer>> entry: registeredTypes ) {
            try {
                delivererTypes.add(entry.getValue().newInstance());
            } catch (InstantiationException|IllegalAccessException e) {
                LOG.error(e.getMessage(), e);
            }
        }

        return delivererTypes;
    }

    /**
     * Implements by calling getDelivererByName for the delivery type name within the messageDelivery object.
     */
    @Override
    public MessageDeliverer getDeliverer(MessageDelivery messageDelivery) {
        if (messageDelivery == null) {
            throw new RiceIllegalArgumentException("messageDelivery is null");
        }

        MessageDeliverer nmd = getDelivererByName(messageDelivery.getDelivererTypeName());
        if (nmd == null) {
            LOG.error("The message deliverer type ('" + messageDelivery.getDelivererTypeName() + "') " +
                      "associated with message delivery id='" + messageDelivery.getId() + "' was not found in the message deliverer registry.  This deliverer " +
                      "plugin is not in the system.");
        }
        return nmd;
    }

    /**
     * Implements by doing a key lookup in the hashmap that acts as the deliverer plugin registry.  The deliverer name is the key in the hashmap for 
     * all registered deliverers.
     */
    @Override
    public MessageDeliverer getDelivererByName(String messageDelivererName) {
        if (StringUtils.isBlank(messageDelivererName)) {
            throw new RiceIllegalArgumentException("messageDelivererName is null or blank");
        }

        Class<? extends MessageDeliverer> clazz = messageDelivererTypes.get(messageDelivererName.toLowerCase());

        if(clazz == null) {
            LOG.error("The message deliverer type ('" + messageDelivererName + "') " +
                      " was not found in the message deliverer registry.  This deliverer " +
                      "plugin is not in the system.");
            return null;
        }

        MessageDeliverer messageDeliverer = null;
        try {
            messageDeliverer = clazz.newInstance();
        } catch (InstantiationException|IllegalAccessException e) {
            LOG.error(e.getMessage(), e);
        }

        return messageDeliverer;
    }

    @Override
    public void afterPropertiesSet() {
        EmailMessageDeliverer email = new EmailMessageDeliverer();
        MockMessageDeliverer mock = new MockMessageDeliverer();

        messageDelivererTypes = new HashMap<>(4);
        messageDelivererTypes.put(email.getName().toLowerCase(), email.getClass());
        messageDelivererTypes.put(mock.getName().toLowerCase(), mock.getClass());
    }
}
