/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.engine.node.var.schemes;

import java.io.StringReader;

import javax.xml.namespace.QName;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathVariableResolver;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.kew.engine.RouteContext;
import org.kuali.rice.kew.engine.node.BranchState;
import org.kuali.rice.kew.engine.node.service.BranchService;
import org.kuali.rice.kew.engine.node.var.Property;
import org.kuali.rice.kew.engine.node.var.PropertyScheme;
import org.kuali.rice.kew.rule.xmlrouting.XPathHelper;
import org.kuali.rice.kew.service.KEWServiceLocator;
import org.xml.sax.InputSource;


/**
 * A PropertyScheme that resolves the Property by evaluating it as an XPath expression.
 * DocumentRouteHeaderValue variables are set on the XPath instance so they are accessible.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class XPathScheme implements PropertyScheme {
    private static final Logger LOG = LogManager.getLogger(XPathScheme.class);

    @Override
    public String getName() {
        return "xpath";
    }

    @Override
    public String getShortName() {
        return "xpath";
    }

    @Override
    public Object load(Property property, final RouteContext context) {
        XPath xpath = XPathHelper.newXPath();
        final BranchService branchService = KEWServiceLocator.getBranchService();
        xpath.setXPathVariableResolver(new XPathVariableResolver() {
            @Override
            public Object resolveVariable(QName name) {
                LOG.debug("Resolving XPath variable: " + name);
                String value = branchService.getScopedVariableValue(context.getNodeInstance().getBranch(), BranchState.VARIABLE_PREFIX + name.getLocalPart());
                LOG.debug("Resolved XPath variable " + name + " to " + value);
                return value;
            }
        });
        try {
            String docContent = context.getDocument().getDocContent();
            LOG.debug("Executing xpath expression '" + property.locator + "' in doc '" + docContent + "'");
            return xpath.evaluate(property.locator, new InputSource(new StringReader(docContent)), XPathConstants.STRING);
        } catch (XPathExpressionException xpee) {
            throw new RuntimeException("Error evaluating xpath expression '" + property.locator + "'", xpee);
        }
    }
}
