/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.impl.document;

import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.core.api.exception.RiceIllegalArgumentException;
import org.kuali.rice.kew.api.WorkflowRuntimeException;
import org.kuali.rice.kew.api.document.DocumentProcessingOptions;
import org.kuali.rice.kew.api.document.DocumentProcessingQueue;
import org.kuali.rice.kew.api.document.attribute.DocumentAttributeIndexingQueue;
import org.kuali.rice.kew.engine.OrchestrationConfig;
import org.kuali.rice.kew.engine.WorkflowEngine;
import org.kuali.rice.kew.engine.WorkflowEngineFactory;
import org.kuali.rice.ksb.api.messaging.AsyncCapableService;
import org.kuali.rice.ksb.api.messaging.AsyncWrappable;

import javax.jws.WebParam;
import javax.xml.namespace.QName;
import java.util.Collections;

/**
 * Reference implementation of the {@code DocumentProcessingQueue}.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class DocumentProcessingQueueImpl implements DocumentProcessingQueue, AsyncWrappable<DocumentProcessingQueue> {

    private static final Logger LOG = LogManager.getLogger(DocumentProcessingQueueImpl.class);

    private WorkflowEngineFactory workflowEngineFactory;
    private DocumentAttributeIndexingQueue documentAttributeIndexingQueue;
    private AsyncCapableService asyncCapableService;

    @Override
    public void process(@WebParam(name = "documentId") String documentId) {
        processWithOptions(documentId, null);
    }

    @Override
    public void processWithOptions(@WebParam(name = "documentId") String documentId,
            @WebParam(name = "options") DocumentProcessingOptions options) {
        if (StringUtils.isBlank(documentId)) {
            throw new RiceIllegalArgumentException("documentId was a null or blank value");
        }
        if (options == null) {
            options = DocumentProcessingOptions.createDefault();
        }
        OrchestrationConfig config = new OrchestrationConfig(OrchestrationConfig.EngineCapability.STANDARD,
                Collections.<String>emptySet(), null, options.isSendNotifications(), options.isRunPostProcessor());
        WorkflowEngine engine = getWorkflowEngineFactory().newEngine(config);
        try {
			engine.process(documentId);
		} catch (Exception e) {
			LOG.error("Failed to process document through the workflow engine", e);
            if (e instanceof RuntimeException) {
                throw (RuntimeException)e;
            }
			throw new WorkflowRuntimeException(e);
		}
        if (options.isIndexSearchAttributes()) {
            getDocumentAttributeIndexingQueue().indexDocument(documentId);
        }
    }

    public WorkflowEngineFactory getWorkflowEngineFactory() {
        return workflowEngineFactory;
    }

    public void setWorkflowEngineFactory(WorkflowEngineFactory workflowEngineFactory) {
        this.workflowEngineFactory = workflowEngineFactory;
    }

    public DocumentAttributeIndexingQueue getDocumentAttributeIndexingQueue() {
        return documentAttributeIndexingQueue;
    }

    public void setDocumentAttributeIndexingQueue(DocumentAttributeIndexingQueue documentAttributeIndexingQueue) {
        this.documentAttributeIndexingQueue = documentAttributeIndexingQueue;
    }

    @Override
    public DocumentProcessingQueue wrap(QName qname, String applicationId) {
        return new DocumentProcessingQueueAsyncCapableImpl(asyncCapableService,this, qname, applicationId);
    }

    public AsyncCapableService getAsyncCapableService() {
        return asyncCapableService;
    }

    public void setAsyncCapableService(AsyncCapableService asyncCapableService) {
        this.asyncCapableService = asyncCapableService;
    }
}
