/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package org.kuali.rice.krad.util;

import java.io.Serializable;
import java.util.Objects;

public final class QueryPagingRequest implements Serializable {

    private final int startAtIndex;
    private final int maxResults;

    public QueryPagingRequest(int startAtIndex, int maxResults) {
        if (startAtIndex < 0) {
            throw new IllegalArgumentException("startAtIndex < 0");
        }

        if (maxResults < 0) {
            throw new IllegalArgumentException("maxResults < 0");
        }

        this.startAtIndex = startAtIndex;
        this.maxResults = maxResults;
    }

    /**
     * Returns the optional zero-based "start" index for rows returned.  When
     * this query is executed, this property should be read to determine the
     * first row which should be returned.  If the given index is beyond the
     * end of the result set, then the resulting query should effectively
     * return no rows (as opposed to producing an index-based out of bounds
     * error).
     *
     * <p>
     * Will never be less than 0
     *
     * @return the starting row index requested by this query, or null if
     * the results should start at the beginning of the result set
     */
    public int getStartAtIndex() {
        return startAtIndex;
    }

    /**
     * Returns the maximum number of results that this query is requesting
     * to receive.  If the number request is larger than the number of
     * results then all results are returned.
     *
     * <p>
     * Will never be less than 0
     *
     * @return the maximum number of results to return from the query
     */
    public int getMaxResults() {
        return maxResults;
    }


    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        QueryPagingRequest that = (QueryPagingRequest) o;
        return startAtIndex == that.startAtIndex &&
                maxResults == that.maxResults;
    }

    @Override
    public int hashCode() {
        return Objects.hash(startAtIndex, maxResults);
    }

    @Override
    public String toString() {
        return "QueryPagingRequest{" +
                "startAtIndex=" + startAtIndex +
                ", maxResults=" + maxResults +
                '}';
    }
}
