/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.data.jpa;

import org.kuali.rice.core.api.criteria.QueryByCriteria;

import javax.persistence.Query;

/**
 * Translates queries from generic API classes to platform-specific concrete classes.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
interface QueryTranslator<C, Q, QC> {

    /**
     * Translates the given {@link org.kuali.rice.core.api.criteria.QueryByCriteria} to a platform-specific criteria.
     *
     * @param queryClazz the type of the query.
     * @param criteria the {@link org.kuali.rice.core.api.criteria.QueryByCriteria} to translate.
     * @return a criteria for the given {@link org.kuali.rice.core.api.criteria.QueryByCriteria}.
     */
    C translateCriteria(Class<?> queryClazz, QueryByCriteria criteria);

    /**
     * Translates the given {@link org.kuali.rice.core.api.criteria.QueryByCriteria} to a platform-specific criteria
     * but instead of returning entities will return a count.  The query is also ignore paging instructions such that
     * the count reflects the total count and is not limited in any way.  Order by instructions are also ignored because
     * they are irrelevant to the count.
     *
     * @param queryClazz the type of the query.
     * @param criteria the {@link org.kuali.rice.core.api.criteria.QueryByCriteria} to translate.
     * @return a criteria for the given {@link org.kuali.rice.core.api.criteria.QueryByCriteria}.
     */
    C translateCriteriaForCount(Class<?> queryClazz, QueryByCriteria criteria);

    /**
     * Creates a query from the given criteria.
     *
     * @param queryClazz the type of the query.
     * @param criteria the criteria to translate.
     * @return a query from the given criteria.
     */
    Q createQuery(Class<?> queryClazz, C criteria);

    QC createQueryForCount(Class<?> queryClazz, C criteria);

    /**
     * Creates a query to delete records from the given criteria
     * @param queryClass the type of the query
     * @param criteria the criteria to translate
     * @return a query from the given criteria
     */
    Query createDeletionQuery(Class<?> queryClass, C criteria);

    /**
     * Translates the {@link QueryByCriteria} flags to the query.
     * @param qbc the {@link QueryByCriteria} to translate from.
     * @param query the query to translate to.
     */
    void convertQueryFlags(QueryByCriteria qbc, Q query);
}
