/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package org.kuali.common.util.runnable.impl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.text.NumberFormat;
import java.util.Date;
import java.util.List;

/**
 * Execute the list of <code>runnables</code> supplied to this bean
 */
public class RunnablesRunnable implements Runnable {

    private static final String NO_NULLS = "null not allowed";
    public static final double SECOND = 1000;
    public static final double MINUTE = 60 * SECOND;
    public static final double HOUR = 60 * MINUTE;
    public static final double DAY = 24 * HOUR;
    public static final double YEAR = 365 * DAY;
    private static final Logger logger = LogManager.getLogger(RunnablesRunnable.class);

    private final List<Runnable> runnables;
    private final boolean skip;
    private final boolean timed;

    public RunnablesRunnable(List<? extends Runnable> runnables) {
        this(runnables, false);
    }

    public RunnablesRunnable(List<? extends Runnable> runnables, boolean skip) {
        this(runnables, skip, false);
    }

    public RunnablesRunnable(List<? extends Runnable> runnables, boolean skip, boolean timed) {
        noNulls(runnables);
        this.runnables = List.copyOf(runnables);
        this.skip = skip;
        this.timed = timed;
    }

    @Override
    public void run() {
        if (skip) {
            logger.info("Skipping execution of {} runnables", runnables.size());
            return;
        }
        long start = System.currentTimeMillis();
        for (Runnable runnable : runnables) {
            runnable.run();
        }
        if (timed) {
            long stop = System.currentTimeMillis();
            logger.info("------------------------------------------------------------------------");
            logger.info("Total Time: {}", getTime(stop - start));
            logger.info("Finished at: {}", new Date(stop));
            logger.info("------------------------------------------------------------------------");
        }
    }

    public List<Runnable> getrunnables() {
        return runnables;
    }

    public boolean isSkip() {
        return skip;
    }

    public boolean isTimed() {
        return timed;
    }

    private static void noNulls(Object... objects) {
        noNullsWithMsg(NO_NULLS, objects);
    }

    private static void noNullsWithMsg(String msg, Object... objects) {
        for (Object object : objects) {
            if (object == null) {
                throw new AssertionError(msg);
            }
        }
    }

    /**
     * Given milliseconds, return milliseconds, seconds, minutes, hours, days, or years as appropriate. Note that years is approximate since the logic always assumes there are
     * exactly 365 days per year.
     */
    public static String getTime(long millis) {
        return getTime(millis, NumberFormat.getInstance());
    }

    /**
     * Given milliseconds, return milliseconds, seconds, minutes, hours, days, or years as appropriate. Note that years is approximate since the logic always assumes there are
     * exactly 365 days per year.
     */
    public static String getTime(long millis, NumberFormat formatter) {
        long abs = Math.abs(millis);
        if (abs < SECOND) {
            return millis + "ms";
        } else if (abs < MINUTE) {
            return formatter.format(millis / SECOND) + "s";
        } else if (abs < HOUR) {
            return formatter.format(millis / MINUTE) + "m";
        } else if (abs < DAY) {
            return formatter.format(millis / HOUR) + "h";
        } else if (abs < YEAR) {
            return formatter.format(millis / DAY) + "d";
        } else {
            return formatter.format(millis / YEAR) + "y";
        }
    }

}
