/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.keyvalues;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.commons.beanutils.BeanUtils;
import org.kuali.rice.core.api.util.ConcreteKeyValue;
import org.kuali.rice.core.api.util.KeyValue;
import org.kuali.rice.krad.service.KRADServiceLocatorWeb;

/**
 * This class is a Generic ValuesFinder that builds the list of KeyValuePairs it returns
 * in getKeyValues() based on a BO along with a keyAttributeName and labelAttributeName
 * that are specified.
 */
public class PersistableBusinessObjectValuesFinder extends KeyValuesBase {
    private static final long serialVersionUID = 1L;

    protected Class<?> businessObjectClass;
    protected String keyAttribute;
    protected String labelAttribute;
    protected boolean includeKeyInLabel = false;
    protected boolean includeBlankRow = false;

    /**
     * Build the list of KeyValues using the key (keyAttributeName) and
     * label (labelAttributeName) of the list of all business objects found
     * for the BO class specified.
     */
    @Override
	public List<KeyValue> getKeyValues() {
    	try {
            @SuppressWarnings("deprecation")
            Collection<?> objects = KRADServiceLocatorWeb.getLegacyDataAdapter().findMatching(businessObjectClass, Collections.emptyMap());
            List<KeyValue> labels = new ArrayList<>(objects.size());
            if(includeBlankRow) {
            	labels.add(new ConcreteKeyValue("", ""));
            }
            for (Object object : objects) {
            	String key = BeanUtils.getProperty(object, keyAttribute);
            	String label = BeanUtils.getProperty(object, labelAttribute);
            	if (includeKeyInLabel) {
            	    label = key + " - " + label;
            	}
            	labels.add(new ConcreteKeyValue(key, label));
    	    }
            return labels;
    	} catch (Exception e) {
            throw new RuntimeException("Exception occurred while trying to build keyValues List: " + this, e);
    	}
    }

    public void setBusinessObjectClass(Class<?> businessObjectClass) {
        this.businessObjectClass = businessObjectClass;
    }

    public void setIncludeKeyInLabel(boolean includeKeyInLabel) {
        this.includeKeyInLabel = includeKeyInLabel;
    }

    /**
     * @deprecated use {@link #setIncludeKeyInLabel(boolean)}
     */
    @Deprecated
    public void setIncludeKeyInDescription(boolean includeKeyInDescription) {
        setIncludeKeyInLabel(includeKeyInDescription);
    }

    public void setKeyAttribute(String keyAttribute) {
        this.keyAttribute = keyAttribute;
    }

    /**
     * @deprecated use {@link #setKeyAttribute(String)}
     */
    @Deprecated
    public void setKeyAttributeName(String keyAttributeName) {
        setKeyAttribute(keyAttributeName);
    }

    public void setLabelAttribute(String labelAttribute) {
        this.labelAttribute = labelAttribute;
    }

    /**
     * @deprecated use {@link #setLabelAttribute(String)}
     */
    @Deprecated
    public void setLabelAttributeName(String labelAttributeName) {
        setLabelAttribute(labelAttributeName);
    }

	public void setIncludeBlankRow(boolean includeBlankRow) {
		this.includeBlankRow = includeBlankRow;
	}

    @Override
    public String toString() {
        return "PersistableBusinessObjectValuesFinder{" +
                "businessObjectClass=" + businessObjectClass +
                ", keyAttribute='" + keyAttribute + '\'' +
                ", labelAttribute='" + labelAttribute + '\'' +
                ", includeKeyInLabel=" + includeKeyInLabel +
                ", includeBlankRow=" + includeBlankRow +
                '}';
    }
}
