/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.rules;

import org.kuali.rice.krad.document.Document;
import org.kuali.rice.krad.maintenance.MaintenanceDocument;
import org.kuali.rice.krad.rules.rule.event.AddCollectionLineEvent;
import org.kuali.rice.krad.rules.rule.event.ApproveDocumentEvent;

/**
 * Rule event interface for implementing business rules against a maintenance document.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public interface MaintenanceDocumentRule {

    /**
     * Runs all business rules needed prior to saving. This includes both common rules for all maintenance documents,
     * plus class-specific business rules.
     *
     * Will only return false if it fails the isValidForSave() test. Otherwise, it will always return positive
     * regardless of the outcome of the business rules. However, any error messages resulting from the business rules
     * will still be populated, for display to the consumer of this service.
     *
     * @see org.kuali.rice.krad.rules.rule.SaveDocumentRule#processSaveDocument(org.kuali.rice.krad.document.Document)
     */
    public abstract boolean processSaveDocument(Document document);

    /**
     * Runs all business rules needed prior to routing. This includes both common rules for all maintenance documents,
     * plus class-specific business rules.
     *
     * Will return false if any business rule fails, or if the document is in an invalid state, and not routable (see
     * isDocumentValidForRouting()).
     *
     * @see org.kuali.rice.krad.rules.rule.RouteDocumentRule#processRouteDocument(org.kuali.rice.krad.document.Document)
     */
    public abstract boolean processRouteDocument(Document document);

    /**
     * Runs all business rules needed prior to approving. This includes both common rules for all maintenance documents,
     * plus class-specific business rules.
     *
     * Will return false if any business rule fails, or if the document is in an invalid state, and not approvable (see
     * isDocumentValidForApproving()).
     *
     * @see org.kuali.rice.krad.rules.rule.ApproveDocumentRule#processApproveDocument(org.kuali.rice.krad.rules.rule.event.ApproveDocumentEvent)
     */
    public abstract boolean processApproveDocument(ApproveDocumentEvent approveEvent);

    /**
     * Runs all business rules needed prior to adding a collection to a line. This includes both common rules for all
     * maintenance documents, plus class-specific business rules.
     *
     * Will return false if any business rule fails.
     *
     * @see org.kuali.rice.krad.rules.rule.AddCollectionLineRule#processAddCollectionLine(org.kuali.rice.krad.rules.rule.event.AddCollectionLineEvent)
     */
    public abstract boolean processAddCollectionLine(AddCollectionLineEvent addEvent);

    /**
     * Sets the convenience objects like newAccount and oldAccount, so you have short and easy handles to the new and
     * old objects contained in the maintenance document.
     *
     * It also calls the BusinessObjectBase.refresh(), which will attempt to load all sub-objects from the DB by their
     * primary keys, if available.
     *
     * @param document - the maintenanceDocument being evaluated
     */
    public void setupBaseConvenienceObjects(MaintenanceDocument document);

    /**
     * Should always be overridden if a subclass is created.
     *
     * The goal for this is to cast the oldBo and newBo into the correct types of the subclass.
     */
    public void setupConvenienceObjects();
}
