/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.uif.container;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.kuali.rice.krad.datadictionary.Copyable;
import org.kuali.rice.krad.datadictionary.parse.BeanTag;
import org.kuali.rice.krad.datadictionary.parse.BeanTagAttribute;
import org.kuali.rice.krad.uif.UifConstants;
import org.kuali.rice.krad.uif.lifecycle.ViewLifecycle;
import org.kuali.rice.krad.uif.util.ObjectPropertyUtils;
import org.kuali.rice.krad.uif.view.ExpressionEvaluator;
import org.kuali.rice.krad.uif.view.View;

/**
 * Collection filter that evaluates a configured el expression against each line
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
@BeanTag(name = "conditionalFilter", parent = "Uif-ConditionalCollectionFilter")
public class ELCollectionFilter implements CollectionFilter, Copyable {
    private static final long serialVersionUID = 3273495753269940272L;

    private String expression = "";

    /**
     * Iterates through the collection and evaluates the el expression in context of the line. If the expression
     * evaluates to true, the line will remain, else be filtered out
     *
     * @see org.kuali.rice.krad.uif.container.CollectionFilter#filter(org.kuali.rice.krad.uif.view.View, Object,
     *      CollectionGroup)
     */
    @Override
    public List<Integer> filter(View view, Object model, CollectionGroup collectionGroup) {
        // get the collection for this group from the model
        List<Object> modelCollection = ObjectPropertyUtils.getPropertyValue(model,
                collectionGroup.getBindingInfo().getBindingPath());

        ExpressionEvaluator expressionEvaluator = ViewLifecycle.getExpressionEvaluator();

        // iterate through and add index that pass the expression
        List<Integer> showIndexes = new ArrayList<Integer>();

        int lineIndex = 0;
        for (Object line : modelCollection) {
            Map<String, Object> context = new HashMap<String, Object>(collectionGroup.getContext());
            context.put(UifConstants.ContextVariableNames.LINE, line);
            context.put(UifConstants.ContextVariableNames.INDEX, lineIndex);

            Boolean conditionPasses = (Boolean) expressionEvaluator.evaluateExpression(context, expression);
            if (conditionPasses) {
                showIndexes.add(lineIndex);
            }

            lineIndex++;
        }

        return showIndexes;
    }

    /**
     * Expression that will be evaluated for each line to determine whether the line should be filtered
     *
     * <p>
     * If expression passes, the line will remain in the collection, otherwise be filtered out. The expression given
     * should evaluate to a boolean
     * </p>
     *
     * @return valid el expression that evaluates to a boolean
     */
    @BeanTagAttribute
    public String getExpression() {
        return expression;
    }

    /**
     * Setter for the expression to use for filtering
     *
     * @param expression
     */
    public void setExpression(String expression) {
        this.expression = expression;
    }

    /**
     * @see Copyable#clone()
     */
    @Override
    public ELCollectionFilter clone() throws CloneNotSupportedException {
        return (ELCollectionFilter) super.clone();
    }

}
