/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.datadictionary.mask;

import org.kuali.rice.krad.datadictionary.parse.BeanTag;
import org.kuali.rice.krad.datadictionary.parse.BeanTagAttribute;

import java.io.Serializable;

/**
 * The displayMask element specifies the type of masking to
 * be used to hide the value from un-authorized users.
 * There are three types of masking.
 */
@BeanTag(name = "mask")
public class Mask implements Serializable {
    private static final long serialVersionUID = 4035984416568235531L;

    protected MaskFormatter maskFormatter;
    protected Class<? extends MaskFormatter> maskFormatterClass;

    /**
     * Masks a data value with the configured maskFormatter;
     *
     * @param value of the object
     * @return string value of the masked object
     */
    public String maskValue(Object value) {
        if (maskFormatter == null) {
            if (maskFormatterClass != null) {
                try {
                    maskFormatter = maskFormatterClass.newInstance();
                } catch (Exception e) {
                    throw new RuntimeException(
                            "Unable to create instance of mask formatter class: " + maskFormatterClass.getName(), e);
                }
            } else {
                throw new RuntimeException("Mask formatter not set for secure field.");
            }
        }

        return maskFormatter.maskValue(value);
    }

    /**
     * Gets the maskFormatter attribute.
     *
     * @return Returns the maskFormatter.
     */
    @BeanTagAttribute(name = "maskFormater", type = BeanTagAttribute.AttributeType.SINGLEBEAN)
    public MaskFormatter getMaskFormatter() {
        return maskFormatter;
    }

    /**
     * @param maskFormatter instance to be used for masking field values.
     */
    public void setMaskFormatter(MaskFormatter maskFormatter) {
        this.maskFormatter = maskFormatter;
    }

    /**
     * Gets the maskFormatterClass attribute.
     *
     * @return Returns the maskFormatterClass.
     */
    @BeanTagAttribute(name = "maskFormatterClass", type = BeanTagAttribute.AttributeType.SINGLEBEAN)
    public Class<? extends MaskFormatter> getMaskFormatterClass() {
        return maskFormatterClass;
    }

    /**
     * @param maskFormatterClass element is used when a custom masking
     * algorithm is desired.  This element specifies the name of a
     * class that will perform the masking for unauthorized users.
     */
    public void setMaskFormatterClass(Class<? extends MaskFormatter> maskFormatterClass) {
        this.maskFormatterClass = maskFormatterClass;
    }

}
