/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.uif.service.impl;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.data.DataType;
import org.kuali.rice.kim.api.KimConstants;
import org.kuali.rice.krad.data.DataObjectService;
import org.kuali.rice.krad.data.metadata.DataObjectAttribute;
import org.kuali.rice.krad.data.metadata.DataObjectMetadata;
import org.kuali.rice.krad.data.metadata.DataObjectRelationship;
import org.kuali.rice.krad.datadictionary.AttributeDefinition;
import org.kuali.rice.krad.datadictionary.validation.constraint.ValidCharactersConstraint;
import org.kuali.rice.krad.service.DataDictionaryService;
import org.kuali.rice.krad.uif.control.Control;
import org.kuali.rice.krad.uif.control.TextAreaControl;
import org.kuali.rice.krad.uif.control.TextControl;
import org.kuali.rice.krad.uif.control.UserControl;
import org.kuali.rice.krad.uif.service.UifDefaultingService;
import org.kuali.rice.krad.uif.util.ComponentFactory;

public class UifDefaultingServiceImpl implements UifDefaultingService {
    private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(UifDefaultingServiceImpl.class);

    protected DataDictionaryService dataDictionaryService;
    protected DataObjectService dataObjectService;

    protected static final String ANY_CHARACTER_PATTERN_CONSTRAINT = "UTF8AnyCharacterPatternConstraint";
    protected static final String DATE_PATTERN_CONSTRAINT = "BasicDatePatternConstraint";
    protected static final String FLOATING_POINT_PATTERN_CONSTRAINT = "FloatingPointPatternConstraintTemplate";
    protected static final String BIG_DECIMAL_PATTERN_CONSTRAINT = "BigDecimalPatternConstraintTemplate";
    protected static final String TIMESTAMP_PATTERN_CONSTRAINT = "TimestampPatternConstraint";
    protected static final String CURRENCY_PATTERN_CONSTRAINT = "CurrencyPatternConstraint";

    @Override
    public String deriveHumanFriendlyNameFromPropertyName(String camelCasedName) {
        // quick check to make sure there is a property name to modify
        if(StringUtils.isBlank(camelCasedName)) {
            return camelCasedName;
        }

        // We only want to include the component after the last property separator
        if (camelCasedName.contains(".")) {
            camelCasedName = StringUtils.substringAfterLast(camelCasedName, ".");
        }
        
        StringBuilder label = new StringBuilder(camelCasedName);
        
        // upper case the 1st letter
        label.replace(0, 1, label.substring(0, 1).toUpperCase());
        
        // loop through, inserting spaces when cap
        for (int i = 0; i < label.length(); i++) {
            if (Character.isUpperCase(label.charAt(i)) || Character.isDigit(label.charAt(i)) ) {
                label.insert(i, ' ');
                i++;
            }
        }

        return label.toString().trim();
    }

    protected Control getControlInstance( AttributeDefinition attrDef, DataObjectAttribute dataObjectAttribute ) {
        Control c = null;
        if ( attrDef.getOptionsFinder() != null ) {
            c = ComponentFactory.getSelectControl();
        } else if ( attrDef.getName().endsWith( ".principalName" ) && dataObjectAttribute != null ) {
            // FIXME: JHK: Yes, I know this is a *HORRIBLE* hack - but the alternative
            // would look even more "hacky" and error-prone
            c = ComponentFactory.getUserControl();
            // Need to find the relationship information
            // get the relationship ID by removing .principalName from the attribute name
            String relationshipName = StringUtils.removeEnd(attrDef.getName(), ".principalName");
            DataObjectMetadata metadata = dataObjectService.getMetadataRepository().getMetadata(
                    dataObjectAttribute.getOwningType());
            if ( metadata != null ) {
                DataObjectRelationship relationship = metadata.getRelationship(relationshipName);
                if ( relationship != null && CollectionUtils.isNotEmpty(relationship.getAttributeRelationships())) {
                    ((UserControl)c).setPrincipalIdPropertyName(relationship.getAttributeRelationships().get(0).getParentAttributeName());
                    ((UserControl)c).setPersonNamePropertyName(relationshipName + "." + KimConstants.AttributeConstants.NAME);
                    ((UserControl)c).setPersonObjectPropertyName(relationshipName);
                }
            } else {
                LOG.warn( "Attempt to pull relationship name: " + relationshipName + " resulted in missing metadata when looking for: " + dataObjectAttribute.getOwningType() );
            }
        } else {
            switch ( attrDef.getDataType() ) {
                case STRING :
                    // TODO: Determine better way to store the "200" metric below
                    if ( attrDef.getMaxLength() != null && attrDef.getMaxLength().intValue() > 200 ) {
                        c = ComponentFactory.getTextAreaControl();
                    } else {
                        c = ComponentFactory.getTextControl();
                    }
                    break;
                case BOOLEAN:
                    c = ComponentFactory.getCheckboxControl();
                    break;
                case DATE:
                case DATETIME:
                case TRUNCATED_DATE:
                    c = ComponentFactory.getDateControl();
                    break;
                case CURRENCY:
                case DOUBLE:
                case FLOAT:
                case INTEGER:
                case LARGE_INTEGER:
                case LONG:
                case PRECISE_DECIMAL:
                    c = ComponentFactory.getTextControl();
                    break;
                case MARKUP:
                    c = ComponentFactory.getTextAreaControl();
                    break;
                default:
                    c = ComponentFactory.getTextControl();
                    break;
            }
        }
        return c;
    }

    protected void customizeControlInstance( Control c, AttributeDefinition attrDef, DataObjectAttribute dataObjectAttribute ) {
        c.setRequired(attrDef.isRequired());
        if ( c instanceof TextControl) {
            if ( attrDef.getMaxLength() != null ) {
                ((TextControl) c).setMaxLength( attrDef.getMaxLength() );
                ((TextControl) c).setSize( attrDef.getMaxLength() );
                // If it's a larger field, add the expand icon by default
                if ( attrDef.getMaxLength() > 80 ) { // JHK : yes, this was a mostly arbitrary choice
                    ((TextControl) c).setTextExpand(true);
                }
            }
            if ( attrDef.getMinLength() != null ) {
                ((TextControl) c).setMinLength( attrDef.getMinLength() );
            }
        }
        if ( c instanceof TextAreaControl ) {
            if ( attrDef.getMaxLength() != null ) {
                ((TextAreaControl) c).setMaxLength( attrDef.getMaxLength() );
                ((TextAreaControl) c).setRows(attrDef.getMaxLength()/((TextAreaControl) c).getCols());
            }
            if ( attrDef.getMinLength() != null ) {
                ((TextAreaControl) c).setMinLength( attrDef.getMinLength() );
            }
        }
    }

    @Override
    public Control deriveControlAttributeFromMetadata(AttributeDefinition attrDef ) {
        DataObjectAttribute dataObjectAttribute = attrDef.getDataObjectAttribute();
        Control c = getControlInstance(attrDef, dataObjectAttribute);
        // If we a have a control...we should - but just in case - don't want to be too dependent on assumptions of the above code
        if (c != null) {
            customizeControlInstance(c, attrDef, dataObjectAttribute);
        }
        return c;
    }

    @Override
    public ValidCharactersConstraint deriveValidCharactersConstraint(AttributeDefinition attrDef) {
        ValidCharactersConstraint validCharactersConstraint = null;
        
        // First - see if one was defined in the metadata (provided by krad-data module annotations)
        if (attrDef.getDataObjectAttribute() != null) {
            if (StringUtils.isNotBlank(attrDef.getDataObjectAttribute().getValidCharactersConstraintBeanName())) {
                Object consObj = dataDictionaryService.getDictionaryBean(attrDef.getDataObjectAttribute()
                        .getValidCharactersConstraintBeanName());
                if (consObj != null && consObj instanceof ValidCharactersConstraint) {
                    validCharactersConstraint = (ValidCharactersConstraint) consObj;
                }
            }
        }
        
        // if not, make an intelligent guess from the data type
        if (validCharactersConstraint == null) {
            if (attrDef.getDataType() != null) {
                if (attrDef.getDataType() == DataType.CURRENCY) {
                    validCharactersConstraint = (ValidCharactersConstraint) dataDictionaryService
                            .getDictionaryBean(CURRENCY_PATTERN_CONSTRAINT);
                }else if (attrDef.getDataType() == DataType.PRECISE_DECIMAL ) {
                    validCharactersConstraint = (ValidCharactersConstraint) dataDictionaryService
                            .getDictionaryBean(BIG_DECIMAL_PATTERN_CONSTRAINT);
                } else if (attrDef.getDataType().isNumeric()) {
                    validCharactersConstraint = (ValidCharactersConstraint) dataDictionaryService
                            .getDictionaryBean(FLOATING_POINT_PATTERN_CONSTRAINT);
                } else if (attrDef.getDataType().isTemporal()) {
                    if (attrDef.getDataType() == DataType.DATE) {
                        validCharactersConstraint = (ValidCharactersConstraint) dataDictionaryService
                                .getDictionaryBean(DATE_PATTERN_CONSTRAINT);
                    } else if (attrDef.getDataType() == DataType.TIMESTAMP) {
                        validCharactersConstraint = (ValidCharactersConstraint) dataDictionaryService
                                .getDictionaryBean(TIMESTAMP_PATTERN_CONSTRAINT);
                    }
                }
            }
        }
        
        // default to UTF8
        if (validCharactersConstraint == null) {
            validCharactersConstraint = (ValidCharactersConstraint) dataDictionaryService
                    .getDictionaryBean(ANY_CHARACTER_PATTERN_CONSTRAINT);
        }

        return validCharactersConstraint;
    }

    public void setDataDictionaryService(DataDictionaryService dataDictionaryService) {
        this.dataDictionaryService = dataDictionaryService;
    }

    public void setDataObjectService(DataObjectService dataObjectService) {
        this.dataObjectService = dataObjectService;
    }
}
