/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.web.bind;

import org.kuali.rice.core.api.util.RiceKeyConstants;
import org.kuali.rice.core.web.format.FormatException;

import java.io.Serializable;
import java.sql.Date;
import java.text.ParseException;
import java.util.Calendar;

/**
 * PropertyEditor converts between date display strings and <code>java.util.Calendar</code> objects using the
 * <code>org.kuali.rice.core.api.datetime.DateTimeService</code>.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class UifCalendarEditor extends UifDateEditor implements Serializable {
    private static final long serialVersionUID = 8123569337264797008L;

    /**
     * This overridden method uses the
     * <code>org.kuali.rice.core.api.datetime.DateTimeService</code> to convert
     * the calendar object to the display string.
     */
    @Override
    public String getAsText() {
        if (this.getValue() == null) {
            return null;
        }

        if ("".equals(this.getValue())) {
            return null;
        }

        return getDateTimeService().toDateString(new Date(((java.util.Calendar) this.getValue()).getTimeInMillis()));
    }

    /**
     * Convert display text to <code>java.util.Calendar</code> object using the
     * <code>org.kuali.rice.core.api.datetime.DateTimeService</code>.
     *
     * @param text the display text
     * @return the <code>java.util.Calendar</code> object
     * @throws IllegalArgumentException the illegal argument exception
     */
    @Override
    protected Object convertToObject(String text) throws IllegalArgumentException {
        try {
            // Allow user to clear dates
            if (text == null || text.equals("")) {
                return null;
            }

            Date result = getDateTimeService().convertToSqlDate(text);
            Calendar calendar = getDateTimeService().getCalendar(result);
            calendar.setTime(result);

            if (calendar.get(Calendar.YEAR) < 1000 && verbatimYear(text).length() < 4) {
                throw new FormatException("illegal year format", RiceKeyConstants.ERROR_DATE, text);
            }

            return calendar;
        } catch (ParseException e) {
            throw new FormatException("parsing", RiceKeyConstants.ERROR_DATE, text, e);
        }
    }

}
