/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.util.documentserializer;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * A node in the trie.
 *
 */
public class PropertySerializerTrieNode implements PropertySerializabilityMetadata {
    private String pathString;
    private String propertyNameComponent;
    private PropertySerializability propertySerializability;

    private List<PropertySerializerTrieNode> childNodes;

    public PropertySerializerTrieNode(String pathString, String propertyNameComponent) {
        this.pathString = pathString;
        this.propertyNameComponent = propertyNameComponent;
        this.childNodes = null;
        this.propertySerializability = PropertySerializability.SERIALIZE_OBJECT;
    }

    public void addChildNode(PropertySerializerTrieNode child) {
        if (childNodes == null) {
            childNodes = new ArrayList<PropertySerializerTrieNode>();
        }
        childNodes.add(child);
    }

    /**
     * The name of this property, relative to the parent node (i.e. the child node name relative to its parents).
     *
     * @return
     */
    public String getPropertyNameComponent() {
        return propertyNameComponent;
    }

    /**
     * Retrieves the child node with the given name
     *
     * @param propertyNameComponent
     * @return
     */
    public PropertySerializerTrieNode getChildNode(String propertyNameComponent) {
        if (childNodes == null) {
            return null;
        }
        for (int i = 0; i < childNodes.size(); i++) {
            PropertySerializerTrieNode childNode = childNodes.get(i);
            if (childNode.getPropertyNameComponent().equals(propertyNameComponent)) {
                return childNode;
            }
        }
        return null;
    }

    /**
     * @see org.kuali.rice.krad.util.documentserializer.PropertySerializabilityMetadata#getSerializableChildProperty(java.lang.String)
     */
    @Override
    public PropertySerializabilityMetadata getSerializableChildProperty(String propertyNameComponent) {
        return getChildNode(propertyNameComponent);
    }

    /**
     * @see org.kuali.rice.krad.util.documentserializer.PropertySerializabilityMetadata#getPathString()
     */
    @Override
    public String getPathString() {
        return pathString;
    }

    /**
     * @see org.kuali.rice.krad.util.documentserializer.PropertySerializabilityMetadata#getPropertySerializability()
     */
    @Override
    public PropertySerializability getPropertySerializability() {
        return propertySerializability;
    }

    /**
     * Marks that all primitives of this object will be serialized.
     */
    public void setPropertySerializabilityToObjectAndAllPrimitives() {
        this.propertySerializability = PropertySerializability.SERIALIZE_OBJECT_AND_ALL_PRIMITIVES;
    }

    @Override
    public String toString() {
        StringBuilder buf = new StringBuilder();
        buf.append("Path String: ").append(pathString).append(" Name component: ").append(propertyNameComponent);
        if (childNodes == null || childNodes.isEmpty()) {
            buf.append(" No child nodes.");
        }
        else {
            buf.append(" Child nodes: ");
            for (Iterator<PropertySerializerTrieNode> i = childNodes.iterator(); i.hasNext();) {
                buf.append(i.next().getPropertyNameComponent());
                if (i.hasNext()) {
                    buf.append(", ");
                }
            }
        }
        return super.toString();
    }
}
