/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krms.impl.provider.repository;

import org.kuali.rice.krms.api.engine.ExecutionEnvironment;
import org.kuali.rice.krms.api.repository.action.ActionDefinition;
import org.kuali.rice.krms.framework.engine.Action;
import org.kuali.rice.krms.framework.type.ActionTypeService;
import org.kuali.rice.krms.impl.type.KrmsTypeResolver;

/**
 * An {@link Action} that doesn't load its members until execution.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 *
 */
final class LazyAction implements Action {

	private final ActionDefinition actionDefinition;
	private final KrmsTypeResolver typeResolver;

	private final Object mutex = new Object();
	
	// volatile for double-checked locking idiom
	private volatile Action action;

    /**
     *
     * @param actionDefinition ActionDefinition
     * @param typeResolver KrmsTypeResolver
     */
	LazyAction(ActionDefinition actionDefinition, KrmsTypeResolver typeResolver) {
		this.actionDefinition = actionDefinition;
		this.typeResolver = typeResolver;
		this.action = null;
	}
	
	@Override
	public void execute(ExecutionEnvironment environment) {
		getAction().execute(environment);
	}

	@Override
	public void executeSimulation(ExecutionEnvironment environment) {
		getAction().executeSimulation(environment);
	}
	
	/**
	 * Gets the action using a lazy double-checked locking mechanism as documented in Effective Java Item 71.
     * @return Action
	 */
	private Action getAction() {
		Action localAction = action;
		if (localAction == null) {
			synchronized (mutex) {
				localAction = action;
				if (localAction == null) {
					action = localAction = constructAction();
				}
			}
		}
		return localAction;
	}

    /**
     * builder method
     * @return Action
     */
	private Action constructAction() {
		ActionTypeService actionTypeService = typeResolver.getActionTypeService(actionDefinition);
		Action action = actionTypeService.loadAction(actionDefinition);
		if (action == null) {
			action = new Action() {
				@Override
				public void execute(ExecutionEnvironment environment) {
				}
				@Override
                public void executeSimulation(ExecutionEnvironment environment) {
				}
			};
		}
		return action;
	}

	
}
