/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.messaging;

import org.kuali.rice.core.api.config.CoreConfigHelper;
import org.kuali.rice.core.api.util.RiceUtilities;
import org.kuali.rice.core.api.util.io.SerializationUtils;
import org.kuali.rice.krad.data.jpa.PortableSequenceGenerator;
import org.kuali.rice.ksb.api.bus.ServiceConfiguration;
import org.kuali.rice.ksb.api.messaging.AsynchronousCall;
import org.kuali.rice.ksb.util.KSBConstants;

import javax.persistence.*;
import java.sql.Timestamp;

@Entity
@Table(name="KRSB_MSG_QUE_T")
@NamedQueries({
  @NamedQuery(name="PersistedMessageBO.FindAll", query="select pm from PersistedMessageBO pm"),
  @NamedQuery(name="PersistedMessageBO.GetNextDocuments", query="select pm from PersistedMessageBO pm where pm.applicationId = :applicationId and pm.queueStatus <> :queueStatus and pm.ipNumber = :ipNumber order by pm.queuePriority asc, pm.routeQueueId asc, pm.queueDate asc")
})
public class PersistedMessageBO implements PersistedMessage {

	private static final long serialVersionUID = -7047766894738304195L;

	@Id
	@GeneratedValue(generator = "KRSB_MSG_QUE_S")
    @PortableSequenceGenerator(name = "KRSB_MSG_QUE_S")
	@Column(name="MSG_QUE_ID")
	private Long routeQueueId;

	@Column(name="PRIO")
	private Integer queuePriority;

	@Column(name="STAT_CD")
	private String queueStatus;

	@Column(name="DT")
	private Timestamp queueDate;

	@Column(name="EXP_DT")
	private Timestamp expirationDate;

	@Column(name="RTRY_CNT")
	private Integer retryCount;

	@Version
	@Column(name="VER_NBR")
	private Integer lockVerNbr;

    @Column(name="IP_NBR")
	private String ipNumber;

    @Column(name="SVC_NM")
	private String serviceName;

    @Column(name="APPL_ID")
	private String applicationId;

    @Column(name="SVC_MTHD_NM")
	private String methodName;

    @Column(name="APP_VAL_ONE")
    private String value1;

    @Column(name="APP_VAL_TWO")
    private String value2;

    @OneToOne(mappedBy = "persistedMessage", orphanRemoval = true, cascade = CascadeType.ALL, optional = false )
    private PersistedMessagePayload persistedMessagePayload;

    @Transient
    private transient AsynchronousCall methodCall;

    public static PersistedMessageBO buildMessage(ServiceConfiguration serviceConfiguration, AsynchronousCall methodCall) {
        final PersistedMessageBO message = new PersistedMessageBO();
        message.setIpNumber(RiceUtilities.getIpNumber());
        message.setServiceName(serviceConfiguration.getServiceName().toString());
        message.setQueueDate(new Timestamp(System.currentTimeMillis()));
        if (serviceConfiguration.getPriority() == null) {
            message.setQueuePriority(KSBConstants.ROUTE_QUEUE_DEFAULT_PRIORITY);
        } else {
            message.setQueuePriority(serviceConfiguration.getPriority());
        }
        message.setQueueStatus(KSBConstants.ROUTE_QUEUE_QUEUED);
        message.setRetryCount(0);
        if (serviceConfiguration.getMillisToLive() > 0) {
            message.setExpirationDate(new Timestamp(System.currentTimeMillis() + serviceConfiguration.getMillisToLive()));
        }
        message.setApplicationId(CoreConfigHelper.getApplicationId());
        message.setMethodName(methodCall.getMethodName());

        final PersistedMessagePayload payload = new PersistedMessagePayload();
        payload.setPayload(SerializationUtils.serializeToBase64(methodCall));
        payload.setPersistedMessage(message);
        message.setPersistedMessagePayload(payload);

        return message;
    }

    public PersistedMessageBO copy() {
        final PersistedMessageBO message = new PersistedMessageBO();
        message.setRouteQueueId(getRouteQueueId());
        message.setQueuePriority(getQueuePriority());
        message.setQueueStatus(getQueueStatus());
        message.setQueueDate(getQueueDate());
        message.setExpirationDate(getExpirationDate());
        message.setRetryCount(getRetryCount());
        message.setLockVerNbr(getLockVerNbr());
        message.setIpNumber(getIpNumber());
        message.setServiceName(getServiceName());
        message.setApplicationId(getApplicationId());
        message.setMethodName(getMethodName());
        message.setValue1(getValue1());
        message.setValue2(getValue2());
        message.setMethodCall(getMethodCall());
        final PersistedMessagePayload payload = new PersistedMessagePayload();
        payload.setPayload(getPersistedMessagePayload().getPayload());
        payload.setPersistedMessage(message);
        message.setPersistedMessagePayload(payload);

        return message;
    }

	@Override
    public String getApplicationId() {
		return this.applicationId;
	}

	public void setApplicationId(String applicationId) {
		this.applicationId = applicationId;
	}

    @Override
    public String getIpNumber() {
        return this.ipNumber;
    }
    
    public void setIpNumber(String ipNumber) {
        this.ipNumber = ipNumber;
    }
    
	@Override
    public Timestamp getQueueDate() {
		return this.queueDate;
	}

    public void setQueueDate(Timestamp timestamp) {
        this.queueDate = timestamp;
    }

	@Override
    public Integer getQueuePriority() {
		return this.queuePriority;
	}

    public void setQueuePriority(Integer integer) {
        this.queuePriority = integer;
    }

    @Override
    public String getQueueStatus() {
		return this.queueStatus;
	}

    public void setQueueStatus(String string) {
        this.queueStatus = string;
    }

    @Override
    public Integer getRetryCount() {
		return this.retryCount;
	}

	public void setRetryCount(Integer integer) {
	    this.retryCount = integer;
	}

    public Integer getLockVerNbr() {
        return this.lockVerNbr;
    }
    
    public void setLockVerNbr(Integer lockVerNbr) {
        this.lockVerNbr = lockVerNbr;
    }
    
    @Override
    public Long getRouteQueueId() {
        return this.routeQueueId;
    }
    
    public void setRouteQueueId(Long queueSequence) {
        this.routeQueueId = queueSequence;
    }
    
	@Override
    public String getServiceName() {
		return this.serviceName;
	}

	public void setServiceName(String serviceName) {
		this.serviceName = serviceName;
	}

	@Override
    public String getMethodName() {
		return this.methodName;
	}

	public void setMethodName(String methodName) {
		this.methodName = methodName;
	}

	@Override
    public Timestamp getExpirationDate() {
		return this.expirationDate;
	}

	public void setExpirationDate(Timestamp expirationDate) {
		this.expirationDate = expirationDate;
	}

    @Override
    public String getValue1() {
        return this.value1;
    }

    public void setValue1(String value1) {
        this.value1 = value1;
    }

    @Override
    public String getValue2() {
        return this.value2;
    }

    public void setValue2(String value2) {
        this.value2 = value2;
    }

    @Override
    public String getPayload() {
        return (getPersistedMessagePayload() != null) ? getPersistedMessagePayload().getPayload() : null;
    }

    public void setPayload(String payload) {
        if (getPersistedMessagePayload() == null) {
            PersistedMessagePayload pmp = new PersistedMessagePayload();
            pmp.setPersistedMessage(this);
            setPersistedMessagePayload(pmp);
        }

        getPersistedMessagePayload().setPayload(payload);
    }

    @Override
    public AsynchronousCall getMethodCall() {
        if (this.methodCall == null) {
            this.methodCall = (AsynchronousCall) SerializationUtils.deserializeFromBase64(getPayload());
        }

        return this.methodCall;
    }

    public void setMethodCall(AsynchronousCall methodCall) {
        this.methodCall = methodCall;
    }

    public PersistedMessagePayload getPersistedMessagePayload() {
        return persistedMessagePayload;
    }

    public void setPersistedMessagePayload(PersistedMessagePayload persistedMessagePayload) {
        this.persistedMessagePayload = persistedMessagePayload;
    }

    @Override
    public String toString() {
        return "PersistedMessageBO{" +
                "routeQueueId=" + routeQueueId +
                ", queuePriority=" + queuePriority +
                ", queueStatus='" + queueStatus + '\'' +
                ", queueDate=" + queueDate +
                ", expirationDate=" + expirationDate +
                ", retryCount=" + retryCount +
                ", lockVerNbr=" + lockVerNbr +
                ", ipNumber='" + ipNumber + '\'' +
                ", serviceName='" + serviceName + '\'' +
                ", applicationId='" + applicationId + '\'' +
                ", methodName='" + methodName + '\'' +
                ", value1='" + value1 + '\'' +
                ", value2='" + value2 + '\'' +
                '}';
    }
}
