/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.api.spring.config.parsers;

import org.mule.api.lifecycle.Disposable;
import org.mule.api.lifecycle.Initialisable;
import org.mule.munit.api.spring.config.parsers.model.ParseableElement;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.beans.factory.parsing.BeanDefinitionParsingException;
import org.springframework.beans.factory.parsing.Location;
import org.springframework.beans.factory.parsing.Problem;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

import java.util.HashMap;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang.StringUtils.isNotBlank;


public class MunitGenericDefinitionParser extends AbstractDefinitionParser {
    private Class messageProcessorClass;

    private String scope;
    private Map<String, String> attributePropertyMap;
    private Map<String, ParseableElement> childElementMap;
    private Boolean withAttachProcessorDefinition = true;

    // Allows to modify the xml element with new attributes added at runtime like the line number
    private Map<String, String> elementAttributes;

    private MunitGenericDefinitionParser(Class messageProcessorClass) {
        this.messageProcessorClass = messageProcessorClass;

        this.attributePropertyMap = new HashMap<>();
        this.childElementMap = new HashMap<>();
        this.elementAttributes = new HashMap<>();
    }

    public static class MunitGenericDefinitionParserBuilder {
        protected Class messageProcessorClass;

        protected String scope;
        protected Map<String, String> attributePropertyMap;
        protected Map<String, ParseableElement> childElementMap;
        protected Boolean withAttachProcessorDefinition = true;

        protected Map<String, String> elementAttributes;

        public MunitGenericDefinitionParserBuilder(Class messageProcessorClass) {
            checkNotNull(messageProcessorClass, "The message processor class must not be null.");
            this.messageProcessorClass = messageProcessorClass;
            this.attributePropertyMap = new HashMap<>();
            this.childElementMap = new HashMap<>();

            this.elementAttributes = new HashMap<>();
        }

        public MunitGenericDefinitionParserBuilder withScope(String scope) {
            checkArgument(isNotBlank(scope), "The scope must not be null nor empty.");
            if (!scope.equals(ConfigurableBeanFactory.SCOPE_PROTOTYPE) && !scope.equals(ConfigurableBeanFactory.SCOPE_SINGLETON)) {
                throw new IllegalArgumentException("The provided scope is not valid.");
            }

            this.scope = scope;
            return this;
        }

        public MunitGenericDefinitionParserBuilder withAttribute(String attributeName, String propertyName) {
            checkArgument(isNotBlank(attributeName), "The attribute name must not be null nor empty.");
            checkArgument(isNotBlank(propertyName), "The property name must not be null nor empty.");
            this.attributePropertyMap.put(attributeName, propertyName);
            return this;
        }

        public MunitGenericDefinitionParserBuilder withChildElement(String childElementName, ParseableElement childElement) {
            checkArgument(isNotBlank(childElementName), "The child element name must not be null nor empty.");
            checkNotNull(childElement, "The child Element must not be null.");
            this.childElementMap.put(childElementName, childElement);
            return this;
        }

        public MunitGenericDefinitionParserBuilder withNoAttachProcessorDefinition() {
            this.withAttachProcessorDefinition = false;
            return this;
        }

        public MunitGenericDefinitionParserBuilder withElementAttribute(String name, String value) {
            checkArgument(isNotBlank(name), "The name must not be null nor empty.");
            checkArgument(isNotBlank(value), "The value must not be null nor empty.");
            this.elementAttributes.put(name, value);
            return this;
        }

        public MunitGenericDefinitionParser build() {
            checkNotNull(messageProcessorClass, "Build Fail. The message processor class can not be null.");
            checkArgument(isNotBlank(scope), "Build Fail.The scope can not be null nor empty.");

            MunitGenericDefinitionParser parser = new MunitGenericDefinitionParser(messageProcessorClass);
            parser.scope = this.scope;
            parser.attributePropertyMap.putAll(this.attributePropertyMap);

            parser.withAttachProcessorDefinition = this.withAttachProcessorDefinition;

            parser.childElementMap.putAll(this.childElementMap);

            parser.elementAttributes.putAll(this.elementAttributes);

            return parser;
        }
    }

    public BeanDefinition parse(Element element, ParserContext parserContext) {
        addAttributesToElement(element);

        BeanDefinitionBuilder builder = getBeanDefinitionBuilder(parserContext);

        builder.setScope(scope);

        defineLifeCyleMethodMapping(builder);

        parseMessageProcessorAttributes(element, builder);

        parseChildElements(element, builder, parserContext);

        BeanDefinition definition = builder.getBeanDefinition();
        setNoRecurseOnDefinition(definition);

        if (withAttachProcessorDefinition) {
            attachProcessorDefinition(parserContext, definition);
        }

        return definition;
    }

    protected void addAttributesToElement(Element element) {
        for (String name : elementAttributes.keySet()) {
            String value = elementAttributes.get(name);

            element.setAttribute(name, value);
        }
    }

    protected BeanDefinitionBuilder getBeanDefinitionBuilder(ParserContext parserContext) {
        try {
            return BeanDefinitionBuilder.rootBeanDefinition(messageProcessorClass.getName());
        } catch (NoClassDefFoundError noClassDefFoundError) {
            throw new BeanDefinitionParsingException(new Problem(("Cannot launch the mule app, the message processor class " + messageProcessorClass.getName() + " was not found in the classpath."), new Location(parserContext.getReaderContext().getResource()), null, noClassDefFoundError));
        }
    }

    protected void defineLifeCyleMethodMapping(BeanDefinitionBuilder builder) {
        if (FactoryBean.class.isAssignableFrom(messageProcessorClass)) {
            if (Initialisable.class.isAssignableFrom(messageProcessorClass)) {
                builder.setInitMethodName(Initialisable.PHASE_NAME);
            }
            if (Disposable.class.isAssignableFrom(messageProcessorClass)) {
                builder.setDestroyMethodName(Disposable.PHASE_NAME);
            }
        }

    }

    protected void parseMessageProcessorAttributes(Element element, BeanDefinitionBuilder builder) {
        parseElementAttributes(this.attributePropertyMap, element, builder);
    }

    protected void parseChildElements(Element element, BeanDefinitionBuilder builder, ParserContext parserContext) {
        for (String elementName : this.childElementMap.keySet()) {
            ParseableElement parseableElement = this.childElementMap.get(elementName);
            parseChildElementByType(parseableElement, element, builder, parserContext);
        }
    }

    public Class getMessageProcessorClass() {
        return messageProcessorClass;
    }

    public Map<String, String> getAttributePropertyMap() {
        Map<String, String> toReturn = new HashMap<>();
        toReturn.putAll(attributePropertyMap);
        return toReturn;
    }

    public Map<String, ParseableElement> getChildElementMap() {
        Map<String, ParseableElement> toReturn = new HashMap<>();
        toReturn.putAll(childElementMap);
        return toReturn;
    }

    public Boolean getWithAttachProcessorDefinition() {
        return withAttachProcessorDefinition;
    }

    public Map<String, String> getElementAttributes() {
        Map<String, String> toReturn = new HashMap<>();
        toReturn.putAll(elementAttributes);
        return toReturn;
    }

    public String getScope() {
        return scope;
    }
}
