/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.api.spring.config.parsers.model;


import java.util.HashMap;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang.StringUtils.isNotBlank;


public class ListChildElement implements ParseableElement {

    private String elementName;
    private String elementPropertyName;

    private Class itemClass;
    private String itemName;
    private Map<String, String> itemAttributePropertyMap;

    public ListChildElement() {
        this.itemAttributePropertyMap = new HashMap<>();
    }

    public static class ListChildElementBuilder {
        protected String elementName;
        protected String elementPropertyName;

        protected Class itemClass;
        protected String itemName;
        protected Map<String, String> itemAttributePropertyMap;

        public ListChildElementBuilder(String elementName, String elementPropertyName) {
            checkArgument(isNotBlank(elementName), "The element name must not be null nor empty.");
            checkArgument(isNotBlank(elementPropertyName), "The element  property name must not be null nor empty.");

            this.elementName = elementName;
            this.elementPropertyName = elementPropertyName;
            this.itemAttributePropertyMap = new HashMap<>();
        }

        public ListChildElementBuilder withItemClass(Class itemClass) {
            checkNotNull(itemClass, "The item class class must not be null.");
            this.itemClass = itemClass;
            return this;
        }

        public ListChildElementBuilder withItemName(String itemName) {
            checkArgument(isNotBlank(itemName), "The item name must not be null nor empty.");
            this.itemName = itemName;
            return this;
        }

        public ListChildElementBuilder withItemAttribute(String attributeName, String propertyName) {
            checkArgument(isNotBlank(attributeName), "The attribute name must not be null nor empty.");
            checkArgument(isNotBlank(propertyName), "The property name must not be null nor empty.");
            this.itemAttributePropertyMap.put(attributeName, propertyName);
            return this;
        }

        public ListChildElement build() {
            checkNotNull(itemClass, "Build Fail. The item class must not be null.");
            checkArgument(isNotBlank(itemName), "Build Fail. The item name must not be null.");

            ListChildElement listChildElement = new ListChildElement();
            listChildElement.elementName = this.elementName;
            listChildElement.elementPropertyName = this.elementPropertyName;
            listChildElement.itemClass = this.itemClass;
            listChildElement.itemName = this.itemName;
            listChildElement.itemAttributePropertyMap.putAll(this.itemAttributePropertyMap);

            return listChildElement;
        }
    }


    public String getElementName() {
        return elementName;
    }

    public String getElementPropertyName() {
        return elementPropertyName;
    }

    public Class getItemClass() {
        return itemClass;
    }

    public String getItemName() {
        return itemName;
    }

    public Map<String, String> getItemAttributePropertyMap() {
        return itemAttributePropertyMap;
    }
}


