/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner;

import org.mule.api.MuleContext;
import org.mule.api.config.ConfigurationBuilder;
import org.mule.api.config.MuleProperties;
import org.mule.api.context.MuleContextBuilder;
import org.mule.api.context.MuleContextFactory;
import org.mule.api.context.notification.MessageProcessorNotificationListener;
import org.mule.config.AnnotationsConfigurationBuilder;
import org.mule.config.DefaultMuleConfiguration;
import org.mule.config.builders.ExtensionsManagerConfigurationBuilder;
import org.mule.config.builders.SimpleConfigurationBuilder;
import org.mule.context.DefaultMuleContextBuilder;
import org.mule.context.DefaultMuleContextFactory;
import org.mule.context.notification.MessageProcessorNotification;
import org.mule.munit.common.log.TestsLogConfigurationHelper;
import org.mule.munit.common.util.TestingWorkListener;
import org.mule.munit.runner.actions.ConfigureLoggingForTestAction;
import org.mule.munit.runner.actions.MuleContextFactoryAction;
import org.mule.util.ClassUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import static com.google.common.base.Preconditions.checkNotNull;


/**
 * The goal of this class is to provide an instance of a MuleContext class.
 */
public class MunitMuleContextFactory {
    public static final String CLASSNAME_ANNOTATIONS_CONFIG_BUILDER = AnnotationsConfigurationBuilder.class.getCanonicalName();

    protected Properties startUpProperties;
    protected List<ConfigurationBuilder> builders;

    protected List<MuleContextFactoryAction> beforeCreationActions;
    protected List<MuleContextFactoryAction> afterCreationActions;

    public static void clearLoggingConfiguration() {
        TestsLogConfigurationHelper.clearLoggingConfig();
    }

    public MunitMuleContextFactory(Properties startUpProperties, List<ConfigurationBuilder> builders) {
        checkNotNull(builders, "The builder list must not be null.");
        checkNotNull(startUpProperties, "The start up properties must not be null.");
        this.builders = builders;
        this.startUpProperties = startUpProperties;

        this.beforeCreationActions = new ArrayList<>();
        this.afterCreationActions = new ArrayList<>();

        this.registerDefaultActions();
    }

    public void registerBeforeCreationAction(MuleContextFactoryAction action) {
        checkNotNull(action, "The action must not be null.");
        beforeCreationActions.add(action);
    }

    public void registerAfterCreationAction(MuleContextFactoryAction action) {
        checkNotNull(action, "The action must not be null.");
        afterCreationActions.add(action);
    }

    public MuleContext createMuleContext() throws Exception {

        performActionsBeforeCreation();

        List<ConfigurationBuilder> builders = getConfigurationBuilderList();

        // Create context builder
        MuleContextBuilder contextBuilder = new DefaultMuleContextBuilder();
        configureMuleContextBuilder(contextBuilder);

        // Create context
        MuleContextFactory muleContextFactory = new DefaultMuleContextFactory();

        MuleContext context = muleContextFactory.createMuleContext(builders, contextBuilder);

        ((DefaultMuleConfiguration) context.getConfiguration()).setShutdownTimeout(0);

        // register notification listeners
        context.getNotificationManager().setNotificationDynamic(true);
        context.getNotificationManager().addInterfaceToType(MessageProcessorNotificationListener.class, MessageProcessorNotification.class);

        performActionsAfterCreation(context);

        return context;
    }


    protected void performActionsBeforeCreation() {
        for (MuleContextFactoryAction action : beforeCreationActions) {
            action.perform(null);
        }
    }

    protected void performActionsAfterCreation(MuleContext context) {
        for (MuleContextFactoryAction action : afterCreationActions) {
            action.perform(context);
        }
    }

    protected List<ConfigurationBuilder> getConfigurationBuilderList() throws Exception {
        List<ConfigurationBuilder> builders = new ArrayList<>();

        builders.add(new SimpleConfigurationBuilder(startUpProperties));

        builders.add(new ExtensionsManagerConfigurationBuilder());

        addBuilderIfPresent(builders, CLASSNAME_ANNOTATIONS_CONFIG_BUILDER);

        builders.addAll(this.builders);

        return builders;
    }

    protected void configureMuleContextBuilder(MuleContextBuilder contextBuilder) {
        contextBuilder.setWorkListener(new TestingWorkListener());
        DefaultMuleConfiguration config = new DefaultMuleConfiguration();
        String muleWorkingDir = System.getProperty(MuleProperties.MULE_WORKING_DIRECTORY_PROPERTY);
        if (null != muleWorkingDir) {
            config.setWorkingDirectory(muleWorkingDir);
        }
        contextBuilder.setMuleConfiguration(config);
    }


    protected void addBuilderIfPresent(List<ConfigurationBuilder> builders, String builderClassName) throws Exception {
        if (ClassUtils.isClassOnPath(builderClassName, this.getClass())) {
            builders.add((ConfigurationBuilder) ClassUtils.instanciateClass(builderClassName, ClassUtils.NO_ARGS, this.getClass()));
        }
    }

    protected void registerDefaultActions() {
        this.registerBeforeCreationAction(new ConfigureLoggingForTestAction());
    }


}
