-- -
-- #%L
-- %%
-- Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
-- %%
-- You may use and modify this code under the terms of the Kuali, Inc.
-- Pre-Release License Agreement. You may not distribute it.
-- 
-- You should have received a copy of the Kuali, Inc. Pre-Release License
-- Agreement with this file. If not, please write to license@kuali.co.
-- #L%
-- -

-- Current situation:
--     * The uniqueness of (subaward_code, sequence_number) is not enforced on the subaward table
--     * The uniqueness of (subaward_code, sequence_number, entry_number) is enforced on the subaward_notepad table,
--       indirectly enforcing uniqueness of (subaward_code, sequence_number) but only for subawards with notes
--     * None of the other tables for associated objects of subawards with sequence_number are enforcing any uniqueness.
--     * The pre-SPR-251 code ensures that new subaward versions get a unique sequence_number relative to the
--       subaward_code AFTER the associated objects are created with an incorrect sequence_number.
--     * There were previous iterations of the code that did not do this (prior to RESMER-452) and would make subaward
--       records with duplicate sequence_number values.

-- Therefore:
--     * It is possible that there are duplicate (subaward_code, sequence_number) pairs in the subaward table.
--     * There are associated objects of subaward for which (subaward_id, subaward_code, sequence_number) does not match
--       the values in the subaward table with the same subaward_id.
--     * The duplicate (subaward_code, sequence_number) records in the subaward table are sharing one record in the
--       version_history table
--     * There are records in some of the databases in the subaward_* tables where the subaward_code does not match
--       the value in the subaward table with the corresponding subaward_id.  These were examined manually and it was
--       determined that the way forward is to update the subaward_code in the subaward_* table to match.

-- Goals of this script:
--     * Repair the subaward table so that (subaward_code, sequence_number) is unique and enforce this with a
--       constraint.
--     * Repair the version_history table so that subaward records are not sharing an entry
--     * Repair the tables for the associated objects of subaward so that they have the same subaward_code and
--       sequence_number as the subaward record does.
--     * Add foreign keys to the subaward_* tables to ensure that the (subaward_id, subaward_code, sequence_number)
--       values exist in the subaward table.


-- Create backups of affected tables ahead of 2203_001 script
create table if not exists V2203_001_SUBAWARD as select * from subaward;
create table if not exists V2203_001_SUBAWARD_AMOUNT_INFO as select * from subaward_amount_info;
create table if not exists V2203_001_SUBAWARD_AMT_RELEASED as select * from subaward_amt_released;
create table if not exists V2203_001_SUBAWARD_ATTACHMENTS as select * from subaward_attachments;
create table if not exists V2203_001_SUBAWARD_CLOSEOUT as select * from subaward_closeout;
create table if not exists V2203_001_SUBAWARD_COMMENT as select * from subaward_comment;
create table if not exists V2203_001_SUBAWARD_CONTACT as select * from subaward_contact;
create table if not exists V2203_001_SUBAWARD_CUSTOM_DATA as select * from subaward_custom_data;
create table if not exists V2203_001_SUBAWARD_FUNDING_SOURCE as select * from subaward_funding_source;
create table if not exists V2203_001_SUBAWARD_NOTEPAD as select * from subaward_notepad;
create table if not exists V2203_001_SUBAWARD_REPORTS as select * from subaward_reports;
create table if not exists V2203_001_SUBAWARD_TEMPLATE_ATTACHMENTS as select * from subaward_template_attachments;
create table if not exists V2203_001_SUBAWARD_TEMPLATE_INFO as select * from subaward_template_info;
create table if not exists V2203_001_VERSION_HISTORY as select * from version_history;

-- Determine the proper sequence numbers for the subawards, as well as the associated version_history records, and
-- store this in a table to simulate an insensitive cursor  (will be dropped later)
drop table if exists V2203_001_corrected;
create table V2203_001_CORRECTED as
    select s.subaward_id,
        vh.version_history_id,
        s.subaward_code,
        s.sequence_number,
        -- Generate the desired sequence number for each by numbering records with the same subaward_code by subaward_id.
        row_number() over (partition by s.subaward_code order by s.subaward_id) as correct_sequence,
        -- When there are multiple records in the subaward table with the same (subaward_code, sequence_number) values,
        -- we select the one with the lowest subaward_id to remain associated with the existing version_history record
        -- corresponding to that (subaward_code, sequence_number) combination.  The others will get new version_history
        -- records.
        s.subaward_id != (min(s.subaward_id) over (partition by s.subaward_code, s.sequence_number)) as create_version_history
    from subaward s
    left outer join version_history vh
        on vh.seq_owner_class_name = 'org.kuali.kra.subaward.bo.SubAward'
        and s.subaward_code = vh.seq_owner_version_name_value
        and s.sequence_number = vh.seq_owner_seq_number
    order by s.subaward_id
;

drop procedure if exists V2203_001__drop_constraint;
drop procedure if exists V2203_001__drop_foreign_key;
drop procedure if exists V2203_001__update_sequence_numbers;
drop procedure if exists V2203_001__enforce_data_integrity;

delimiter /

create procedure V2203_001__drop_constraint(i_table_name varchar(64), i_constraint_name varchar(64))
begin
    if exists (
        select 1 from information_schema.table_constraints
        where constraint_schema = database()
          and table_name = upper(i_table_name)
          and constraint_name = upper(i_constraint_name)
    ) then
        set @drop_sql := concat('alter table ', i_table_name, ' drop constraint ', i_constraint_name);
        prepare drop_constraint_stmt from @drop_sql;
        execute drop_constraint_stmt;
        deallocate prepare drop_constraint_stmt;
    end if;
end
/

create procedure V2203_001__drop_foreign_key(i_table_name varchar(64), i_constraint_name varchar(64))
begin
    if exists (
        select 1 from information_schema.table_constraints
        where constraint_schema = database()
          and table_name = upper(i_table_name)
          and constraint_name = upper(i_constraint_name)
    ) then
        set @drop_sql := concat('alter table ', i_table_name, ' drop foreign key ', i_constraint_name);
        prepare drop_constraint_stmt from @drop_sql;
        execute drop_constraint_stmt;
        deallocate prepare drop_constraint_stmt;
    end if;
end
/

-- This procedure will use the V2203_001_corrected table to update the subaward table so that the sequence numbers are
-- correct.  It will also ensure that the corresponding version_history table is updated with the new sequence number.
-- Previously for subaward records with duplicate (subaward_code, sequence_number), there is only one version_history
-- record.  So additional records in the version_history table are created based on the create_version_history flag
-- in the V2203_001_corrected table.
create procedure V2203_001__update_sequence_numbers()
begin
    declare no_more_records boolean default false;
    declare sub_id decimal(12);
    declare vh_id decimal(22);
    declare sub_code varchar(20);
    declare seq_no decimal(4);
    declare correct_seq_no int;
    declare create_vh boolean;
    declare corrected_cur cursor for select * from v2203_001_corrected;
    declare continue handler for not found set no_more_records = true;

    open corrected_cur;
    cur_loop:
    loop
        fetch corrected_cur into sub_id, vh_id, sub_code, seq_no, correct_seq_no, create_vh;

        if no_more_records then
            leave cur_loop;
        end if;

        update subaward s set s.sequence_number = correct_seq_no
            where s.subaward_id = sub_id and s.sequence_number != correct_seq_no;

        if vh_id is not null then
            if create_vh then
                insert into seq_version_history_id values (null);
                insert into version_history (version_history_id,
                        seq_owner_class_name, seq_owner_version_name_field, seq_owner_version_name_value,
                        seq_owner_seq_number, version_status, version_date, user_id, update_timestamp,
                        update_user, ver_nbr, obj_id)
                    select (select max(id) from seq_version_history_id) as version_history_id,
                        seq_owner_class_name, seq_owner_version_name_field, seq_owner_version_name_value,
                        correct_seq_no as seq_owner_seq_number, version_status, version_date, user_id, update_timestamp,
                        update_user, ver_nbr, uuid() as obj_id
                    from version_history
                    where version_history_id = vh_id;
            else
                update version_history vh set vh.seq_owner_seq_number = correct_seq_no
                    where vh.version_history_id = vh_id and vh.seq_owner_seq_number != correct_seq_no;
            end if;
        end if;
    end loop;
    close corrected_cur;
end
/

create procedure V2203_001__enforce_data_integrity()
begin
    if exists (
        select count(1) cnt from subaward s
        group by subaward_code, sequence_number
        having cnt > 1
    ) then
        signal sqlstate '45000' set message_text = 'Creation of unique constraint UQ_SUBAWARD_CODE_SEQUENCE would fail.';
    end if;
    if exists (
        select count(1) cnt from subaward s
        group by subaward_id, subaward_code, sequence_number
        having cnt > 1
    ) then
        signal sqlstate '45000' set message_text = 'Creation of unique constraint UQ_SUBAWARD_ID_CODE_SEQUENCE would fail.';
    end if;

    if exists (
        select 1 from subaward_amount_info x
        where not exists (
            select 1 from subaward s
            where s.subaward_id = x.subaward_id
              and s.subaward_code = x.subaward_code
              and s.sequence_number = x.sequence_number
        )
    ) then
        signal sqlstate '45000' set message_text = 'Creation of foreign key FK_SUBAWARD_AMOUNT_INFO_CODE_SEQUENCE would fail.';
    end if;

    if exists (
        select 1 from subaward_amt_released x
        where not exists (
            select 1 from subaward s
            where s.subaward_id = x.subaward_id
              and s.subaward_code = x.subaward_code
              and s.sequence_number = x.sequence_number
        )
    ) then
        signal sqlstate '45000' set message_text = 'Creation of foreign key FK_SUBAWARD_AMT_RELEASED_CODE_SEQUENCE would fail.';
    end if;

    if exists (
        select 1 from subaward_attachments x
        where not exists (
            select 1 from subaward s
            where s.subaward_id = x.subaward_id
              and s.subaward_code = x.subaward_code
              and s.sequence_number = x.sequence_number
        )
    ) then
        signal sqlstate '45000' set message_text = 'Creation of foreign key FK_SUBAWARD_ATTACHMENTS_CODE_SEQUENCE would fail.';
    end if;

    if exists (
        select 1 from subaward_closeout x
        where not exists (
            select 1 from subaward s
            where s.subaward_id = x.subaward_id
              and s.subaward_code = x.subaward_code
              and s.sequence_number = x.sequence_number
        )
    ) then
        signal sqlstate '45000' set message_text = 'Creation of foreign key FK_SUBAWARD_CLOSEOUT_CODE_SEQUENCE would fail.';
    end if;

    if exists (
        select 1 from subaward_comment x
        where not exists (
            select 1 from subaward s
            where s.subaward_id = x.subaward_id
              and s.subaward_code = x.subaward_code
              and s.sequence_number = x.sequence_number
        )
    ) then
        signal sqlstate '45000' set message_text = 'Creation of foreign key FK_SUBAWARD_COMMENT_CODE_SEQUENCE would fail.';
    end if;

    if exists (
        select 1 from subaward_contact x
        where not exists (
            select 1 from subaward s
            where s.subaward_id = x.subaward_id
              and s.subaward_code = x.subaward_code
              and s.sequence_number = x.sequence_number
        )
    ) then
        signal sqlstate '45000' set message_text = 'Creation of foreign key FK_SUBAWARD_CONTACT_CODE_SEQUENCE would fail.';
    end if;

    if exists (
        select 1 from subaward_custom_data x
        where not exists (
            select 1 from subaward s
            where s.subaward_id = x.subaward_id
              and s.subaward_code = x.subaward_code
              and s.sequence_number = x.sequence_number
        )
    ) then
        signal sqlstate '45000' set message_text = 'Creation of foreign key FK_SUBAWARD_CUSTOM_DATA_CODE_SEQUENCE would fail.';
    end if;
    if exists (select 1 from subaward_custom_data x where subaward_id is null) then
        signal sqlstate '45000' set message_text = 'Creation of NOT NULL constraint on SUBAWARD_CUSTOM_DATA.SUBAWARD_ID would fail.';
    end if;

    if exists (
        select 1 from subaward_funding_source x
        where not exists (
            select 1 from subaward s
            where s.subaward_id = x.subaward_id
              and s.subaward_code = x.subaward_code
              and s.sequence_number = x.sequence_number
        )
    ) then
        signal sqlstate '45000' set message_text = 'Creation of foreign key FK_SUBAWARD_FUNDING_SOURCE_CODE_SEQUENCE would fail.';
    end if;

    if exists (
        select 1 from subaward_notepad x
        where not exists (
            select 1 from subaward s
            where s.subaward_id = x.subaward_id
              and s.subaward_code = x.subaward_code
              and s.sequence_number = x.sequence_number
        )
    ) then
        signal sqlstate '45000' set message_text = 'Creation of foreign key FK_SUBAWARD_NOTEPAD_CODE_SEQUENCE would fail.';
    end if;
    if exists (
        select count(1) cnt from subaward_notepad x
        group by subaward_code, sequence_number, entry_number
        having cnt > 1
    ) then
        signal sqlstate '45000' set message_text = 'Creation of unique constraint UQ2_SUBAWARD_NOTEPAD would fail.';
    end if;

    if exists (
        select 1 from subaward_reports x
        where not exists (
            select 1 from subaward s
            where s.subaward_id = x.subaward_id
              and s.subaward_code = x.subaward_code
              and s.sequence_number = x.sequence_number
        )
    ) then
        signal sqlstate '45000' set message_text = 'Creation of foreign key FK_SUBAWARD_REPORTS_CODE_SEQUENCE would fail.';
    end if;

    if exists (
        select 1 from subaward_template_attachments x
        where not exists (
            select 1 from subaward s
            where s.subaward_id = x.subaward_id
              and s.subaward_code = x.subaward_code
              and s.sequence_number = x.sequence_number
        )
    ) then
        signal sqlstate '45000' set message_text = 'Creation of foreign key FK_SUBAWARD_TEMPLATE_ATTACHMENTS_CODE_SEQUENCE would fail.';
    end if;

    if exists (
        select 1 from subaward_template_info x
        where not exists (
            select 1 from subaward s
            where s.subaward_id = x.subaward_id
              and s.subaward_code = x.subaward_code
              and s.sequence_number = x.sequence_number
        )
    ) then
        signal sqlstate '45000' set message_text = 'Creation of foreign key FK_SUBAWARD_TEMPLATE_INFO_CODE_SEQUENCE would fail.';
    end if;
end
/

delimiter ;

call V2203_001__drop_foreign_key('subaward_amount_info', 'fk_subaward_amount_info_code_sequence');
call V2203_001__drop_foreign_key('subaward_amt_released', 'fk_subaward_amt_released_code_sequence');
call V2203_001__drop_foreign_key('subaward_attachments', 'fk_subaward_attachments_code_sequence');
call V2203_001__drop_foreign_key('subaward_closeout', 'fk_subaward_closeout_code_sequence');
call V2203_001__drop_foreign_key('subaward_comment', 'fk_subaward_comment_code_sequence');
call V2203_001__drop_foreign_key('subaward_contact', 'fk_subaward_contact_code_sequence');
call V2203_001__drop_foreign_key('subaward_custom_data', 'fk_subaward_custom_data_code_sequence');
call V2203_001__drop_foreign_key('subaward_funding_source', 'fk_subaward_funding_source_code_sequence');
call V2203_001__drop_foreign_key('subaward_notepad', 'fk_subaward_notepad_code_sequence');
call V2203_001__drop_foreign_key('subaward_reports', 'fk_subaward_reports_code_sequence');
call V2203_001__drop_foreign_key('subaward_template_attachments', 'fk_subaward_template_attachments_code_sequence');
call V2203_001__drop_foreign_key('subaward_template_info', 'fk_subaward_template_info_code_sequence');

call V2203_001__drop_constraint('subaward', 'uq_subaward_code_sequence');
call V2203_001__drop_constraint('subaward', 'uq_subaward_id_code_sequence');
call V2203_001__drop_constraint('subaward_notepad', 'uq2_subaward_notepad');

call V2203_001__update_sequence_numbers();


-- At this point the subaward table has been repaired so that (subaward_code, sequence_number) is unique.  Furthermore
-- there should be exactly one version_history record for each subaward record.


-- Repair the tables for the associated objects of subaward so that they have the same subaward_code and sequence_number
-- as the subaward record does.

with to_be_updated as (
    select x.subaward_amount_info_id as id
    from subaward_amount_info x
    inner join subaward s on s.subaward_id = x.subaward_id
    where x.subaward_code != s.subaward_code or x.sequence_number != s.sequence_number
) update subaward_amount_info y
set subaward_code = (select s.subaward_code from subaward s where s.subaward_id = y.subaward_id),
    sequence_number = (select s.sequence_number from subaward s where s.subaward_id = y.subaward_id)
where y.subaward_amount_info_id in (select id from to_be_updated);

with to_be_updated as (
    select x.subaward_amt_released_id as id
    from subaward_amt_released x
    inner join subaward s on s.subaward_id = x.subaward_id
    where x.subaward_code != s.subaward_code or x.sequence_number != s.sequence_number
) update subaward_amt_released y
set subaward_code = (select s.subaward_code from subaward s where s.subaward_id = y.subaward_id),
    sequence_number = (select s.sequence_number from subaward s where s.subaward_id = y.subaward_id)
where y.subaward_amt_released_id in (select id from to_be_updated);

with to_be_updated as (
    select x.attachment_id as id
    from subaward_attachments x
    inner join subaward s on s.subaward_id = x.subaward_id
    where x.subaward_code != s.subaward_code or x.sequence_number != s.sequence_number
) update subaward_attachments y
set subaward_code = (select s.subaward_code from subaward s where s.subaward_id = y.subaward_id),
    sequence_number = (select s.sequence_number from subaward s where s.subaward_id = y.subaward_id)
where y.attachment_id in (select id from to_be_updated);

with to_be_updated as (
    select x.subaward_closeout_id as id
    from subaward_closeout x
    inner join subaward s on s.subaward_id = x.subaward_id
    where x.subaward_code != s.subaward_code or x.sequence_number != s.sequence_number
) update subaward_closeout y
set subaward_code = (select s.subaward_code from subaward s where s.subaward_id = y.subaward_id),
    sequence_number = (select s.sequence_number from subaward s where s.subaward_id = y.subaward_id)
where y.subaward_closeout_id in (select id from to_be_updated);

with to_be_updated as (
    select x.subaward_comment_id as id
    from subaward_comment x
    inner join subaward s on s.subaward_id = x.subaward_id
    where x.subaward_code != s.subaward_code or x.sequence_number != s.sequence_number
) update subaward_comment y
set subaward_code = (select s.subaward_code from subaward s where s.subaward_id = y.subaward_id),
    sequence_number = (select s.sequence_number from subaward s where s.subaward_id = y.subaward_id)
where y.subaward_comment_id in (select id from to_be_updated);

with to_be_updated as (
    select x.subaward_contact_id as id
    from subaward_contact x
    inner join subaward s on s.subaward_id = x.subaward_id
    where x.subaward_code != s.subaward_code or x.sequence_number != s.sequence_number
) update subaward_contact y
set subaward_code = (select s.subaward_code from subaward s where s.subaward_id = y.subaward_id),
    sequence_number = (select s.sequence_number from subaward s where s.subaward_id = y.subaward_id)
where y.subaward_contact_id in (select id from to_be_updated);

with to_be_updated as (
    select x.subaward_custom_data_id as id
    from subaward_custom_data x
    inner join subaward s on s.subaward_id = x.subaward_id
    where x.subaward_code != s.subaward_code or x.sequence_number != s.sequence_number
) update subaward_custom_data y
set subaward_code = (select s.subaward_code from subaward s where s.subaward_id = y.subaward_id),
    sequence_number = (select s.sequence_number from subaward s where s.subaward_id = y.subaward_id)
where y.subaward_custom_data_id in (select id from to_be_updated);

with to_be_updated as (
    select x.subaward_funding_source_id as id
    from subaward_funding_source x
    inner join subaward s on s.subaward_id = x.subaward_id
    where x.subaward_code != s.subaward_code or x.sequence_number != s.sequence_number
) update subaward_funding_source y
set subaward_code = (select s.subaward_code from subaward s where s.subaward_id = y.subaward_id),
    sequence_number = (select s.sequence_number from subaward s where s.subaward_id = y.subaward_id)
where y.subaward_funding_source_id in (select id from to_be_updated);

with to_be_updated as (
    select x.subaward_notepad_id as id
    from subaward_notepad x
    inner join subaward s on s.subaward_id = x.subaward_id
    where x.subaward_code != s.subaward_code or x.sequence_number != s.sequence_number
) update subaward_notepad y
set subaward_code = (select s.subaward_code from subaward s where s.subaward_id = y.subaward_id),
    sequence_number = (select s.sequence_number from subaward s where s.subaward_id = y.subaward_id)
where y.subaward_notepad_id in (select id from to_be_updated);

with to_be_updated as (
    select x.subaward_report_id as id
    from subaward_reports x
    inner join subaward s on s.subaward_id = x.subaward_id
    where x.subaward_code != s.subaward_code or x.sequence_number != s.sequence_number
) update subaward_reports y
set subaward_code = (select s.subaward_code from subaward s where s.subaward_id = y.subaward_id),
    sequence_number = (select s.sequence_number from subaward s where s.subaward_id = y.subaward_id)
where y.subaward_report_id in (select id from to_be_updated);

with to_be_updated as (
    select x.attachment_id as id
    from subaward_template_attachments x
    inner join subaward s on s.subaward_id = x.subaward_id
    where x.subaward_code != s.subaward_code or x.sequence_number != s.sequence_number
) update subaward_template_attachments y
set subaward_code = (select s.subaward_code from subaward s where s.subaward_id = y.subaward_id),
    sequence_number = (select s.sequence_number from subaward s where s.subaward_id = y.subaward_id)
where y.attachment_id in (select id from to_be_updated);

-- subaward_template_info is unusual in that subaward_id is the primary key
with to_be_updated as (
    select x.subaward_id as id
    from subaward_template_info x
    inner join subaward s on s.subaward_id = x.subaward_id
    where x.subaward_code != s.subaward_code or x.sequence_number != s.sequence_number
) update subaward_template_info y
set subaward_code = (select s.subaward_code from subaward s where s.subaward_id = y.subaward_id),
    sequence_number = (select s.sequence_number from subaward s where s.subaward_id = y.subaward_id)
where y.subaward_id in (select id from to_be_updated);

-- subaward_attachments has orphans despite having a foreign key
delete from subaward_attachments sta where not exists (select 1 from subaward s where s.subaward_id = sta.subaward_id);

-- subaward_custom_data did not previously have a foreign key reference on subaward_id and some environments have
-- orphan records
delete from subaward_custom_data scd where not exists (select 1 from subaward s where s.subaward_id = scd.subaward_id);

-- subaward_reports did not previously have a foreign key reference on subaward_id and some environments have
-- orphan records
delete from subaward_reports sr where not exists (select 1 from subaward s where s.subaward_id = sr.subaward_id);

-- subaward_notepad has orphans despite having a foreign key
delete from subaward_notepad sn where not exists (select 1 from subaward s where s.subaward_id = sn.subaward_id);

-- subaward_template_attachments has orphans despite having a foreign key
delete from subaward_template_attachments sta where not exists (select 1 from subaward s where s.subaward_id = sta.subaward_id);

call V2203_001__enforce_data_integrity();

-- This constraint enforces uniqueness of (subaward_code, sequence_number) within subaward
alter table subaward add constraint UQ_SUBAWARD_CODE_SEQUENCE unique (subaward_code, sequence_number);

-- This constraint, while redundant, allows us to create the desired foreign keys in the subaward_* tables
alter table subaward add constraint UQ_SUBAWARD_ID_CODE_SEQUENCE unique (subaward_id, subaward_code, sequence_number);

alter table subaward_custom_data modify SUBAWARD_ID decimal(12) not null;

alter table subaward_amount_info
    add constraint FK_SUBAWARD_AMOUNT_INFO_CODE_SEQUENCE foreign key (subaward_id, subaward_code, sequence_number)
    references subaward (subaward_id, subaward_code, sequence_number);

alter table subaward_amt_released
    add constraint FK_SUBAWARD_AMT_RELEASED_CODE_SEQUENCE foreign key (subaward_id, subaward_code, sequence_number)
    references subaward (subaward_id, subaward_code, sequence_number);

alter table subaward_attachments
    add constraint FK_SUBAWARD_ATTACHMENTS_CODE_SEQUENCE foreign key (subaward_id, subaward_code, sequence_number)
        references subaward (subaward_id, subaward_code, sequence_number);

alter table subaward_closeout
    add constraint FK_SUBAWARD_CLOSEOUT_CODE_SEQUENCE foreign key (subaward_id, subaward_code, sequence_number)
    references subaward (subaward_id, subaward_code, sequence_number);

alter table subaward_comment
    add constraint FK_SUBAWARD_COMMENT_CODE_SEQUENCE foreign key (subaward_id, subaward_code, sequence_number)
    references subaward (subaward_id, subaward_code, sequence_number);

alter table subaward_contact
    add constraint FK_SUBAWARD_CONTACT_CODE_SEQUENCE foreign key (subaward_id, subaward_code, sequence_number)
    references subaward (subaward_id, subaward_code, sequence_number);

alter table subaward_custom_data
    add constraint FK_SUBAWARD_CUSTOM_DATA_CODE_SEQUENCE foreign key (subaward_id, subaward_code, sequence_number)
    references subaward (subaward_id, subaward_code, sequence_number);

alter table subaward_funding_source
    add constraint FK_SUBAWARD_FUNDING_SOURCE_CODE_SEQUENCE foreign key (subaward_id, subaward_code, sequence_number)
    references subaward (subaward_id, subaward_code, sequence_number);

alter table subaward_notepad
    add constraint FK_SUBAWARD_NOTEPAD_CODE_SEQUENCE foreign key (subaward_id, subaward_code, sequence_number)
    references subaward (subaward_id, subaward_code, sequence_number);
alter table subaward_notepad
    add constraint UQ2_SUBAWARD_NOTEPAD unique (subaward_code, sequence_number, entry_number);

alter table subaward_reports
    add constraint FK_SUBAWARD_REPORTS_CODE_SEQUENCE foreign key (subaward_id, subaward_code, sequence_number)
    references subaward (subaward_id, subaward_code, sequence_number);

alter table subaward_template_attachments
    add constraint FK_SUBAWARD_TEMPLATE_ATTACHMENTS_CODE_SEQUENCE foreign key (subaward_id, subaward_code, sequence_number)
    references subaward (subaward_id, subaward_code, sequence_number);

alter table subaward_template_info
    add constraint FK_SUBAWARD_TEMPLATE_INFO_CODE_SEQUENCE foreign key (subaward_id, subaward_code, sequence_number)
    references subaward (subaward_id, subaward_code, sequence_number);

drop procedure if exists V2203_001__drop_constraint;
drop procedure if exists V2203_001__drop_foreign_key;
drop procedure if exists V2203_001__update_sequence_numbers;
drop procedure if exists V2203_001__enforce_data_integrity;
drop table if exists V2203_001_corrected;
