/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.api.core;

import java.util.Arrays;

/**
 * This is an error class that is used to capture an s2s error.
 * These errors are usually displayed to the end user.
 *
 * These errors contain five parts:
 * 1) an error key: which field or section was in error
 * 2) a message key: a key to use to lookup a message to display to an end user
 * 3) a link to place the error on
 * 4) the level of the error
 * 5) any error parameters that could be injected into the error message
 */
public final class AuditError {

    public static final String NO_FIELD_ERROR_KEY = "noField";
    public static final String GG_LINK = "grantsGov" + "." + "Opportunity";

    private String errorKey;
    private String messageKey;
    private String link;
    private Level level;
    private String[] errorParms;

    public AuditError() {
    }

    public AuditError(String errorKey, String messageKey, String link, String... errorParms) {
        this.errorKey = errorKey;
        this.messageKey = messageKey;
        this.link = link;
        this.level = Level.ERROR;
        this.errorParms = errorParms;
    }

    public AuditError(String errorKey, String messageKey, String link, Level level, String... errorParms) {
        this.errorKey = errorKey;
        this.messageKey = messageKey;
        this.link = link;
        this.level = level;
        this.errorParms = errorParms;
    }

    public String getErrorKey() {
        return errorKey;
    }

    public void setErrorKey(String errorKey) {
        this.errorKey = errorKey;
    }

    public String getLink() {
        return link;
    }

    public void setLink(String link) {
        this.link = link;
    }

    public String getMessageKey() {
        return messageKey;
    }

    public void setMessageKey(String messageKey) {
        this.messageKey = messageKey;
    }

    public Level getLevel() {
        return level;
    }

    public void setLevel(Level level) {
        this.level = level;
    }

    public String[] getErrorParms() {
        return errorParms;
    }

    public void setErrorParms(String[] errorParms) {
        this.errorParms = errorParms;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        AuditError that = (AuditError) o;

        if (errorKey != null ? !errorKey.equals(that.errorKey) : that.errorKey != null) return false;
        if (messageKey != null ? !messageKey.equals(that.messageKey) : that.messageKey != null) return false;
        if (link != null ? !link.equals(that.link) : that.link != null) return false;
        if (level != that.level) return false;

        return Arrays.equals(errorParms, that.errorParms);
    }

    @Override
    public int hashCode() {
        int result = errorKey != null ? errorKey.hashCode() : 0;
        result = 31 * result + (messageKey != null ? messageKey.hashCode() : 0);
        result = 31 * result + (link != null ? link.hashCode() : 0);
        result = 31 * result + (level != null ? level.hashCode() : 0);
        result = 31 * result + Arrays.hashCode(errorParms);
        return result;
    }

    @Override
    public String toString() {
        return "AuditError{" +
                "errorKey='" + errorKey + '\'' +
                ", messageKey='" + messageKey + '\'' +
                ", link='" + link + '\'' +
                ", level=" + level +
                ", errorParms=" + Arrays.toString(errorParms) +
                '}';
    }

    public enum Level { ERROR, WARNING }
}

