/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;


import gov.grants.apply.forms.phs398CoverPageSupplement50V50.PHS398CoverPageSupplement50Document;
import gov.grants.apply.forms.phs398CoverPageSupplement50V50.PHS398CoverPageSupplement50Document.PHS398CoverPageSupplement50;
import gov.grants.apply.forms.phs398CoverPageSupplement50V50.PHS398CoverPageSupplement50Document.PHS398CoverPageSupplement50.IncomeBudgetPeriod;
import gov.grants.apply.forms.phs398CoverPageSupplement50V50.PHS398CoverPageSupplement50Document.PHS398CoverPageSupplement50.StemCells;
import gov.grants.apply.forms.phs398CoverPageSupplement50V50.PHS398CoverPageSupplement50Document.PHS398CoverPageSupplement50.VertebrateAnimals;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.globalLibraryV20.HumanNameDataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.rolodex.RolodexContract;
import org.kuali.coeus.common.api.rolodex.RolodexService;
import org.kuali.coeus.common.api.ynq.YnqConstant;
import org.kuali.coeus.common.budget.api.income.BudgetProjectIncomeContract;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerHeaderContract;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.budget.ProposalDevelopmentBudgetExtContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.budget.S2SCommonBudgetService;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.util.FieldValueConstants;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Consumer;

@FormGenerator("PHS398CoverPageSupplement_5_0V5_0Generator")
public class PHS398CoverPageSupplement_5_0V5_0Generator extends PHS398CoverPageSupplementBaseGenerator<PHS398CoverPageSupplement50Document> implements S2SFormGeneratorPdfFillable<PHS398CoverPageSupplement50Document> {

    protected static final Integer PROPOSAL_YNQ_QUESTION_114 = 114;
    protected static final Integer PROPOSAL_YNQ_QUESTION_115 = 115;
    protected static final Integer PROPOSAL_YNQ_QUESTION_116 = 116;
    protected static final Integer PROPOSAL_YNQ_QUESTION_117 = 117;
    protected static final Integer PROPOSAL_YNQ_QUESTION_118 = 118;
    protected static final Integer PROPOSAL_YNQ_QUESTION_119 = 119;
    protected static final Integer PROPOSAL_YNQ_QUESTION_120 = 120;
    protected static final Integer PROPOSAL_YNQ_QUESTION_145 = 145;
    protected static final Integer PROPOSAL_YNQ_QUESTION_146 = 146;
    protected static final Integer PROPOSAL_YNQ_QUESTION_147 = 147;
    protected static final Integer PROPOSAL_YNQ_QUESTION_148 = 148;
    protected static final int HUMAN_FETAL_TISSUE_INVOLVED = -10145;
    public static final Integer INCREASED_REGISTRATION_NUMBER = 149;
    protected static final int PROJECT_INCOME_DESCRIPTION_MAX_LENGTH = 150;
    private static final int MAX_EUTHANASIA_METHOD_DESC = 1000;
    protected static final int HFT_COMPLIANCE_ASSURANCE = -5;
    protected static final int HFT_IRB_CONSENT_FORM = -6;

    private List<? extends AnswerHeaderContract> answerHeaders;

    @Value("http://apply.grants.gov/forms/PHS398_CoverPageSupplement_5_0-V5.0")
    private String namespace;

    @Value("PHS398_CoverPageSupplement_5_0-V5.0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/PHS398_CoverPageSupplement-V5.0.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/PHS398_CoverPageSupplement_5_0-V5.0.pdf")
    private Resource pdfForm;

    @Value("158")
    private int sortIndex;

    @Autowired
    @Qualifier("rolodexService")
    private RolodexService rolodexService;

    @Autowired
    @Qualifier("s2SCommonBudgetService")
    private S2SCommonBudgetService s2SCommonBudgetService;

    private PHS398CoverPageSupplement50Document getCoverPageSupplement() {
        PHS398CoverPageSupplement50Document coverPageSupplementDocument = PHS398CoverPageSupplement50Document.Factory.newInstance();
        PHS398CoverPageSupplement50 coverPageSupplement = PHS398CoverPageSupplement50.Factory.newInstance();
        answerHeaders = getPropDevQuestionAnswerService().getQuestionnaireAnswerHeaders(pdDoc.getDevelopmentProposal().getProposalNumber(), getNamespace(), getFormName());
        coverPageSupplement.setFormVersion(FormVersion.v5_0.getVersion());
        setIsInventionsAndPatentsAndIsPreviouslyReported(coverPageSupplement);
        setFormerPDNameAndIsChangeOfPDPI(coverPageSupplement);
        setFormerInstitutionNameAndChangeOfInstitution(coverPageSupplement);
        ProposalDevelopmentBudgetExtContract budget = s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());

        if (budget != null) {
            setIncomeBudgetPeriods(coverPageSupplement, budget.getBudgetProjectIncomes());
        } else {
            coverPageSupplement.setProgramIncome(YesNoDataType.N_NO);
        }

        StemCells stemCells = getStemCells();
        coverPageSupplement.setStemCells(stemCells);

        VertebrateAnimals vertebrateAnimals = getVertebrateAnimals();
        if (vertebrateAnimals != null) {
            coverPageSupplement.setVertebrateAnimals(vertebrateAnimals);
        }

        setHFTInvolved(coverPageSupplement);

        coverPageSupplementDocument.setPHS398CoverPageSupplement50(coverPageSupplement);
        return coverPageSupplementDocument;
    }

    private VertebrateAnimals getVertebrateAnimals() {
        final VertebrateAnimals vertebrateAnimals = VertebrateAnimals.Factory.newInstance();
        String answer = getAnswer(PROPOSAL_YNQ_QUESTION_145, answerHeaders);
        if (answer == null || answer.equals(NOT_ANSWERED) || YnqConstant.NO.code().equals(answer))
            return null;

        setVertebrateAnimalsSubQuestions(vertebrateAnimals);
        return vertebrateAnimals;
    }

    private void setVertebrateAnimalsSubQuestions(VertebrateAnimals vertebrateAnimals) {
        String answer = getAnswer(PROPOSAL_YNQ_QUESTION_146, answerHeaders);
        if (answer != null && YnqConstant.YES.code().equals(answer)) {
            vertebrateAnimals.setAnimalEuthanasiaIndicator(YesNoDataType.Y_YES);
        } else if (answer != null && YnqConstant.NO.code().equals(answer)) {
            vertebrateAnimals.setAnimalEuthanasiaIndicator(YesNoDataType.N_NO);
        }

        answer = getAnswer(PROPOSAL_YNQ_QUESTION_147, answerHeaders);
        if (answer != null && YnqConstant.YES.code().equals(answer)) {
            vertebrateAnimals.setAVMAConsistentIndicator(YesNoDataType.Y_YES);
        } else if (answer != null && YnqConstant.NO.code().equals(answer)) {
            vertebrateAnimals.setAVMAConsistentIndicator(YesNoDataType.N_NO);
            answer = getAnswer(PROPOSAL_YNQ_QUESTION_148, answerHeaders);
            if (answer != null) {
                vertebrateAnimals.setEuthanasiaMethodDescription(StringUtils.substring(answer.trim(), 0, MAX_EUTHANASIA_METHOD_DESC));
            }
        }
    }

    private void setIsInventionsAndPatentsAndIsPreviouslyReported(PHS398CoverPageSupplement50 coverPageSupplement) {
        String answer = getAnswer(PROPOSAL_YNQ_QUESTION_118, answerHeaders);
        if (answer != null && !answer.equals(NOT_ANSWERED)) {
            if (YnqConstant.YES.code().equals(answer)) {
                setInventionsAndPatentsAndIsPreviouslyReportedSubQuestions(coverPageSupplement);
            }
        }
    }

    private void setInventionsAndPatentsAndIsPreviouslyReportedSubQuestions(PHS398CoverPageSupplement50 coverPageSupplement) {
        String explanation = getAnswer(PROPOSAL_YNQ_QUESTION_119, answerHeaders);
        if (explanation != null && !explanation.equals(NOT_ANSWERED)) {
            if (YnqConstant.YES.code().equals(explanation)) {
                coverPageSupplement.setIsInventionsAndPatents(YesNoDataType.Y_YES);
                String subQuestionExplanation = getAnswer(PROPOSAL_YNQ_QUESTION_120, answerHeaders);
                if (subQuestionExplanation != null && !subQuestionExplanation.equals(NOT_ANSWERED)) {
                    if (YnqConstant.YES.code().equals(subQuestionExplanation)) {
                        coverPageSupplement.setIsPreviouslyReported(YesNoDataType.Y_YES);
                    } else if (YnqConstant.NO.code().equals(subQuestionExplanation)) {
                        coverPageSupplement.setIsPreviouslyReported(YesNoDataType.N_NO);
                    }
                } else {
                    coverPageSupplement.setIsPreviouslyReported(null);
                }

            } else if (YnqConstant.NO.code().equals(explanation)) {
                coverPageSupplement.setIsInventionsAndPatents(YesNoDataType.N_NO);
            }
        } else {
            coverPageSupplement.setIsInventionsAndPatents(null);
        }
    }

    private void setFormerPDNameAndIsChangeOfPDPI(PHS398CoverPageSupplement50 coverPageSupplement) {
        String answer = getAnswer(PROPOSAL_YNQ_QUESTION_114, answerHeaders);
        String explanation = getAnswer(PROPOSAL_YNQ_QUESTION_115, answerHeaders);
        if (YnqConstant.YES.code().equals(answer)) {
            coverPageSupplement.setIsChangeOfPDPI(YesNoDataType.Y_YES);
            if (explanation != null) {
                RolodexContract rolodex = rolodexService.getRolodex(Integer.valueOf(explanation));
                HumanNameDataType formerPDName = globLibV20Generator.getHumanNameDataType(rolodex);
                if (formerPDName != null
                        && formerPDName.getFirstName() != null
                        && formerPDName.getLastName() != null) {
                    coverPageSupplement.setFormerPDName(formerPDName);
                }
            } else {
                coverPageSupplement.setFormerPDName(null);
            }
        } else {
            coverPageSupplement.setIsChangeOfPDPI(YesNoDataType.N_NO);
        }
    }

    private void setFormerInstitutionNameAndChangeOfInstitution(PHS398CoverPageSupplement50 coverPageSupplement) {
        String answer = getAnswer(PROPOSAL_YNQ_QUESTION_116, answerHeaders);
        String explanation = getAnswer(PROPOSAL_YNQ_QUESTION_117, answerHeaders);

        if (YnqConstant.YES.code().equals(answer)) {
            coverPageSupplement.setIsChangeOfInstitution(YesNoDataType.Y_YES);
            if (explanation != null) {
                coverPageSupplement.setFormerInstitutionName(explanation);
            } else {
                coverPageSupplement.setFormerInstitutionName(null);
            }
        } else {
            coverPageSupplement.setIsChangeOfInstitution(YesNoDataType.N_NO);
        }
    }

    private static void setIncomeBudgetPeriods(PHS398CoverPageSupplement50 coverPageSupplement, List<? extends BudgetProjectIncomeContract> projectIncomes) {
        if (projectIncomes.isEmpty()) {
            coverPageSupplement.setProgramIncome(YesNoDataType.N_NO);
        } else {
            coverPageSupplement.setProgramIncome(YesNoDataType.Y_YES);
        }
        coverPageSupplement.setIncomeBudgetPeriodArray(getIncomeBudgetPeriod(projectIncomes));
    }

    private static IncomeBudgetPeriod[] getIncomeBudgetPeriod(final List<? extends BudgetProjectIncomeContract> projectIncomes) {
        //TreeMap Used to maintain the order of the Budget periods.
        Map<Integer, IncomeBudgetPeriod> incomeBudgetPeriodMap = new TreeMap<>();
        BigDecimal anticipatedAmount;
        for (BudgetProjectIncomeContract projectIncome : projectIncomes) {

            Integer budgetPeriodNumber = projectIncome.getBudgetPeriodNumber();
            IncomeBudgetPeriod incomeBudgPeriod = incomeBudgetPeriodMap
                    .get(budgetPeriodNumber);
            if (incomeBudgPeriod == null) {
                incomeBudgPeriod = IncomeBudgetPeriod.Factory.newInstance();
                incomeBudgPeriod.setBudgetPeriod(budgetPeriodNumber);
                anticipatedAmount = BigDecimal.ZERO;
            } else {
                anticipatedAmount = incomeBudgPeriod.getAnticipatedAmount();
            }
            anticipatedAmount = anticipatedAmount.add(projectIncome
                    .getProjectIncome().bigDecimalValue());
            incomeBudgPeriod.setAnticipatedAmount(anticipatedAmount);
            String description = getProjectIncomeDescription(projectIncome);
            if (description != null) {
                if (incomeBudgPeriod.getSource() != null) {
                    incomeBudgPeriod.setSource(incomeBudgPeriod.getSource()
                            + ";" + description);
                } else {
                    incomeBudgPeriod.setSource(description);
                }
            }
            incomeBudgetPeriodMap.put(budgetPeriodNumber, incomeBudgPeriod);
        }
        return incomeBudgetPeriodMap.values().toArray(new IncomeBudgetPeriod[0]);
    }

    protected static String getProjectIncomeDescription(BudgetProjectIncomeContract projectIncome) {
        return projectIncome.getDescription() != null ? StringUtils.substring(projectIncome.getDescription(), 0, PROJECT_INCOME_DESCRIPTION_MAX_LENGTH) : null;
    }

    private StemCells getStemCells() {

        StemCells stemCells = StemCells.Factory.newInstance();

        String answer = getAnswer(IS_HUMAN_STEM_CELLS_INVOLVED, answerHeaders);

        if (answer != null && !answer.equals(NOT_ANSWERED)) {
            if (YnqConstant.YES.code().equals(answer)) {
                stemCells.setIsHumanStemCellsInvolved(YesNoDataType.Y_YES);
                setStemCellsSubQuestions(stemCells);
            } else {
                stemCells.setIsHumanStemCellsInvolved(YesNoDataType.N_NO);
            }
        }
        return stemCells;
    }

    private void setStemCellsSubQuestions(StemCells stemCells) {
        String subAnswer = getAnswer(SPECIFIC_STEM_CELL_LINE, answerHeaders);
        String childAnswer = null;
        if (subAnswer != null) {
            if (!subAnswer.equals(NOT_ANSWERED)) {
                if (YnqConstant.YES.code().equals(subAnswer)) {
                    stemCells.setStemCellsIndicator(YesNoDataType.N_NO);
                    childAnswer = getAnswers(INCREASED_REGISTRATION_NUMBER, answerHeaders);
                } else {
                    stemCells.setStemCellsIndicator(YesNoDataType.Y_YES);
                }
            }
        }
        if (childAnswer != null) {
            if (FieldValueConstants.VALUE_UNKNOWN.equalsIgnoreCase(childAnswer)) {
                stemCells.setStemCellsIndicator(YesNoDataType.Y_YES);
            } else {
                List<String> cellLines = getCellLines(childAnswer);
                if (cellLines.size() > 0) {
                    stemCells.setCellLinesArray(cellLines.toArray(new String[0]));
                }
            }
        }
    }

    private void setHFTInvolved(PHS398CoverPageSupplement50 coverPageSupplement) {
        String answer = getAnswer(HUMAN_FETAL_TISSUE_INVOLVED, answerHeaders);

        if (answer != null && !answer.equals(NOT_ANSWERED)) {
            if (YnqConstant.YES.code().equals(answer)) {
                coverPageSupplement.setIsHumanFetalTissueInvolved(YesNoDataType.Y_YES);
                addAttachmentsForHFT(coverPageSupplement);
            } else {
                coverPageSupplement.setIsHumanFetalTissueInvolved(YesNoDataType.N_NO);
            }
        }
    }

    private void addAttachmentsForHFT(PHS398CoverPageSupplement50 coverPageSupplement) {
        pdDoc.getDevelopmentProposal().getNarratives().forEach(narrative -> {
            switch (Integer.parseInt(narrative.getNarrativeType().getCode())) {
                case HFT_IRB_CONSENT_FORM:
                    doWithAttachedFileType(narrative, attachedFileDataType -> {
                        PHS398CoverPageSupplement50.HFTIRBConsentForm consentForm = PHS398CoverPageSupplement50.HFTIRBConsentForm.Factory.newInstance();
                        consentForm.setAttFile(attachedFileDataType);
                        coverPageSupplement.setHFTIRBConsentForm(consentForm);
                    });
                    break;
                case HFT_COMPLIANCE_ASSURANCE:
                    doWithAttachedFileType(narrative, attachedFileDataType -> {
                        PHS398CoverPageSupplement50.ComplianceAssurance complianceAssurance = PHS398CoverPageSupplement50.ComplianceAssurance.Factory.newInstance();
                        complianceAssurance.setAttFile(attachedFileDataType);
                        coverPageSupplement.setComplianceAssurance(complianceAssurance);
                    });
                    break;
            }
        });
    }

    private void doWithAttachedFileType(NarrativeContract narrative, Consumer<AttachedFileDataType> fn) {
        Optional.ofNullable(addAttachedFileType(narrative)).ifPresent(fn);
    }

    @Override
    public PHS398CoverPageSupplement50Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getCoverPageSupplement();
    }

    public RolodexService getRolodexService() {
        return rolodexService;
    }

    public void setRolodexService(RolodexService rolodexService) {
        this.rolodexService = rolodexService;
    }

    public S2SCommonBudgetService getS2SCommonBudgetService() {
        return s2SCommonBudgetService;
    }

    public void setS2SCommonBudgetService(S2SCommonBudgetService s2SCommonBudgetService) {
        this.s2SCommonBudgetService = s2SCommonBudgetService;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(PHS398CoverPageSupplement50Document form, List<AttachmentData> attachments) {
        return new Attachments(Collections.emptyMap(), attachments);
    }

    @Override
    public DocumentFactory<PHS398CoverPageSupplement50Document> factory() {
        return PHS398CoverPageSupplement50Document.Factory;
    }
}
