/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.coeus.additionalEquipment.AdditionalEquipmentListDocument;
import gov.grants.apply.coeus.additionalEquipment.AdditionalEquipmentListDocument.AdditionalEquipmentList;
import gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument;
import gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList;
import gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons.Compensation;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.xmlbeans.XmlObject;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemContract;
import org.kuali.coeus.common.budget.api.personnel.BudgetPersonnelDetailsContract;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.attachment.NarrativeService;
import org.kuali.coeus.propdev.api.budget.ProposalDevelopmentBudgetExtContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.s2sgen.api.budget.S2SBudgetInfoService;
import org.kuali.coeus.s2sgen.api.budget.S2SBudgetPeriodDto;
import org.kuali.coeus.s2sgen.api.budget.S2SCostDto;
import org.kuali.coeus.s2sgen.api.budget.S2SKeyPersonDto;
import org.kuali.coeus.s2sgen.api.core.AuditError;
import org.kuali.coeus.s2sgen.api.core.S2SException;
import org.kuali.coeus.s2sgen.impl.budget.S2SBudgetValidationService;
import org.kuali.coeus.s2sgen.impl.budget.S2SCommonBudgetService;
import org.kuali.coeus.s2sgen.impl.datetime.S2SDateTimeService;
import org.kuali.coeus.s2sgen.impl.generate.S2SBaseFormGenerator;
import org.kuali.coeus.s2sgen.impl.print.GenericPrintable;
import org.kuali.coeus.s2sgen.impl.print.S2SPrintingService;
import org.kuali.coeus.sys.api.model.KcFile;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * This abstract class has methods that are common to all the versions of
 * RRBudget form.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
public abstract class RRBudgetBaseGenerator<T extends XmlObject> extends S2SBaseFormGenerator<T> {

    private static final Logger LOG = LogManager.getLogger(RRBudgetBaseGenerator.class);

    private static final String ADDITIONAL_EQUIPMENT = "ADDITIONAL_EQUIPMENT";
	public static final String OTHERCOST_DESCRIPTION = "Other";
	public static final String OTHERPERSONNEL_POSTDOC = "PostDoc";
	public static final String OTHERPERSONNEL_GRADUATE = "Grad";
	public static final String OTHERPERSONNEL_UNDERGRADUATE = "UnderGrad";
	public static final String OTHERPERSONNEL_SECRETARIAL = "Sec";
	public static final int BUDGET_JUSTIFICATION_ATTACHMENT = 7;

	protected static final int OTHERPERSONNEL_MAX_ALLOWED = 6;
	protected static final int ARRAY_LIMIT_IN_SCHEMA = 4;
	protected static final String NID_PD_PI = "PD/PI";
	protected static final String NID_CO_PD_PI = "CO-INVESTIGATOR";		
	protected static final String KEYPERSON_CO_PD_PI = "CO-PD/PI";
	
	private static final String EXTRA_KEYPERSONS = "EXTRA_KEYPERSONS";
	private static final String EQUIPMENT_NARRATIVE_TYPE_CODE = "12";

	private static final String EXTRA_KEYPERSONS_TYPE = "11";

    @Autowired
    @Qualifier("s2SBudgetInfoService")
    protected S2SBudgetInfoService s2sBudgetInfoService;

    @Autowired
    @Qualifier("s2SDateTimeService")
    protected S2SDateTimeService s2SDateTimeService;

    @Autowired
    @Qualifier("narrativeService")
    protected NarrativeService narrativeCleanupService;

    @Autowired
    @Qualifier("s2SPrintingService")
    protected S2SPrintingService s2SPrintingService;

    @Autowired
    @Qualifier("s2SCommonBudgetService")
    protected S2SCommonBudgetService s2SCommonBudgetService;

	@Autowired
	@Qualifier("s2SBudgetValidationService")
	protected S2SBudgetValidationService s2SBudgetValidationService;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/AdditionalEquipmentAttachment.xsl")
    private Resource additionalEquipmentAttachmentStyleSheet;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/ExtraKeyPersonAttachment.xsl")
    private Resource extraKeyPersonAttachmentStyleSheet;

	protected void deleteAutoGenNarratives() {
        narrativeCleanupService.deleteAutoGeneratedNarratives(pdDoc.getDevelopmentProposal().getNarratives());
	}
	
	protected NarrativeContract saveAdditionalEquipments(S2SBudgetPeriodDto periodInfo, List<S2SCostDto> extraEquipmentArrayList) {
        NarrativeContract narrative = null;
		if (extraEquipmentArrayList.size() > 0) {
			AdditionalEquipmentList additionalEquipmentList = AdditionalEquipmentList.Factory
					.newInstance();
			additionalEquipmentList.setProposalNumber(pdDoc
					.getDevelopmentProposal().getProposalNumber());
			additionalEquipmentList.setBudgetPeriod(new BigInteger(Integer
					.toString(periodInfo.getBudgetPeriod())));
			additionalEquipmentList
					.setEquipmentListArray(getEquipmentListArray(extraEquipmentArrayList));

			AdditionalEquipmentListDocument additionalEquipmentDoc = AdditionalEquipmentListDocument.Factory.newInstance();
			additionalEquipmentDoc.setAdditionalEquipmentList(additionalEquipmentList);

			String xmlData = additionalEquipmentDoc.xmlText();
			GenericPrintable printable = new GenericPrintable();
			printable.setXslTemplates(Collections.singletonList(additionalEquipmentAttachmentStyleSheet));
			printable.setName("");
			printable.setXml(xmlData);
			try {
				KcFile printData = s2SPrintingService
						.print(printable);
				final String name = pdDoc.getDevelopmentProposal().getProposalNumber() + periodInfo.getBudgetPeriod() + ADDITIONAL_EQUIPMENT;
				narrative = saveNarrative(printData.getData(), EQUIPMENT_NARRATIVE_TYPE_CODE, name + ".pdf", name);
			} catch (S2SException e) {
				LOG.error(e.getMessage(), e);
			}
		}
		return narrative;
	}

	private gov.grants.apply.coeus.additionalEquipment.AdditionalEquipmentListDocument.AdditionalEquipmentList.EquipmentList[] getEquipmentListArray(
			List<S2SCostDto> extraEquipmentArrayList) {
		List<AdditionalEquipmentList.EquipmentList> additionalEquipmentListList = new ArrayList<>();
		AdditionalEquipmentList.EquipmentList equipmentList = null;
		for (S2SCostDto costInfo : extraEquipmentArrayList) {
			equipmentList = AdditionalEquipmentList.EquipmentList.Factory
					.newInstance();
			equipmentList.setFundsRequested(costInfo.getCost()
					.bigDecimalValue());
			equipmentList
					.setEquipmentItem(costInfo.getDescription() != null ? costInfo
							.getDescription()
							: costInfo.getCategory());
			additionalEquipmentListList.add(equipmentList);
		}
		return additionalEquipmentListList
				.toArray(new gov.grants.apply.coeus.additionalEquipment.AdditionalEquipmentListDocument.AdditionalEquipmentList.EquipmentList[0]);
	}
	protected NarrativeContract saveExtraKeyPersons(S2SBudgetPeriodDto periodInfo) {
        NarrativeContract extraKPNarrative = null;
		if (periodInfo.getExtraKeyPersons() != null && !periodInfo.getExtraKeyPersons().isEmpty()) {
			ExtraKeyPersonListDocument  extraKeyPersonListDocument = ExtraKeyPersonListDocument.Factory.newInstance();
			ExtraKeyPersonList extraKeyPersonList = ExtraKeyPersonList.Factory.newInstance(); 
			extraKeyPersonList.setProposalNumber(pdDoc.getDevelopmentProposal().getProposalNumber());
			extraKeyPersonList.setBudgetPeriod(new BigInteger(""+periodInfo.getBudgetPeriod()));
			extraKeyPersonList.setKeyPersonsArray(getExtraKeyPersons(periodInfo.getExtraKeyPersons()));
			extraKeyPersonListDocument.setExtraKeyPersonList(extraKeyPersonList);
			String xmlData = extraKeyPersonListDocument.xmlText();
			GenericPrintable printable = new GenericPrintable();
			printable.setXslTemplates(Collections.singletonList(extraKeyPersonAttachmentStyleSheet));
			printable.setName("");
			printable.setXml(xmlData);
			try {
				KcFile printData = s2SPrintingService.print(printable);
				final String name = pdDoc.getDevelopmentProposal().getProposalNumber() + periodInfo.getBudgetPeriod() + "_" + EXTRA_KEYPERSONS;
				extraKPNarrative = saveNarrative(printData.getData(), EXTRA_KEYPERSONS_TYPE, name + ".pdf", name);
			} catch (S2SException e) {
				LOG.error(e.getMessage(), e);
			}
		}
		return extraKPNarrative;
	}
	private gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons[] getExtraKeyPersons(List<S2SKeyPersonDto> keyPersonList) {
		List<gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons> keypersonslist = new ArrayList<>();
		for(S2SKeyPersonDto keyPersonInfo : keyPersonList){
			gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons keyPerson = gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons.Factory.newInstance();
			keyPerson.setFirstName(keyPersonInfo.getFirstName());
			keyPerson.setMiddleName(keyPersonInfo.getMiddleName());
			keyPerson.setLastName(keyPersonInfo.getLastName());
			keyPerson.setProjectRole(keyPersonInfo.getKeyPersonRole());
			keyPerson.setCompensation(getExtraKeyPersonCompensation(keyPersonInfo));
			keypersonslist.add(keyPerson);
		}
		return keypersonslist.toArray(new gov.grants.apply.coeus.extraKeyPerson.ExtraKeyPersonListDocument.ExtraKeyPersonList.KeyPersons[0]);
	}
	private Compensation getExtraKeyPersonCompensation(
			S2SKeyPersonDto keyPersonInfo) {
		Compensation compensation = Compensation.Factory.newInstance();
		if(keyPersonInfo.getCompensation().getAcademicMonths() != null){
			compensation.setAcademicMonths(keyPersonInfo.getCompensation().getAcademicMonths().bigDecimalValue());
		}
		if(keyPersonInfo.getCompensation().getBaseSalary() != null){
			compensation.setBaseSalary(keyPersonInfo.getCompensation().getBaseSalary().bigDecimalValue());
		}
		if(keyPersonInfo.getCompensation().getCalendarMonths() != null){
			compensation.setCalendarMonths(keyPersonInfo.getCompensation().getCalendarMonths().bigDecimalValue());
		}
		if(keyPersonInfo.getCompensation().getFringe() != null){
			compensation.setFringeBenefits(keyPersonInfo.getCompensation().getFringe().bigDecimalValue());
		}
		if(keyPersonInfo.getCompensation().getFundsRequested() != null){
			compensation.setFundsRequested(keyPersonInfo.getCompensation().getFundsRequested().bigDecimalValue());
		}
		if(keyPersonInfo.getCompensation().getRequestedSalary() != null){
			compensation.setRequestedSalary(keyPersonInfo.getCompensation().getRequestedSalary().bigDecimalValue());
		}
		if(keyPersonInfo.getCompensation().getSummerMonths() != null){
			compensation.setSummerMonths(keyPersonInfo.getCompensation().getSummerMonths().bigDecimalValue());
		}
		return compensation;
	}
   protected boolean isProposalPersonEqualsKeyPerson(ProposalPersonContract proposalPerson, S2SKeyPersonDto keyPerson) {
        if(keyPerson.getPersonId()!=null){
            return keyPerson.getPersonId().equals(proposalPerson.getPersonId());
        }else if(keyPerson.getRolodexId()!=null){
            return keyPerson.getRolodexId().equals(proposalPerson.getRolodexId());
        }
        return false;
    }

   /**
    * This method check whether the key person has a personnel budget  
    * 
    * @param keyPerson
    *            (KeyPersonInfo) key person entry.
    * @param period
    *            budget period
    * @return true if key person has personnel budget else false.
    */
   protected Boolean hasPersonnelBudget(S2SKeyPersonDto keyPerson, int period){
       final List<? extends BudgetLineItemContract> budgetLineItemList;

       ProposalDevelopmentBudgetExtContract budget = s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());

        budgetLineItemList = budget.getBudgetPeriods().get(period-1).getBudgetLineItems();

        for(BudgetLineItemContract budgetLineItem : budgetLineItemList) {
            for(BudgetPersonnelDetailsContract budgetPersonnelDetails : budgetLineItem.getBudgetPersonnelDetailsList()){
                if( budgetPersonnelDetails.getPersonId().equals(keyPerson.getPersonId())){
                    return true;
                } else if (keyPerson.getRolodexId() != null && budgetPersonnelDetails.getPersonId().equals(keyPerson.getRolodexId().toString())) {
                    return true;
                }
            }
        }
       return false;       
   }
   
   /**
    * Perform manual validations on the budget. Similarly done in RRFedNonFedBudgetBaseGenerator due to object graph.
    */
   protected boolean validateBudgetForForm(ProposalDevelopmentDocumentContract pdDoc) {
	   final List<AuditError> errors = s2SBudgetValidationService.validateBudgetForForm(pdDoc, getFormName());
	   getAuditErrors().addAll(errors);
	   return errors.isEmpty();
   }

	public S2SBudgetInfoService getS2sBudgetInfoService() {
		return s2sBudgetInfoService;
	}

	public void setS2sBudgetInfoService(S2SBudgetInfoService s2sBudgetInfoService) {
		this.s2sBudgetInfoService = s2sBudgetInfoService;
	}

	public S2SBudgetValidationService getS2SBudgetValidationService() {
		return s2SBudgetValidationService;
	}

	public void setS2SBudgetValidationService(S2SBudgetValidationService s2SBudgetValidationService) {
		this.s2SBudgetValidationService = s2SBudgetValidationService;
	}

	public NarrativeService getNarrativeCleanupService() {
        return narrativeCleanupService;
    }

    public void setNarrativeCleanupService(NarrativeService narrativeCleanupService) {
        this.narrativeCleanupService = narrativeCleanupService;
    }

    public S2SPrintingService getS2SPrintingService() {
        return s2SPrintingService;
    }

    public void setS2SPrintingService(S2SPrintingService s2SPrintingService) {
        this.s2SPrintingService = s2SPrintingService;
    }

	public S2SDateTimeService getS2SDateTimeService() {
		return s2SDateTimeService;
	}

	public void setS2SDateTimeService(S2SDateTimeService s2SDateTimeService) {
		this.s2SDateTimeService = s2SDateTimeService;
	}

    public S2SCommonBudgetService getS2SCommonBudgetService() {
        return s2SCommonBudgetService;
    }

    public void setS2SCommonBudgetService(S2SCommonBudgetService s2SCommonBudgetService) {
        this.s2SCommonBudgetService = s2SCommonBudgetService;
    }

    public Resource getAdditionalEquipmentAttachmentStyleSheet() {
        return additionalEquipmentAttachmentStyleSheet;
    }

    public void setAdditionalEquipmentAttachmentStyleSheet(Resource additionalEquipmentAttachmentStyleSheet) {
        this.additionalEquipmentAttachmentStyleSheet = additionalEquipmentAttachmentStyleSheet;
    }

    public Resource getExtraKeyPersonAttachmentStyleSheet() {
        return extraKeyPersonAttachmentStyleSheet;
    }

    public void setExtraKeyPersonAttachmentStyleSheet(Resource extraKeyPersonAttachmentStyleSheet) {
        this.extraKeyPersonAttachmentStyleSheet = extraKeyPersonAttachmentStyleSheet;
    }
}
