/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import com.lowagie.text.pdf.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.s2s.S2sUserAttachedFormAttContract;
import org.kuali.coeus.propdev.api.s2s.S2sUserAttachedFormContract;
import org.kuali.coeus.propdev.api.s2s.S2sUserAttachedFormFileContract;
import org.kuali.coeus.s2sgen.api.core.AuditError;
import org.kuali.coeus.s2sgen.api.core.InfrastructureConstants;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.api.hash.GrantApplicationHashService;
import org.kuali.coeus.s2sgen.impl.generate.*;
import org.springframework.beans.factory.BeanNameAware;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.Resource;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.*;

@FormGenerator("UserAttachedFormGenerator")
public class UserAttachedFormGenerator implements S2SFormGenerator<XmlObject>, S2SFormGeneratorPdfFillable<XmlObject>, DynamicForm, BeanNameAware, Named {

    public static final String META_GRANT_APPLICATION_NS = "http://apply.grants.gov/system/MetaGrantApplication";
    public static final String FORMS = "Forms";

    private String beanName;

    private List<AuditError> auditErrors = new ArrayList<>();
    private List<AttachmentData> attachments = new ArrayList<>();

    private S2sUserAttachedFormContract s2sUserAttachedForm;

    private Optional<S2SFormGenerator<?>> supportedS2SFormGenerator;

    @Autowired
    @Qualifier("grantApplicationHashService")
    private GrantApplicationHashService grantApplicationHashService;

    @Override
    public XmlObject getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {

        final S2sUserAttachedFormFileContract userAttachedFormFile = getS2sUserAttachedForm().getS2sUserAttachedFormFileList().get(0);
        if (userAttachedFormFile == null) {
            throw new RuntimeException("Cannot find XML Data");
        }
        final String formXml = userAttachedFormFile.getXmlFile();
        final XmlObject xmlObject;
        try {
            xmlObject = XmlObject.Factory.parse(formXml);
        }catch (XmlException e) {
            throw new RuntimeException("XmlObject not ready");
        }

        s2sUserAttachedForm.getS2sUserAttachedFormAtts().forEach(this::addAttachment);

        return xmlObject;
    }

    protected void addAttachment(AttachmentData attachment) {
        attachments.add(attachment);
    }

    private void addAttachment(S2sUserAttachedFormAttContract s2sUserAttachedFormAtt) {
        final String hash = getGrantApplicationHashService().computeAttachmentHash(s2sUserAttachedFormAtt.getData());
        addAttachment(new AttachmentData(s2sUserAttachedFormAtt.getFileDataId(), s2sUserAttachedFormAtt.getContentId(), s2sUserAttachedFormAtt.getContentId(), s2sUserAttachedFormAtt.getData(), s2sUserAttachedFormAtt.getType(), InfrastructureConstants.HASH_ALGORITHM, hash, s2sUserAttachedFormAtt.getUploadUser(), s2sUserAttachedFormAtt.getUploadTimestamp()));
    }

    @Override
    public List<AuditError> getAuditErrors() {
        return auditErrors;
    }

    public void setAuditErrors(List<AuditError> auditErrors) {
        this.auditErrors = auditErrors;
    }

    @Override
    public List<AttachmentData> getAttachments() {
        return attachments;
    }

    public void setAttachments(List<AttachmentData> attachments) {
        this.attachments = attachments;
    }

    public GrantApplicationHashService getGrantApplicationHashService() {
        return grantApplicationHashService;
    }

    public void setGrantApplicationHashService(GrantApplicationHashService grantApplicationHashService) {
        this.grantApplicationHashService = grantApplicationHashService;
    }

    @Override
    public String getNamespace() {
        return getS2sUserAttachedForm().getNamespace();
    }

    @Override
    public String getFormName() {
        return getS2sUserAttachedForm().getFormName();
    }

    @Override
    public int getSortIndex() {
        return supportedS2SFormGenerator.map(S2SFormGenerator::getSortIndex).orElse(Integer.MAX_VALUE);
    }

    @Override
    public List<Resource> getStylesheets() {
        return supportedS2SFormGenerator.map(S2SFormGenerator::getStylesheets).orElse(Collections.emptyList());
    }

    @Override
    public boolean supportsPdfFilling() {
        return true;
    }

    @Override
    public boolean supportsXslTransform() {
        return supportedS2SFormGenerator.map(S2SFormGenerator::supportsXslTransform).orElse(false);
    }

    @Override
    public Resource getPdfForm() {
        if (supportedS2SFormGenerator.isPresent()) {
            final S2SFormGenerator<?> generator = supportedS2SFormGenerator.get();
            if (generator instanceof S2SFormGeneratorPdfFillable<?>) {
                return ((S2SFormGeneratorPdfFillable<?>) generator).getPdfForm();
            }
        }

        try (final ByteArrayOutputStream baos = new ByteArrayOutputStream();
             final PdfReader reader = new PdfReader(getS2sUserAttachedFormFile().getFormFile());
             final PdfStamper stamper = new PdfStamper(reader, baos)) {
            clearAttachments(reader);
            clearXfaFormData(reader);
            stamper.close();
            return new ByteArrayResource(baos.toByteArray());
        } catch (IOException e) {
            throw new RuntimeException("Cannot read PDF");
        }
    }

    private void clearXfaFormData(PdfReader reader) {
        final XfaForm xfaForm = reader.getAcroFields().getXfa();
        if (xfaForm.isXfaPresent()) {
            org.w3c.dom.Document xfaDocument = xfaForm.getDomDocument();
            if (xfaDocument != null) {
                final NodeList formsNode = xfaDocument.getElementsByTagNameNS(META_GRANT_APPLICATION_NS, FORMS);
                if (formsNode != null) {
                    final Node firstForm = formsNode.item(0);
                    if (firstForm != null) {
                        xfaForm.setNodeText(firstForm, "");
                    }
                }
            }
        }
    }

    private void clearAttachments(PdfReader reader) {
        final PdfDictionary catalog = reader.getCatalog();
        final PdfDictionary names = (PdfDictionary) PdfReader.getPdfObject(catalog.get(PdfName.NAMES));
        if (names != null) {
            final List<? extends S2sUserAttachedFormAttContract> atts = getS2sUserAttachedForm().getS2sUserAttachedFormAtts();
            if (atts != null && !atts.isEmpty()) {
                final PdfDictionary embFiles = (PdfDictionary) PdfReader.getPdfObject(names.get(PdfName.EMBEDDEDFILES));
                if (embFiles != null) {
                    removeMatchingAttachments(atts, embFiles);
                }
            }
        }
    }

    private void removeMatchingAttachments(final List<? extends S2sUserAttachedFormAttContract> atts, final PdfDictionary embFiles) {
        final PdfArray efNames = embFiles.getAsArray(PdfName.NAMES);
        // Cannot use the list to remove since it is a copy
        final ListIterator<PdfObject> iterator = efNames.listIterator();
        while (iterator.hasNext()) {
            final PdfObject key = iterator.next();
            if (iterator.hasNext()) {
                final PdfDictionary file = (PdfDictionary) PdfReader.getPdfObject(iterator.next());
                final String fileName = getFilename(file);
                if (atts.stream()
                        .map(S2sUserAttachedFormAttContract::getName)
                        .anyMatch(name -> name.equals(fileName))) {
                    //remove the file
                    iterator.remove();
                    //go back to the key
                    iterator.previous();
                    //remove the key
                    iterator.remove();
                }
            }
        }
    }

    @Override
    public Attachments getMappedAttachments(XmlObject form, List<AttachmentData> attachments) {
        try (final PdfReader reader = new PdfReader(getS2sUserAttachedFormFile().getFormFile())) {

            final PdfDictionary catalog = reader.getCatalog();

            final PdfDictionary names = (PdfDictionary) PdfReader.getPdfObject(catalog.get(PdfName.NAMES));
            if (names != null) {
                final PdfDictionary embFiles = (PdfDictionary) PdfReader.getPdfObject(names.get(PdfName.EMBEDDEDFILES));
                if (embFiles != null) {
                    final Map<String, PdfObject> embMap = readTree(embFiles);
                    final Map<String, String> embeddedFiles = embMap.entrySet().stream()
                            .map(o -> entry(o.getKey(), (PdfDictionary) PdfReader.getPdfObject(o.getValue())))
                            .filter(e -> Objects.nonNull(e.getValue()))
                            .map(e -> entry(getFilename(e.getValue()), e.getKey()))
                            .filter(e -> Objects.nonNull(e.getValue()))
                            .collect(nullSafeEntriesToMap());

                    final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = getS2sUserAttachedForm().getS2sUserAttachedFormAtts().stream()
                            .map( a -> {
                                final String hash = getGrantApplicationHashService().computeAttachmentHash(a.getData());
                                return new AttachmentData(a.getFileDataId(), a.getName(), a.getContentId(), a.getData(), a.getType(), InfrastructureConstants.HASH_ALGORITHM, hash, a.getUploadUser(), a.getUploadTimestamp());
                            })
                            .map(a -> entry(embeddedFiles.get(a.getFileName()), a))
                            .collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

                    return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                            attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
                }

                return new Attachments(Collections.emptyMap(), attachments);
            }
        } catch (IOException e) {
            throw new RuntimeException("Cannot read PDF");
        }

        return new Attachments(Collections.emptyMap(), attachments);
    }

    private String getFilename(PdfDictionary filespec) {
        final PdfString fn = (PdfString) PdfReader.getPdfObject(filespec.get(PdfName.F));
        if (fn == null) {
            return null;
        } else {
            return fn.toUnicodeString();
        }
    }

    @Override
    public String getBeanName() {
        return beanName;
    }

    @Override
    public void setBeanName(String beanName) {
        this.beanName = beanName;
    }

    @Override
    public DocumentFactory<XmlObject> factory() {
        return XmlObject.Factory;
    }

    @Override
    public S2sUserAttachedFormContract getS2sUserAttachedForm() {
        return s2sUserAttachedForm;
    }

    @Override
    public void setS2sUserAttachedForm(S2sUserAttachedFormContract s2sUserAttachedForm) {
        this.s2sUserAttachedForm = s2sUserAttachedForm;
    }

    public S2sUserAttachedFormFileContract getS2sUserAttachedFormFile() {
        return getS2sUserAttachedForm().getS2sUserAttachedFormFileList().get(0);
    }

    @Override
    public Optional<S2SFormGenerator<?>> getSupportedS2SFormGenerator() {
        return supportedS2SFormGenerator;
    }

    @Override
    public void setSupportedS2SFormGenerator(Optional<S2SFormGenerator<?>> supportedS2SFormGenerator) {
        this.supportedS2SFormGenerator = supportedS2SFormGenerator;
    }


    private static void iterateItems(PdfDictionary dic, Map<String, PdfObject> items) {
        PdfArray nn = (PdfArray) PdfReader.getPdfObjectRelease(dic.get(PdfName.NAMES));
        if (nn != null) {
            for (int k = 0; k < nn.size(); ++k) {
                final PdfString s = (PdfString) PdfReader.getPdfObjectRelease(nn.getPdfObject(k++));
                items.put(s.toUnicodeString(), nn.getPdfObject(k));
            }
        } else if ((nn = (PdfArray) PdfReader.getPdfObjectRelease(dic.get(PdfName.KIDS))) != null) {
            for (int k = 0; k < nn.size(); ++k) {
                final PdfDictionary kid = (PdfDictionary) PdfReader.getPdfObjectRelease(nn.getPdfObject(k));
                iterateItems(kid, items);
            }
        }
    }

    /**
     * This method is Similar to {@link com.lowagie.text.pdf.PdfNameTree#readTree}
     * except that the PdfString is converted to a unicodeString
     */
    private static Map<String, PdfObject> readTree(PdfDictionary dic) {
        final Map<String, PdfObject> items = new HashMap<>();
        if (dic != null) {
            iterateItems(dic, items);
        }
        return items;
    }
}
