/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package org.kuali.coeus.s2sgen.impl.print;

import gov.grants.apply.system.metaGrantApplication.GrantApplicationDocument;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.xmlbeans.XmlCursor;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.propdev.api.attachment.NarrativeService;
import org.kuali.coeus.propdev.api.core.DevelopmentProposalContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.s2s.S2sAppSubmissionContract;
import org.kuali.coeus.propdev.api.s2s.override.S2sOverrideContract;
import org.kuali.coeus.s2sgen.api.core.AuditError;
import org.kuali.coeus.s2sgen.api.core.InfrastructureConstants;
import org.kuali.coeus.s2sgen.api.core.S2SException;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.api.print.FormElements;
import org.kuali.coeus.s2sgen.api.print.FormPackage;
import org.kuali.coeus.s2sgen.api.print.FormPackageService;
import org.kuali.coeus.s2sgen.impl.datetime.S2SDateTimeService;
import org.kuali.coeus.s2sgen.impl.generate.FormApplicationService;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorRetrievalService;
import org.kuali.coeus.s2sgen.impl.util.XPathExecutor;
import org.kuali.coeus.s2sgen.impl.validate.S2SValidatorService;
import org.kuali.coeus.s2sgen.impl.validate.ValidationResult;
import org.kuali.coeus.sys.api.model.KcFile;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.MimeTypeUtils;
import org.w3c.dom.Node;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;

@Component("formPackageService")
public class FormPackageServiceImpl implements FormPackageService {

    private static final Logger LOG = LogManager.getLogger(FormPackageServiceImpl.class);

    @Autowired
    @Qualifier("s2SDateTimeService")
    private S2SDateTimeService s2SDateTimeService;

    @Autowired
    @Qualifier("narrativeService")
    private NarrativeService narrativeService;

    @Autowired
    @Qualifier("s2SFormGeneratorRetrievalService")
    private S2SFormGeneratorRetrievalService s2SFormGeneratorRetrievalService;

    @Autowired
    @Qualifier("s2SValidatorService")
    private S2SValidatorService s2SValidatorService;

    @Autowired
    @Qualifier("formApplicationService")
    private FormApplicationService formApplicationService;

    @Autowired
    private ResourceLoader resourceLoader;

    @Override
    public FormPackage retrieveFormPackage(Set<String> namespaces, ProposalDevelopmentDocumentContract pdDoc) {
        if (namespaces == null) {
            throw new IllegalArgumentException("namespaces is null");
        }

        if (pdDoc == null) {
            throw new IllegalArgumentException("pdDoc is null");
        }

        if (namespaces.isEmpty()) {
            final FormPackage formPackage = new FormPackage();
            final GrantApplicationDocument.GrantApplication.Forms forms = GrantApplicationDocument.GrantApplication.Forms.Factory.newInstance();
            final String grantApplication = getFormApplicationService().getGrantApplicationDocument(pdDoc, forms);
            formPackage.setXmlFile(new S2SFile("Grant Application.xml", "text/xml", grantApplication.getBytes(StandardCharsets.UTF_8)));
            return formPackage;
        }

        final DevelopmentProposalContract developmentProposal = pdDoc.getDevelopmentProposal();

        final S2sAppSubmissionContract s2sAppSubmission = getLatestS2SAppSubmission(developmentProposal.getS2sAppSubmission());
        final boolean submitted = s2sAppSubmission != null && s2sAppSubmission.getGgTrackingId() != null;

        final S2sOverrideContract s2sOverride = developmentProposal.getS2sOverride();
        final boolean overridden = s2sOverride != null && s2sOverride.isActive();

        final boolean requiresValidation = !submitted;

        final List<FormElements> forms = retrieveFormElements(namespaces, pdDoc);

        final FormPackage formPackage = new FormPackage();

        final String grantApplication = createGrantApplication(pdDoc, s2sAppSubmission, submitted, s2sOverride, overridden, forms.stream().map(FormElements::getXmlFile).collect(Collectors.toList()));
        formPackage.setXmlFile(new S2SFile("Grant Application.xml", "text/xml", grantApplication.getBytes(StandardCharsets.UTF_8)));
        if (requiresValidation) {
            final ValidationResult result = s2SValidatorService.validateApplication(grantApplication, resourceLoader.getResource(developmentProposal.getS2sOpportunity().getSchemaUrl()));
            formPackage.setErrors(result.getErrors());
        }

        formPackage.setFormElements(forms);
        return formPackage;
    }

    @Override
    public List<FormElements> retrieveFormElements(Set<String> namespaces, ProposalDevelopmentDocumentContract pdDoc) {
        if (namespaces == null) {
            throw new IllegalArgumentException("namespaces is null");
        }

        if (pdDoc == null) {
            throw new IllegalArgumentException("pdDoc is null");
        }

        if (namespaces.isEmpty()) {
            return Collections.emptyList();
        }

        final DevelopmentProposalContract developmentProposal = pdDoc.getDevelopmentProposal();
        final String proposalNumber = developmentProposal.getProposalNumber();

        final S2sAppSubmissionContract s2sAppSubmission = getLatestS2SAppSubmission(developmentProposal.getS2sAppSubmission());
        final boolean submitted = s2sAppSubmission != null && s2sAppSubmission.getGgTrackingId() != null;

        final S2sOverrideContract s2sOverride = developmentProposal.getS2sOverride();
        final boolean overridden = s2sOverride != null && s2sOverride.isActive();

        final boolean requiresValidation = !submitted;
        final boolean requiresNarrativeCleanup = !submitted && !overridden;
        final boolean requiresXmlModification = !submitted && !overridden;
        /*
            Ideally we have one code path and different kinds of generators:

            1. PD document --> XmlObject Generator (the most common)
            2. UserAttachedForm --> XmlObject Generator
            3. S2S Override --> XmlObject Generator
            4. Submitted Application --> XmlObject Generator

            Instead, we have generator types 1 & 2 and special code paths for types 3 & 4.
            It is out of scope to transform the entire codebase into this model at this time but when it comes to producing
            form elements it should be mostly the same regardless of the source of the xml & attachment information.
        */

        if (!requiresNarrativeCleanup) {
            //delete internal narratives because generators recreate them even when just printing
            getNarrativeService().deleteSystemGeneratedNarratives(pdDoc.getDevelopmentProposal().getNarratives());
        }

        return s2SFormGeneratorRetrievalService.getS2SGenerators(namespaces, proposalNumber).stream()
                .map(generator -> determineGenerator(s2sAppSubmission, submitted, s2sOverride, overridden, generator))
                .map(generator -> {
                    final FormElements formElements = new FormElements();
                    formElements.setNamespace(generator.getNamespace());

                    final List<AuditError> errors = new ArrayList<>(generator.getAuditErrors());

                    final XmlObject formObject = generator.getFormObject(pdDoc);
                    if (formObject != null) {
                        if (requiresValidation) {
                            final ValidationResult result = s2SValidatorService.validateForm(formObject, getFormNameFromNamespace(generator.getNamespace()));
                            errors.addAll(result.getErrors());
                        }

                        final String applicationXml = formObject.xmlText(s2SFormGeneratorRetrievalService.getXmlOptionsPrefixes());
                        final String filteredApplicationXml = requiresXmlModification ? s2SDateTimeService.removeTimezoneFactor(applicationXml) : applicationXml;
                        formElements.setXmlFile(getXmlFile(generator.getNamespace(), filteredApplicationXml));

                        final List<AttachmentData> attachments = generator.getAttachments();
                        if (generator.supportsPdfFilling()) {
                            formElements.setPdfElements(getPdfElements(generator.getNamespace(), (S2SFormGeneratorPdfFillable<XmlObject>) generator, formObject, attachments));
                        }

                        if (generator.supportsXslTransform()) {
                            formElements.setStylesheetElements(getStylesheetElements(generator, attachments));
                        }
                    }

                    formElements.setErrors(errors);

                    return formElements;
                }).collect(Collectors.toList());
    }

    private String createGrantApplication(ProposalDevelopmentDocumentContract pdDoc, S2sAppSubmissionContract s2sAppSubmission, boolean submitted, S2sOverrideContract s2sOverride, boolean overridden, List<KcFile> formFiles) {
        if (submitted) {
            return s2sAppSubmission.getS2sApplication().getApplication();
        } else {
            if (overridden) {
                return s2sOverride.getApplicationOverride().getApplication();
            } else {
                final GrantApplicationDocument.GrantApplication.Forms forms = GrantApplicationDocument.GrantApplication.Forms.Factory.newInstance();
                formFiles.stream().map(file -> {
                    try {
                        return XmlObject.Factory.parse(new String(file.getData(), StandardCharsets.UTF_8));
                    } catch (XmlException e) {
                        throw new RuntimeException(e);
                    }
                }).forEach(formObject -> setFormObject(forms, formObject));
                return getFormApplicationService().getGrantApplicationDocument(pdDoc, forms);
            }
        }
    }

    private void setFormObject(GrantApplicationDocument.GrantApplication.Forms forms, XmlObject formObject) {
        // Create a cursor from the grants.gov form
        XmlCursor formCursor = formObject.newCursor();
        formCursor.toStartDoc();
        formCursor.toNextToken();

        // Create a cursor from the Forms object
        XmlCursor metaGrantCursor = forms.newCursor();
        metaGrantCursor.toNextToken();

        // Add the form to the Forms object.
        formCursor.moveXml(metaGrantCursor);
    }

    private S2SFormGenerator<XmlObject> determineGenerator(S2sAppSubmissionContract s2sAppSubmission, boolean submitted, S2sOverrideContract s2sOverride, boolean overridden, S2SFormGenerator<XmlObject> generator) {
        if (submitted) {
            if (generator instanceof S2SFormGeneratorPdfFillable<?>) {
                return new SubmittedPdfFillableFormGeneratorAdapter<>(generator, s2sAppSubmission);
            } else {
                return new SubmittedFormGeneratorAdapter<>(generator, s2sAppSubmission);
            }
        } else {
            if (overridden) {
                if (generator instanceof S2SFormGeneratorPdfFillable<?>) {
                    return new OverriddenPdfFillableFormGeneratorAdapter<>(generator, s2sOverride);
                } else {
                    return new OverriddenFormGeneratorAdapter<>(generator, s2sOverride);
                }
            } else {
                return generator;
            }
        }
    }

    private <T extends XmlObject> FormElements.PdfElements getPdfElements(String namespace, S2SFormGeneratorPdfFillable<T> generator, T xmlObject, List<AttachmentData> attachments) {
        final FormElements.PdfElements pdfElements = new FormElements.PdfElements();
        final S2SFormGeneratorPdfFillable.Attachments mappedAttachments = generator.getMappedAttachments(xmlObject, attachments);
        if (!CollectionUtils.isEmpty(mappedAttachments.getUnmapped())) {
            LOG.warn("Unmapped attachments found " + mappedAttachments.getUnmapped().stream().map(AttachmentData::getFileName).toList());
        }
        pdfElements.setPdfFile(getPdfFile(namespace, generator.getPdfForm()));
        pdfElements.setAttachments(mappedAttachments.getMapped());

        return pdfElements;
    }

    private <T extends XmlObject> FormElements.StylesheetElements getStylesheetElements(S2SFormGenerator<T> generator, List<AttachmentData> attachments) {
        final FormElements.StylesheetElements stylesheetElements = new FormElements.StylesheetElements();
        stylesheetElements.setStylesheets(generator.getStylesheets().stream().map(this::getStylesheetFile).collect(Collectors.toList()));
        stylesheetElements.setAttachments(attachments);

        return stylesheetElements;
    }

    private String getFormNameFromNamespace(String namespace) {
        return namespace.substring(namespace.lastIndexOf('/') + 1);
    }

    private S2SFile getXmlFile(String namespace, String xml) {
        final S2SFile xmlFile = new S2SFile();
        xmlFile.setData(xml.getBytes());
        xmlFile.setName(getFormNameFromNamespace(namespace) + ".xml");
        xmlFile.setType(MimeTypeUtils.TEXT_XML_VALUE);

        return xmlFile;
    }

    private S2SFile getPdfFile(String namespace, Resource pdfResource) {

        try(InputStream inputStream = pdfResource.getInputStream()) {
            final S2SFile pdfFile = new S2SFile();
            pdfFile.setData(inputStream.readAllBytes());
            pdfFile.setName(getFormNameFromNamespace(namespace) + ".pdf");
            pdfFile.setType("application/pdf");
            return pdfFile;
        } catch (IOException e) {
            throw new S2SException(e);
        }
    }

    private S2SFile getStylesheetFile(Resource stylesheetResource) {

        try(InputStream inputStream = stylesheetResource.getInputStream())  {
            final S2SFile pdfFile = new S2SFile();
            pdfFile.setData(inputStream.readAllBytes());
            pdfFile.setName(stylesheetResource.getFilename());
            pdfFile.setType("text/xsl");
            return pdfFile;
        } catch (IOException e) {
            throw new S2SException(e);
        }
    }

    /**
     *
     * This method gets the latest S2sAppSubmission record from the list of
     * S2sAppSubmissions. It iterates through the list and returns the record
     * that has the highest submission number.
     */
    protected S2sAppSubmissionContract getLatestS2SAppSubmission(List<? extends S2sAppSubmissionContract> submissions) {
        return submissions.stream().max(Comparator.comparing(S2sAppSubmissionContract::getSubmissionNumber)).orElse(null);
    }

    public S2SDateTimeService getS2SDateTimeService() {
        return s2SDateTimeService;
    }

    public void setS2SDateTimeService(S2SDateTimeService s2SDateTimeService) {
        this.s2SDateTimeService = s2SDateTimeService;
    }

    public NarrativeService getNarrativeService() {
        return narrativeService;
    }

    public void setNarrativeService(NarrativeService narrativeService) {
        this.narrativeService = narrativeService;
    }

    public S2SFormGeneratorRetrievalService getS2SFormGeneratorRetrievalService() {
        return s2SFormGeneratorRetrievalService;
    }

    public void setS2SFormGeneratorRetrievalService(S2SFormGeneratorRetrievalService s2SFormGeneratorRetrievalService) {
        this.s2SFormGeneratorRetrievalService = s2SFormGeneratorRetrievalService;
    }

    public S2SValidatorService getS2SValidatorService() {
        return s2SValidatorService;
    }

    public void setS2SValidatorService(S2SValidatorService s2SValidatorService) {
        this.s2SValidatorService = s2SValidatorService;
    }

    public FormApplicationService getFormApplicationService() {
        return formApplicationService;
    }

    public void setFormApplicationService(FormApplicationService formApplicationService) {
        this.formApplicationService = formApplicationService;
    }

    public ResourceLoader getResourceLoader() {
        return resourceLoader;
    }

    public void setResourceLoader(ResourceLoader resourceLoader) {
        this.resourceLoader = resourceLoader;
    }

    private static class SubmittedPdfFillableFormGeneratorAdapter<T extends XmlObject> extends SubmittedFormGeneratorAdapter<T> implements S2SFormGeneratorPdfFillable<T> {

        private SubmittedPdfFillableFormGeneratorAdapter(S2SFormGenerator<T> generator, S2sAppSubmissionContract s2sAppSubmission) {
            super(generator, s2sAppSubmission);
        }

        @Override
        public String getBeanName() {
            return SubmittedPdfFillableFormGeneratorAdapter.class.getName();
        }

        @Override
        public Resource getPdfForm() {
            return ((S2SFormGeneratorPdfFillable<T>)getGenerator()).getPdfForm() ;
        }

        @Override
        public Attachments getMappedAttachments(T form, List<AttachmentData> attachments) {
            return ((S2SFormGeneratorPdfFillable<T>)getGenerator()).getMappedAttachments(form, attachments);
        }
    }

    private static class SubmittedFormGeneratorAdapter<T extends XmlObject> implements S2SFormGenerator<T> {

        private final S2SFormGenerator<T> generator;
        private T xmlObject;
        private final List<AttachmentData> attachments;
        private final List<AuditError> auditErrors = new ArrayList<>();

        private SubmittedFormGeneratorAdapter(S2SFormGenerator<T> generator, S2sAppSubmissionContract s2sAppSubmission) {
            this.generator = generator;

            attachments = s2sAppSubmission.getS2sApplication().getS2sAppAttachmentList().stream().map(a -> new AttachmentData(a.getFileDataId(), a.getName(), a.getContentId(), a.getData(), a.getContentType(), InfrastructureConstants.HASH_ALGORITHM, a.getSha1Hash(), a.getUploadUser(), a.getUploadTimestamp())).toList();
            try {
                final XmlObject formsObject = getForms(getApplication(s2sAppSubmission));
                final String formsXml = formsObject.toString();
                final String formsXpath = formsXml.startsWith("<xml-fragment") ?
                        "/xml-fragment/*[namespace-uri(.) = '" + generator.getNamespace() + "']" :
                        "/*[namespace-uri(.) = '" + generator.getNamespace() + "']";

                final XPathExecutor executer = new XPathExecutor(formsXml);
                final Node d = executer.getNode(formsXpath);
                xmlObject = factory().parse(d);
            } catch (XmlException e) {
                LOG.error(e.getMessage(), e);
                auditErrors.add(new AuditError(AuditError.NO_FIELD_ERROR_KEY, "error.proposalDevelopment.grants.gov.form.not.found", AuditError.GG_LINK, getFormName()));
            }
        }

        @Override
        public DocumentFactory<T> factory() {
            return generator.factory();
        }

        @Override
        public String getBeanName() {
            return SubmittedFormGeneratorAdapter.class.getName();
        }

        @Override
        public String getNamespace() {
            return generator.getNamespace();
        }

        @Override
        public String getFormName() {
            return generator.getFormName();
        }

        @Override
        public int getSortIndex() {
            return generator.getSortIndex();
        }

        @Override
        public List<Resource> getStylesheets() {
            return generator.getStylesheets();
        }

        @Override
        public boolean supportsPdfFilling() {
            return generator.supportsPdfFilling();
        }

        @Override
        public boolean supportsXslTransform() {
            return generator.supportsXslTransform();
        }

        @Override
        public T getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) throws S2SException {
            return xmlObject;
        }

        @Override
        public List<AttachmentData> getAttachments() {
            return attachments;
        }

        @Override
        public List<AuditError> getAuditErrors() {
            return auditErrors;
        }

        private GrantApplicationDocument getApplication(S2sAppSubmissionContract s2sAppSubmission) {
            final String submittedApplicationXml = s2sAppSubmission.getS2sApplication().getApplication();
            try {
                return GrantApplicationDocument.Factory.parse(submittedApplicationXml);
            } catch (XmlException e) {
                throw new S2SException(e);
            }
        }

        private XmlObject getForms(GrantApplicationDocument grantApplicationDocument) {
            try (XmlCursor cursor = grantApplicationDocument.getGrantApplication().getForms().newCursor()) {
                return cursor.getObject();
            }
        }

        public S2SFormGenerator<T> getGenerator() {
            return generator;
        }
    }

    private static class OverriddenPdfFillableFormGeneratorAdapter<T extends XmlObject> extends OverriddenFormGeneratorAdapter<T> implements S2SFormGeneratorPdfFillable<T> {

        private OverriddenPdfFillableFormGeneratorAdapter(S2SFormGenerator<T> generator, S2sOverrideContract s2sOverride) {
            super(generator, s2sOverride);
            if (!s2sOverride.isActive()) {
                throw new IllegalStateException("S2sOverride is not active");
            }
        }

        @Override
        public String getBeanName() {
            return OverriddenPdfFillableFormGeneratorAdapter.class.getName();
        }

        @Override
        public Resource getPdfForm() {
            return ((S2SFormGeneratorPdfFillable<T>)getGenerator()).getPdfForm() ;
        }

        @Override
        public Attachments getMappedAttachments(T form, List<AttachmentData> attachments) {
            return ((S2SFormGeneratorPdfFillable<T>)getGenerator()).getMappedAttachments(form, attachments);
        }
    }

    private static class OverriddenFormGeneratorAdapter<T extends XmlObject> implements S2SFormGenerator<T> {

        private final S2SFormGenerator<T> generator;
        private T xmlObject;
        private final List<AttachmentData> attachments;
        private final List<AuditError> auditErrors = new ArrayList<>();

        private OverriddenFormGeneratorAdapter(S2SFormGenerator<T> generator, S2sOverrideContract s2sOverride) {
            if (!s2sOverride.isActive()) {
                throw new IllegalStateException("S2sOverride is not active");
            }

            this.generator = generator;

            attachments = s2sOverride.getApplicationOverride().getAttachments().stream().map(a -> new AttachmentData(a.getFileDataId(), a.getName(), a.getContentId(), a.getData(), a.getType(), InfrastructureConstants.HASH_ALGORITHM, a.getSha1Hash(), a.getUploadUser(), a.getUploadTimestamp())).toList();
            try {
                final XmlObject formsObject = getForms(getApplication(s2sOverride));
                final String formsXml = formsObject.toString();
                final String formsXpath = formsXml.startsWith("<xml-fragment") ?
                        "/xml-fragment/*[namespace-uri(.) = '" + generator.getNamespace() + "']" :
                        "/*[namespace-uri(.) = '" + generator.getNamespace() + "']";

                final XPathExecutor executer = new XPathExecutor(formsXml);
                final Node d = executer.getNode(formsXpath);
                xmlObject = factory().parse(d);
            } catch (XmlException e) {
                LOG.error(e.getMessage(), e);
                auditErrors.add(new AuditError(AuditError.NO_FIELD_ERROR_KEY, "error.proposalDevelopment.grants.gov.form.not.found", AuditError.GG_LINK, getFormName()));
            }
        }

        @Override
        public DocumentFactory<T> factory() {
            return generator.factory();
        }

        @Override
        public String getBeanName() {
            return OverriddenFormGeneratorAdapter.class.getName();
        }

        @Override
        public String getNamespace() {
            return generator.getNamespace();
        }

        @Override
        public String getFormName() {
            return generator.getFormName();
        }

        @Override
        public int getSortIndex() {
            return generator.getSortIndex();
        }

        @Override
        public List<Resource> getStylesheets() {
            return generator.getStylesheets();
        }

        @Override
        public boolean supportsPdfFilling() {
            return generator.supportsPdfFilling();
        }

        @Override
        public boolean supportsXslTransform() {
            return generator.supportsXslTransform();
        }

        @Override
        public T getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) throws S2SException {
            return xmlObject;
        }

        @Override
        public List<AttachmentData> getAttachments() {
            return attachments;
        }

        @Override
        public List<AuditError> getAuditErrors() {
            return auditErrors;
        }

        private GrantApplicationDocument getApplication(S2sOverrideContract s2sOverride) {
            final String submittedApplicationXml = s2sOverride.getApplicationOverride().getApplication();
            try {
                return GrantApplicationDocument.Factory.parse(submittedApplicationXml);
            } catch (XmlException e) {
                throw new S2SException(e);
            }
        }

        private XmlObject getForms(GrantApplicationDocument grantApplicationDocument) {
            try (XmlCursor cursor = grantApplicationDocument.getGrantApplication().getForms().newCursor()) {
                return cursor.getObject();
            }
        }

        public S2SFormGenerator<T> getGenerator() {
            return generator;
        }
    }

}
