/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.edGEPA427V11.GEPA427AttachmentsDocument;
import gov.grants.apply.forms.edGEPA427V11.GEPA427AttachmentsDocument.GEPA427Attachments;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;


/**
 * 
 * This class is used to generate XML Document object for grants.gov EDGEPA427V1.1. This form is generated using XMLBean API's
 * generated by compiling EDGEPA427V1.1 schema.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
@FormGenerator("EDGEPA427V1_1Generator")
public class EDGEPA427V1_1Generator extends EDGEPA427BaseGenerator<GEPA427AttachmentsDocument> {

    @Value("http://apply.grants.gov/forms/ED_GEPA427-V1.1")
    private String namespace;

    @Value("ED_GEPA427-V1.1")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/ED_GEPA427-V1.1.fo.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/ED_GEPA427-V1.1.pdf")
    private Resource pdfForm;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    /**
     * 
     * This method returns GEPA427AttachmentsDocument object based on proposal development document which contains the
     * GEPA427AttachmentsDocument informations about GEPA type attachment for a particular proposal
     * 
     * @return gepa427Document(GEPA427AttachmentsDocument){@link XmlObject} of type GEPA427AttachmentsDocument.
     */
    private GEPA427AttachmentsDocument getGEPA427Attachments() {

        GEPA427AttachmentsDocument gepa427Document = GEPA427AttachmentsDocument.Factory.newInstance();
        GEPA427Attachments gepAttachments = GEPA427Attachments.Factory.newInstance();
        gepAttachments.setFormVersion(FormVersion.v1_1.getVersion());
        AttachedFileDataType attachedFileDataType = null;
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            if (narrative.getNarrativeType().getCode() != null && Integer.parseInt(narrative.getNarrativeType().getCode()) == NARRATIVE_TYPE_ED_GEPA427) {
            	attachedFileDataType = addAttachedFileType(narrative);
            	if(attachedFileDataType != null){
            		gepAttachments.setAttachments(attachedFileDataType);
            		break;
            	}
            }
        }
        gepa427Document.setGEPA427Attachments(gepAttachments);
        return gepa427Document;
    }

    /**
     * This method creates {@link XmlObject} of type {@link GEPA427AttachmentsDocument} by populating data from the given
     * {@link ProposalDevelopmentDocumentContract}
     * 
     * @param proposalDevelopmentDocument for which the {@link XmlObject} needs to be created
     * @return {@link XmlObject} which is generated using the given {@link ProposalDevelopmentDocumentContract}
     * 
     */
    @Override
    public GEPA427AttachmentsDocument getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {

        this.pdDoc = proposalDevelopmentDocument;
        return getGEPA427Attachments();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(GEPA427AttachmentsDocument form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {
            if (form.getGEPA427Attachments().getAttachments() != null && a.getContentId().equals(form.getGEPA427Attachments().getAttachments().getFileLocation().getHref())) {
                return entry("ED_GEPA427_P1.optionalFile0", a);
            }

            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<GEPA427AttachmentsDocument> factory() {
        return GEPA427AttachmentsDocument.Factory;
    }
}
