/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.phsFellowshipSupplemental11V11.CitizenshipDataType;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.DegreeTypeDataType;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.FieldOfTrainingDataType;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.PHSFellowshipSupplemental11Document;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.AdditionalInformation;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.AdditionalInformation.*;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.ApplicationType;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.ApplicationType.TypeOfApplication;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.Budget;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.Budget.InstitutionalBaseSalary;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.Budget.InstitutionalBaseSalary.AcademicPeriod;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.Budget.SupplementationFromOtherSources;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.ResearchTrainingPlan;
import gov.grants.apply.forms.phsFellowshipSupplemental11V11.PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.ResearchTrainingPlan.*;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.person.attr.CitizenshipType;
import org.kuali.coeus.common.api.ynq.YnqConstant;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemContract;
import org.kuali.coeus.common.budget.api.period.BudgetPeriodContract;
import org.kuali.coeus.common.budget.api.personnel.BudgetPersonnelDetailsContract;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerContract;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerHeaderContract;
import org.kuali.coeus.common.questionnaire.api.core.QuestionnaireContract;
import org.kuali.coeus.common.questionnaire.api.core.QuestionnaireQuestionContract;
import org.kuali.coeus.common.questionnaire.api.question.QuestionContract;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.budget.ProposalDevelopmentBudgetExtContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.propdev.api.specialreview.ProposalSpecialReviewContract;
import org.kuali.coeus.s2sgen.api.core.ConfigurationConstants;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.util.FieldValueConstants;
import org.kuali.coeus.sys.api.model.ScaleTwoDecimal;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.math.BigDecimal;
import java.util.*;

@FormGenerator("PHS398FellowshipSupplementalV1_1Generator")
public class PHS398FellowshipSupplementalV1_1Generator extends PHS398FellowshipSupplementalBaseGenerator<PHSFellowshipSupplemental11Document> {

    private static final int DEGREE_TYPE_SOUGHT = 15;
    private static final int OTHER_DBOTH = 20;

	private static final ScaleTwoDecimal POINT_ZERO_ONE = new ScaleTwoDecimal(.01);

    @Value("http://apply.grants.gov/forms/PHS_Fellowship_Supplemental_1_1-V1.1")
    private String namespace;

    @Value("PHS_Fellowship_Supplemental_1_1-V1.1")
    private String formName;

	@FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/PHS_fellowship_supplemental-V1.1.xsl")
    private List<Resource> stylesheets;

    @Value("210")
    private int sortIndex;
    
	/*
	 * This method is used to get PHSFellowshipSupplemental11 XMLObject and set
	 * the data to it from DevelopmentProposal data.
	 */
	private PHSFellowshipSupplemental11 getPHSFellowshipSupplemental11() {
		PHSFellowshipSupplemental11 phsFellowshipSupplemental = PHSFellowshipSupplemental11.Factory
				.newInstance();
		phsFellowshipSupplemental.setFormVersion(FormVersion.v1_1.getVersion());
		phsFellowshipSupplemental.setApplicationType(getApplicationType());
		phsFellowshipSupplemental.setAppendix(getAppendix());
		phsFellowshipSupplemental.setAdditionalInformation(getAdditionalInformation());
		phsFellowshipSupplemental
				.setResearchTrainingPlan(getResearchTrainingPlan());
		phsFellowshipSupplemental.setBudget(getBudget());
		return phsFellowshipSupplemental;
	}

	/*
	 * This method is used to get Budget XMLObject and set the data to it from
	 * ProposalYnq based on questionId and answers.
	 */
	private Budget getBudget() {
		Budget budget = Budget.Factory.newInstance();
		Map<Integer, String> budgetMap = new HashMap<>();

		for (AnswerContract questionnaireAnswer : getPropDevQuestionAnswerService().getQuestionnaireAnswers(pdDoc.getDevelopmentProposal().getProposalNumber(), getNamespace(), getFormName())) {
			String answer = questionnaireAnswer.getAnswer();
			if (answer != null) {
				switch (getQuestionAnswerService().findQuestionById(questionnaireAnswer.getQuestionId()).getQuestionSeqId()) {
				case SENIOR_FELL:
					budgetMap.put(SENIOR_FELL, answer);
					break;
                case OTHER_SUPP_SOURCE:
                    budgetMap.put(OTHER_SUPP_SOURCE, answer);
                    break;
				case SUPP_SOURCE:
					budgetMap.put(SUPP_SOURCE, answer);
					break;
				case SUPP_FUNDING_AMT:
					budgetMap.put(SUPP_FUNDING_AMT, answer);
					break;
				case SUPP_MONTHS:
					budgetMap.put(SUPP_MONTHS, answer);
					break;
				case SUPP_TYPE:
					budgetMap.put(SUPP_TYPE, answer);
					break;
				case SALARY_MONTHS:
					budgetMap.put(SALARY_MONTHS, answer);
					break;
				case ACAD_PERIOD:
					budgetMap.put(ACAD_PERIOD, answer);
					break;
				case BASE_SALARY:
					budgetMap.put(BASE_SALARY, answer);
					break;
				default:
					break;
				}
			}
		}
		budget.setTuitionAndFeesRequested(YesNoDataType.N_NO);
		if(getInstitutionalBaseSalary(budgetMap)!=null){
		    budget.setInstitutionalBaseSalary(getInstitutionalBaseSalary(budgetMap));
		}
		getFederalStipendRequested(budget);
		if(getSupplementationFromOtherSources(budgetMap)!=null){
		    budget.setSupplementationFromOtherSources(getSupplementationFromOtherSources(budgetMap));
		}
		setTutionRequestedYears(budget);
		return budget;
	}

	/*
	 * This method is used to get TutionRequestedYears data to Budget XMLObject
	 * from List of BudgetLineItem based on CostElement value of
	 * TUITION_COST_ELEMENTS
	 */
	private void setTutionRequestedYears(Budget budget) {
        ProposalDevelopmentBudgetExtContract budgetEx = s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());
		if (budgetEx == null) {
			return;
		}
		ScaleTwoDecimal tutionTotal = ScaleTwoDecimal.ZERO;
		for (BudgetPeriodContract budgetPeriod : budgetEx
				.getBudgetPeriods()) {
			ScaleTwoDecimal tution = ScaleTwoDecimal.ZERO;
			for (BudgetLineItemContract budgetLineItem : budgetPeriod.getBudgetLineItems()) {
				if (getCostElementsByParam(ConfigurationConstants.TUITION_COST_ELEMENTS).contains(budgetLineItem.getCostElementBO().getCostElement())) {
					tution = tution.add(budgetLineItem.getLineItemCost());
				}
			}
			tutionTotal = tutionTotal.add(tution);
			switch (budgetPeriod.getBudgetPeriod()) {
			case 1:
				budget.setTuitionRequestedYear1(tution.bigDecimalValue());
				break;
			case 2:
				budget.setTuitionRequestedYear2(tution.bigDecimalValue());
				break;
			case 3:
				budget.setTuitionRequestedYear3(tution.bigDecimalValue());
				break;
			case 4:
				budget.setTuitionRequestedYear4(tution.bigDecimalValue());
				break;
			case 5:
				budget.setTuitionRequestedYear5(tution.bigDecimalValue());
				break;
			case 6:
				budget.setTuitionRequestedYear6(tution.bigDecimalValue());
				break;
			default:
				break;
			}
		}
		budget.setTuitionRequestedTotal(tutionTotal.bigDecimalValue());
		if (!tutionTotal.equals(ScaleTwoDecimal.ZERO)) {
			budget.setTuitionAndFeesRequested(YesNoDataType.Y_YES);
		}
	}


    /*
	 * This method is used to set data to SupplementationFromOtherSources
	 * XMLObject from budgetMap data for Budget
	 */
	private SupplementationFromOtherSources getSupplementationFromOtherSources(
			Map<Integer, String> budgetMap) {
	    SupplementationFromOtherSources supplementationFromOtherSources=null;
		if (budgetMap.get(OTHER_SUPP_SOURCE) != null
				&& budgetMap.get(OTHER_SUPP_SOURCE).equals(
						YnqConstant.YES.code())) {
	        supplementationFromOtherSources = SupplementationFromOtherSources.Factory.newInstance();
			if (budgetMap.get(SUPP_SOURCE) != null) {
				supplementationFromOtherSources.setSource(budgetMap.get(
						SUPP_SOURCE));
			}
			if (budgetMap.get(SUPP_FUNDING_AMT) != null) {
				supplementationFromOtherSources.setAmount(new BigDecimal(
						budgetMap.get(SUPP_FUNDING_AMT)));
			}else{
			    supplementationFromOtherSources.setAmount(null);
			}
			if (budgetMap.get(SUPP_MONTHS) != null) {
				supplementationFromOtherSources
						.setNumberOfMonths(new BigDecimal(budgetMap.get(
								SUPP_MONTHS)));
			}
			if (budgetMap.get(SUPP_TYPE) != null) {
				supplementationFromOtherSources.setType(budgetMap
						.get(SUPP_TYPE));
			}
		}
		return supplementationFromOtherSources;
	}

	protected void getFederalStipendRequested(PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.Budget budget) {
		PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.Budget.FederalStipendRequested federalStipendRequested = PHSFellowshipSupplemental11Document.PHSFellowshipSupplemental11.Budget.FederalStipendRequested.Factory.newInstance();
		ProposalDevelopmentBudgetExtContract pBudget = s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());
		if (pBudget != null) {
			ScaleTwoDecimal sumOfLineItemCost = ScaleTwoDecimal.ZERO;
			ScaleTwoDecimal numberOfMonths = ScaleTwoDecimal.ZERO;

			for (BudgetPeriodContract budgetPeriod : pBudget.getBudgetPeriods()) {
				if (budgetPeriod.getBudgetPeriod() == 1) {
					for (BudgetLineItemContract budgetLineItem : budgetPeriod.getBudgetLineItems()) {
						if (getCostElementsByParam(ConfigurationConstants.STIPEND_COST_ELEMENTS).contains(budgetLineItem.getCostElementBO().getCostElement())) {
							sumOfLineItemCost = sumOfLineItemCost.add(budgetLineItem.getLineItemCost());

							if (CollectionUtils.isNotEmpty(budgetLineItem.getBudgetPersonnelDetailsList())) {
								for (BudgetPersonnelDetailsContract details : budgetLineItem.getBudgetPersonnelDetailsList()){
									final ScaleTwoDecimal detailsNumberOfMonths = details.getPercentEffort() != null ?
											getNumberOfMonths(details.getStartDate(), details.getEndDate()).multiply(details.getPercentEffort().multiply(POINT_ZERO_ONE)) :
											getNumberOfMonths(details.getStartDate(), details.getEndDate());

									numberOfMonths = numberOfMonths.add(detailsNumberOfMonths);
								}
							} else {
								numberOfMonths = numberOfMonths.add(getNumberOfMonths(budgetLineItem.getStartDate(), budgetLineItem.getEndDate()));
							}
						}
					}
				}
			}
			federalStipendRequested.setAmount(sumOfLineItemCost.bigDecimalValue());
			federalStipendRequested.setNumberOfMonths(numberOfMonths.bigDecimalValue());
			budget.setFederalStipendRequested(federalStipendRequested);

		}
	}

	/*
	 * This method is used to set data to InstitutionalBaseSalary XMLObject from
	 * budgetMap data for Budget
	 */
	private InstitutionalBaseSalary getInstitutionalBaseSalary(
			Map<Integer, String> budgetMap) {
	    InstitutionalBaseSalary institutionalBaseSalary=null;
		if (budgetMap.get(SENIOR_FELL) != null
				&& budgetMap.get(SENIOR_FELL).equals(
						YnqConstant.YES.code())) {
	        institutionalBaseSalary = InstitutionalBaseSalary.Factory.newInstance();
			if (budgetMap.get(BASE_SALARY) != null) {
				institutionalBaseSalary.setAmount(new BigDecimal(budgetMap.get(
						BASE_SALARY)));
			}else{
			    institutionalBaseSalary.setAmount(null);
			}
			if (budgetMap.get(ACAD_PERIOD) != null) {
				institutionalBaseSalary.setAcademicPeriod(AcademicPeriod.Enum
						.forString(budgetMap.get(ACAD_PERIOD)));
			}
			if (budgetMap.get(SALARY_MONTHS) != null) {
				institutionalBaseSalary.setNumberOfMonths(new BigDecimal(
						budgetMap.get(SALARY_MONTHS)));
			}
		}
		return institutionalBaseSalary;
	}

	/*
	 * This method used to set data to ResearchTrainingPlan XMLObject from
	 * DevelopmentProposal
	 */
	private ResearchTrainingPlan getResearchTrainingPlan() {
		ResearchTrainingPlan researchTrainingPlan = ResearchTrainingPlan.Factory
				.newInstance();
		setHumanSubjectInvolvedAndVertebrateAnimalUsed(researchTrainingPlan);
		setQuestionnareAnswerForResearchTrainingPlan(researchTrainingPlan);
		setNarrativeDataForResearchTrainingPlan(researchTrainingPlan);
		return researchTrainingPlan;
	}

	/**
	 * This method is used to set Narrative Data to ResearchTrainingPlan
	 * XMLObject based on NarrativeTypeCode.
	 *
	 */
	private void setNarrativeDataForResearchTrainingPlan(
			ResearchTrainingPlan researchTrainingPlan) {
		AttachedFileDataType attachedFileDataType;
        SpecificAims specificAims = SpecificAims.Factory.newInstance();
        ResearchStrategy researchStrategy = ResearchStrategy.Factory.newInstance();
        RespectiveContributions respectiveContributions = RespectiveContributions.Factory.newInstance();
        SelectionOfSponsorAndInstitution selectionOfSponsorAndInstitution = SelectionOfSponsorAndInstitution.Factory.newInstance();
        ResponsibleConductOfResearch responsibleConductOfResearch = ResponsibleConductOfResearch.Factory.newInstance();

		for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
			if (narrative.getNarrativeType().getCode() != null) {
				switch (Integer.parseInt(narrative.getNarrativeType().getCode())) {
				case INTRODUCTION_TO_APPLICATION:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					IntroductionToApplication introductionToApplication = IntroductionToApplication.Factory
							.newInstance();
					introductionToApplication.setAttFile(attachedFileDataType);
					researchTrainingPlan
							.setIntroductionToApplication(introductionToApplication);
					break;
				case SPECIFIC_AIMS:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					specificAims.setAttFile(attachedFileDataType);
					break;

				case INCLUSION_ENROLLMENT_REPORT:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					InclusionEnrollmentReport inclusionEnrollmentReport = InclusionEnrollmentReport.Factory
							.newInstance();
					inclusionEnrollmentReport.setAttFile(attachedFileDataType);
					researchTrainingPlan
							.setInclusionEnrollmentReport(inclusionEnrollmentReport);
					break;
				case PROGRESS_REPORT_PUBLICATION_LIST:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					ProgressReportPublicationList progressReportPublicationList = ProgressReportPublicationList.Factory
							.newInstance();
					progressReportPublicationList
							.setAttFile(attachedFileDataType);
					researchTrainingPlan
							.setProgressReportPublicationList(progressReportPublicationList);
					break;
				case PROTECTION_OF_HUMAN_SUBJECTS:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					ProtectionOfHumanSubjects protectionOfHumanSubjects = ProtectionOfHumanSubjects.Factory
							.newInstance();
					protectionOfHumanSubjects.setAttFile(attachedFileDataType);
					researchTrainingPlan
							.setProtectionOfHumanSubjects(protectionOfHumanSubjects);
					break;
				case INCLUSION_OF_WOMEN_AND_MINORITIES:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					InclusionOfWomenAndMinorities inclusionOfWomenAndMinorities = InclusionOfWomenAndMinorities.Factory
							.newInstance();
					inclusionOfWomenAndMinorities
							.setAttFile(attachedFileDataType);
					researchTrainingPlan
							.setInclusionOfWomenAndMinorities(inclusionOfWomenAndMinorities);
					break;
				case TARGETED_PLANNED_ENROLLMENT:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					TargetedPlannedEnrollment tarPlannedEnrollmentTable = TargetedPlannedEnrollment.Factory
							.newInstance();
					tarPlannedEnrollmentTable.setAttFile(attachedFileDataType);
					researchTrainingPlan
							.setTargetedPlannedEnrollment(tarPlannedEnrollmentTable);
					break;
				case INCLUSION_OF_CHILDREN:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					InclusionOfChildren inclusionOfChildren = InclusionOfChildren.Factory
							.newInstance();
					inclusionOfChildren.setAttFile(attachedFileDataType);
					researchTrainingPlan
							.setInclusionOfChildren(inclusionOfChildren);
					break;
				case VERTEBRATE_ANIMALS:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					VertebrateAnimals vertebrateAnimals = VertebrateAnimals.Factory
							.newInstance();
					vertebrateAnimals.setAttFile(attachedFileDataType);
					researchTrainingPlan
							.setVertebrateAnimals(vertebrateAnimals);
					break;
				case SELECT_AGENT_RESEARCH:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					SelectAgentResearch selectAgentResearch = SelectAgentResearch.Factory
							.newInstance();
					selectAgentResearch.setAttFile(attachedFileDataType);
					researchTrainingPlan
							.setSelectAgentResearch(selectAgentResearch);
					break;
				case RESOURCE_SHARING_PLANS:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					ResourceSharingPlan resourceSharingPlan = ResourceSharingPlan.Factory
							.newInstance();
					resourceSharingPlan.setAttFile(attachedFileDataType);
					researchTrainingPlan
							.setResourceSharingPlan(resourceSharingPlan);
					break;
				case RESPECTIVE_CONTRIBUTIONS:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					respectiveContributions.setAttFile(attachedFileDataType);
					break;
				case SELECTION_OF_SPONSOR_AND_INSTITUTION:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					selectionOfSponsorAndInstitution
							.setAttFile(attachedFileDataType);
					break;
				case RESPONSIBLE_CONDUCT_OF_RESEARCH:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					responsibleConductOfResearch.setAttFile(attachedFileDataType);
					break;
				case RESEARCH_STRATEGY:
	                attachedFileDataType = addAttachedFileType(narrative);
	                if(attachedFileDataType == null){
	                    continue;
	                }
					researchStrategy.setAttFile(attachedFileDataType);
					break;
				default:
					break;
				}
			}
		}
        researchTrainingPlan.setSpecificAims(specificAims);
        researchTrainingPlan.setResearchStrategy(researchStrategy);
        researchTrainingPlan.setRespectiveContributions(respectiveContributions);
        researchTrainingPlan.setSelectionOfSponsorAndInstitution(selectionOfSponsorAndInstitution);
        researchTrainingPlan.setResponsibleConductOfResearch(responsibleConductOfResearch);

	}

	/**
	 * This method is used to set QuestionnareAnswer data to
	 * ResearchTrainingPlan XMLObject
	 *
	 */
	private void setQuestionnareAnswerForResearchTrainingPlan(
			ResearchTrainingPlan researchTrainingPlan) {
	    researchTrainingPlan.setHumanSubjectsIndefinite(YesNoDataType.N_NO);
        researchTrainingPlan.setVertebrateAnimalsIndefinite(YesNoDataType.N_NO);
        researchTrainingPlan.setHumanSubjectsIndefinite(YesNoDataType.N_NO);
        researchTrainingPlan.setClinicalTrial(YesNoDataType.N_NO);
        researchTrainingPlan.setPhase3ClinicalTrial(YesNoDataType.N_NO);
		for (AnswerContract questionnaireAnswer : getPropDevQuestionAnswerService().getQuestionnaireAnswers(pdDoc.getDevelopmentProposal().getProposalNumber(), getNamespace(), getFormName())) {
			String answer = questionnaireAnswer.getAnswer();
			if (answer != null) {
				switch (getQuestionAnswerService().findQuestionById(questionnaireAnswer.getQuestionId()).getQuestionSeqId()) {
				case HUMAN:
					researchTrainingPlan
							.setHumanSubjectsIndefinite(answer
									.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES
									: YesNoDataType.N_NO);
					break;
				case VERT:
					researchTrainingPlan
							.setVertebrateAnimalsIndefinite(answer
									.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES
									: YesNoDataType.N_NO);
					break;
				case CLINICAL:
					researchTrainingPlan
							.setClinicalTrial(answer
									.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES
									: YesNoDataType.N_NO);
					break;
				case PHASE3CLINICAL:
					researchTrainingPlan
							.setPhase3ClinicalTrial(answer
									.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES
									: YesNoDataType.N_NO);
					break;
				default:
					break;
				}
			}
		}
	}

	/**
	 * This method is used to set HumanSubjectInvoved and VertebrateAnimalUsed
	 * XMLObject Data.
	 *
	 */
	private void setHumanSubjectInvolvedAndVertebrateAnimalUsed(
			ResearchTrainingPlan researchTrainingPlan) {
		researchTrainingPlan.setHumanSubjectsInvolved(YesNoDataType.N_NO);
		researchTrainingPlan.setVertebrateAnimalsUsed(YesNoDataType.N_NO);
		for (ProposalSpecialReviewContract propSpecialReview : pdDoc
				.getDevelopmentProposal().getPropSpecialReviews()) {
			if (SPECIAL_REVIEW_HUMAN_SUBJECTS_CODE.equals(propSpecialReview.getSpecialReviewType().getCode())) {
				researchTrainingPlan.setHumanSubjectsInvolved(YesNoDataType.Y_YES);
			} else if (SPECIAL_REVIEW_VERTEBRATE_ANIMALS_CODE.equals(propSpecialReview.getSpecialReviewType().getCode())) {
				researchTrainingPlan.setVertebrateAnimalsUsed(YesNoDataType.Y_YES);
			}
		}
	}

	/*
	 * This method is used to set additional information data to
	 * AdditionalInformation XMLObject from DevelopmentProposal, ProposalYnq
	 */
	private AdditionalInformation getAdditionalInformation() {
        AdditionalInformation additionalInformation = AdditionalInformation.Factory.newInstance();
        additionalInformation.setCitizenship(CitizenshipDataType.U_S_CITIZEN_OR_NONCITIZEN_NATIONAL);
        StemCells stemCells = StemCells.Factory.newInstance();
        stemCells.setIsHumanStemCellsInvolved(YesNoDataType.N_NO);
        stemCells.setStemCellsIndicator(YesNoDataType.N_NO);
        GraduateDegreeSought graduateDegreeSought = GraduateDegreeSought.Factory.newInstance();
        ProposalPersonContract principalInvestigator = s2SProposalPersonService.getPrincipalInvestigator(pdDoc);
        ArrayList<String> cellLinesList = new ArrayList<>(Arrays.asList(stemCells.getCellLinesArray()));
		for (ProposalPersonContract proposalPerson : pdDoc.getDevelopmentProposal().getProposalPersons()) {
			if (proposalPerson.isPrincipalInvestigator()) {
				CitizenshipType citizenShip=s2SProposalPersonService.getCitizenship(proposalPerson);
				if(citizenShip!=null && StringUtils.isNotBlank(citizenShip.getCitizenShip())){
					if(citizenShip.getCitizenShip().trim().equals(CitizenshipDataType.NON_U_S_CITIZEN_WITH_TEMPORARY_VISA.toString())){
						additionalInformation.setCitizenship(CitizenshipDataType.NON_U_S_CITIZEN_WITH_TEMPORARY_VISA);
					}
					else if(citizenShip.getCitizenShip().trim().equals(CitizenshipDataType.PERMANENT_RESIDENT_OF_U_S.toString())){
						additionalInformation.setCitizenship(CitizenshipDataType.PERMANENT_RESIDENT_OF_U_S);
					}
					else if(citizenShip.getCitizenShip().trim().equals(CitizenshipDataType.U_S_CITIZEN_OR_NONCITIZEN_NATIONAL.toString())){
						additionalInformation.setCitizenship(CitizenshipDataType.U_S_CITIZEN_OR_NONCITIZEN_NATIONAL);
					}
					else if(citizenShip.getCitizenShip().trim().equals(CitizenshipDataType.PERMANENT_RESIDENT_OF_U_S_PENDING.toString())){
						additionalInformation.setCitizenship(CitizenshipDataType.PERMANENT_RESIDENT_OF_U_S_PENDING);
					}
				}else{
					additionalInformation.setCitizenship(null);
				}
			}
		}
		if (principalInvestigator != null && principalInvestigator.getMobilePhoneNumber() != null) {
			additionalInformation.setAlernatePhoneNumber(principalInvestigator.getMobilePhoneNumber());
		}
        List<? extends AnswerHeaderContract> answerHeaders = getPropDevQuestionAnswerService().getQuestionnaireAnswerHeaders(pdDoc.getDevelopmentProposal().getProposalNumber(), getNamespace(), getFormName());
		for (AnswerHeaderContract answerHeader : answerHeaders) {
            for (AnswerContract questionnaireAnswer : answerHeader.getAnswers()) {
                String answer = questionnaireAnswer.getAnswer();

                Integer seqId = getQuestionAnswerService().findQuestionById(questionnaireAnswer.getQuestionId()).getQuestionSeqId();
                if(seqId.equals(STEMCELLLINES)){
                    List<AnswerContract> answerList = getAnswers(questionnaireAnswer.getQuestionnaireQuestionsId(),answerHeader);
                    for (AnswerContract questionnaireAnswerBO: answerList) {
                        String questionnaireSubAnswer =  questionnaireAnswerBO.getAnswer();
                        if(questionnaireSubAnswer!=null){
                            cellLinesList.add(questionnaireSubAnswer);
                            stemCells.addCellLines(questionnaireSubAnswer);
                        }
                    }
                }

                if (answer != null) {
                    switch (seqId) {
                        case BROAD_TRAINING:
                case FIELD_TRAINING:
                            if (!answer.toUpperCase().equals(SUB_CATEGORY_NOT_FOUND)) {
                                FieldOfTrainingDataType.Enum e = FieldOfTrainingDataType.Enum.forString(answer);
                                additionalInformation.setFieldOfTraining(e);
                            }
                            break;
                case NRSA_SUPPORT:
                            additionalInformation
                                    .setCurrentPriorNRSASupportIndicator(answer
                                            .equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES
                                            : YesNoDataType.N_NO);
                            break;
                case SUBMITTED_DIFF_INST:
                            additionalInformation
                                    .setChangeOfInstitution(answer
                                            .equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES
                                            : YesNoDataType.N_NO);
                            break;
                case FORMER_INST:
                            additionalInformation.setFormerInstitution(answer);
                            break;
                case STEMCELLS:
                            stemCells
                                    .setIsHumanStemCellsInvolved(answer
                                            .equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES
                                            : YesNoDataType.N_NO);
                            break;
                case CELLLINEIND:
                            stemCells.setStemCellsIndicator(answer
                                    .equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES
                                    : YesNoDataType.N_NO);
                            break;

                case DEGREE_TYPE_SOUGHT:
                            graduateDegreeSought.setDegreeType(DegreeTypeDataType.Enum.forString(answer));
                            break;
                case DEG_EXP_COMP_DATE:
                            graduateDegreeSought.setDegreeDate(answer.substring(6, 10)
                                    + STRING_SEPRATOR + answer.substring(0, 2));
                            break;
                case OTHER_MASTERS:
                            graduateDegreeSought.setOtherDegreeTypeText(answer);
                            break;
                case OTHER_DOCT:
                            graduateDegreeSought.setOtherDegreeTypeText(answer);
                            break;
                case OTHER_DDOT:
                            graduateDegreeSought.setDegreeType(DegreeTypeDataType.DDOT_OTHER_DOCTOR_OF_MEDICAL_DENTISTRY);
                            graduateDegreeSought.setOtherDegreeTypeText(answer);
                            break;
                case OTHER_VDOT:
                            graduateDegreeSought.setDegreeType(DegreeTypeDataType.VDOT_OTHER_DOCTOR_OF_VETERINARY_MEDICINE);
                            graduateDegreeSought.setOtherDegreeTypeText(answer);
                            break;
                case OTHER_DBOTH:
                            graduateDegreeSought.setDegreeType(DegreeTypeDataType.DBOTH_OTHER_DOUBLE_DEGREE_PROGRAM);
                            graduateDegreeSought.setOtherDegreeTypeText(answer);
                            break;
                case OTHER_MDOT:
                            graduateDegreeSought.setDegreeType(DegreeTypeDataType.MDOT_OTHER_DOCTOR_OF_MEDICINE);
                            graduateDegreeSought.setOtherDegreeTypeText(answer);
                            break;
                        default:
                            break;
                    }
                }
            }
        }

        additionalInformation.setStemCells(stemCells);
        if(graduateDegreeSought.getDegreeType()!=null){
            additionalInformation.setGraduateDegreeSought(graduateDegreeSought);
        }
        additionalInformation
                .setCurrentPriorNRSASupportArray(getCurrentPriorNRSASupportArray());
        additionalInformation.setConcurrentSupport(YesNoDataType.N_NO);
        FellowshipTrainingAndCareerGoals fellowshipTrainingAndCareerGoals = FellowshipTrainingAndCareerGoals.Factory.newInstance();
        ActivitiesPlannedUnderThisAward activitiesPlannedUnderThisAward = ActivitiesPlannedUnderThisAward.Factory.newInstance();

        AttachedFileDataType attachedFileDataType;
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal()
                .getNarratives()) {
            if (narrative.getNarrativeType().getCode() != null) {
                switch (Integer.parseInt(narrative.getNarrativeType().getCode())) {
                case CONCURRENT_SUPPORT:
                    attachedFileDataType = addAttachedFileType(narrative);
                    if(attachedFileDataType == null){
                        continue;
                    }
                    ConcurrentSupportDescription concurrentSupportDescription = ConcurrentSupportDescription.Factory
                            .newInstance();
                    concurrentSupportDescription
                            .setAttFile(attachedFileDataType);
                    additionalInformation
                            .setConcurrentSupport(YesNoDataType.Y_YES);
                    additionalInformation
                            .setConcurrentSupportDescription(concurrentSupportDescription);
                    break;
                case FELLOWSHIP:
                    attachedFileDataType = addAttachedFileType(narrative);
                    if(attachedFileDataType == null){
                        continue;
                    }
                    fellowshipTrainingAndCareerGoals.setAttFile(attachedFileDataType);
                    break;
                case DISSERTATION:
                    attachedFileDataType = addAttachedFileType(narrative);
                    if(attachedFileDataType == null){
                        continue;
                    }
                    DissertationAndResearchExperience dissertationAndResearchExperience = DissertationAndResearchExperience.Factory
                            .newInstance();
                    dissertationAndResearchExperience
                            .setAttFile(attachedFileDataType);
                    additionalInformation
                            .setDissertationAndResearchExperience(dissertationAndResearchExperience);
                    break;
                case ACTIVITIES:
                    attachedFileDataType = addAttachedFileType(narrative);
                    if(attachedFileDataType == null){
                        continue;
                    }
                    activitiesPlannedUnderThisAward.setAttFile(attachedFileDataType);
                    break;
                default:
                    break;

                }
            }
        }
        additionalInformation.setFellowshipTrainingAndCareerGoals(fellowshipTrainingAndCareerGoals);
        additionalInformation.setActivitiesPlannedUnderThisAward(activitiesPlannedUnderThisAward);
        return additionalInformation;
    }

	/*
	 * This method is used to get Arrays of CurrentPriorNRSASupport XMLObject
	 * and set data to it from List of ProposalYnq
	 */
	private CurrentPriorNRSASupport[] getCurrentPriorNRSASupportArray() {
        List<CurrentPriorNRSASupport> currentPriorNRSASupportList = new ArrayList<>();
        List<AnswerContract> answerList = new ArrayList<>();
        String nsrSupport = null;
        
        List<? extends AnswerHeaderContract> answers = findQuestionnaireWithAnswers(pdDoc.getDevelopmentProposal());
        for (AnswerHeaderContract answerHeader : answers) {
            QuestionnaireContract questionnaire = questionAnswerService.findQuestionnaireById(answerHeader.getQuestionnaireId());
            List<? extends QuestionnaireQuestionContract> questionnaireQuestions = questionnaire.getQuestionnaireQuestions();
            for (QuestionnaireQuestionContract questionnaireQuestion : questionnaireQuestions) {
                AnswerContract answerBO = getAnswer(questionnaireQuestion,answerHeader);
                String answer = answerBO != null ? answerBO.getAnswer() : null;
                QuestionContract question = questionnaireQuestion.getQuestion();

                if (answer != null) {
                    int questionSeqId = question.getQuestionSeqId();
                switch (questionSeqId) {
                case KIRST_START_KNOWN:
                case KIRST_END_KNOWN:
                case KIRST_START_DT:
                case KIRST_END_DT:
                case KIRST_GRANT_KNOWN:
                case KIRST_GRANT_NUM:
                case PRE_OR_POST:
                case IND_OR_INST:
                    if (questionSeqId == KIRST_START_KNOWN) {
                        if (answer.equals(QUESTION_ANSWER_NO)) {
                            answer = FieldValueConstants.VALUE_UNKNOWN;
                            questionSeqId = KIRST_START_DT;
                        } else {
                            break;
                        }
                    }
                        if (questionSeqId == KIRST_END_KNOWN) {
                            if (answer.equals(QUESTION_ANSWER_NO)) {
                                answer = FieldValueConstants.VALUE_UNKNOWN;
                                questionSeqId = KIRST_END_DT;
                            } else {
                                break;
                            }
                        }
                        if (questionSeqId == KIRST_GRANT_KNOWN) {
                            if (answer.equals(QUESTION_ANSWER_NO)) {
                                answer = FieldValueConstants.VALUE_UNKNOWN;
                                questionSeqId = KIRST_GRANT_NUM;
                            } else {
                                break;
                            }
                        }
                        InternalAnswer quesAnswer = new InternalAnswer();
                        quesAnswer.setAnswer(answer);
                        quesAnswer.setQuestionSeqId(questionSeqId);
                        answerList.add(quesAnswer);

                    break;
                case NRSA_SUPPORT:
                    nsrSupport = answer
                            .equals(YnqConstant.YES.code()) ? NSR_SUPPORT_YES
                            : NSR_SUPPORT_NO;
                    break;
                default:
                    break;
                }
            }
        }
    }
        answerList.sort(Comparator.comparing(AnswerContract::getQuestionSeqId));
        List<CurrentPriorNRSASupport.Level.Enum> levelList = new ArrayList<>();
        List<CurrentPriorNRSASupport.Type.Enum> typeList = new ArrayList<>();
        List<Calendar> startDateList = new ArrayList<>();
        List<Calendar> endDateList = new ArrayList<>();
        List<String> grantNumberList = new ArrayList<>();
        for (AnswerContract questionnaireAnswer : answerList) {
            if (nsrSupport != null && nsrSupport.equals(NSR_SUPPORT_YES)) {
                if (questionnaireAnswer.getQuestionSeqId() == PRE_OR_POST) {
                    levelList.add(CurrentPriorNRSASupport.Level.Enum
                            .forString(questionnaireAnswer.getAnswer()));
                }
                if (questionnaireAnswer.getQuestionSeqId() == IND_OR_INST) {
                    typeList.add(CurrentPriorNRSASupport.Type.Enum
                            .forString(questionnaireAnswer.getAnswer()));
                }
                if (questionnaireAnswer.getQuestionSeqId() == KIRST_START_DT){
                    if(questionnaireAnswer.getAnswer().equals(FieldValueConstants.VALUE_UNKNOWN)) {
                        startDateList.add(null);
                    }else{
                        startDateList.add(s2SDateTimeService.convertDateStringToCalendar(questionnaireAnswer.getAnswer()));
                    }
                }
                if (questionnaireAnswer.getQuestionSeqId() == KIRST_END_DT){
                    if(questionnaireAnswer.getAnswer().equals(FieldValueConstants.VALUE_UNKNOWN)) {
                        endDateList.add(null);
                    }else{
                        endDateList.add(s2SDateTimeService.convertDateStringToCalendar(questionnaireAnswer.getAnswer()));
                    }
                }
                if (questionnaireAnswer.getQuestionSeqId() == KIRST_GRANT_NUM) {
                    grantNumberList.add(questionnaireAnswer.getAnswer());
                }
            }
        }
        for (int index = 0; levelList.size() > index; index++) {
            CurrentPriorNRSASupport currentPriorNRSASupport = CurrentPriorNRSASupport.Factory.newInstance();
            currentPriorNRSASupport.setLevel(levelList.get(index));
            currentPriorNRSASupport.setType(typeList.get(index));
            if(!startDateList.isEmpty() && startDateList.get(index)!=null)
                currentPriorNRSASupport.setStartDate(startDateList.get(index));
            if(!endDateList.isEmpty() && endDateList.get(index)!=null)
                currentPriorNRSASupport.setEndDate(endDateList.get(index));
            currentPriorNRSASupport.setGrantNumber(grantNumberList.get(index));
            currentPriorNRSASupportList.add(currentPriorNRSASupport);
        }
        return currentPriorNRSASupportList.toArray(new CurrentPriorNRSASupport[0]);
    }


	/*
	 * This method is used to get ApplicationType XMLObject and set data to it
	 * from types of Application.
	 */
	private ApplicationType getApplicationType() {
		ApplicationType applicationType = ApplicationType.Factory.newInstance();
		applicationType.setTypeOfApplication(getTypeOfApplication());
		return applicationType;
	}

	/*
	 * This method is used to get TypeOfApplication based on proposalTypeCode of
	 * DevelopmentProposal
	 */
	private TypeOfApplication.Enum getTypeOfApplication() {
        String proposalTypeCode = pdDoc.getDevelopmentProposal().getProposalType().getCode();
        TypeOfApplication.Enum typeOfApplication = null;
        if (proposalTypeCode != null) {
            if (s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_NEW).contains(proposalTypeCode)) {
                typeOfApplication = TypeOfApplication.NEW;
            }
            else if (s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_CONTINUATION).contains(proposalTypeCode)) {
                typeOfApplication = TypeOfApplication.CONTINUATION;
            }
            else if (s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_REVISION).contains(proposalTypeCode)){ 
                typeOfApplication = TypeOfApplication.REVISION;
            }
            else if (s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_RENEWAL).contains(proposalTypeCode)){ 
                typeOfApplication = TypeOfApplication.RENEWAL;
            }
            else if (s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_RESUBMISSION).contains(proposalTypeCode)){ 
                typeOfApplication = TypeOfApplication.RESUBMISSION;
            }
        }
        return typeOfApplication;
    }

	@Override
	public PHSFellowshipSupplemental11Document getFormObject(
			ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
		this.pdDoc = proposalDevelopmentDocument;
        PHSFellowshipSupplemental11 phsFellowshipSupplemental = getPHSFellowshipSupplemental11();
        PHSFellowshipSupplemental11Document phsFellowshipSupplementalDocument = PHSFellowshipSupplemental11Document.Factory
                .newInstance();
        phsFellowshipSupplementalDocument
                .setPHSFellowshipSupplemental11(phsFellowshipSupplemental);
        return phsFellowshipSupplementalDocument;
	}

	@Override
	public DocumentFactory<PHSFellowshipSupplemental11Document> factory() {
		return PHSFellowshipSupplemental11Document.Factory;
	}

	private static class InternalAnswer implements AnswerContract {
        private Long id;
        private Integer questionNumber;
        private Integer answerNumber;
        private String answer;
        private Long answerHeaderId;
        private Long questionId;
        private Integer questionSeqId;
        private Long questionnaireQuestionsId;
        private List<InternalAnswer> parentAnswers;

        @Override
        public Long getId() {
            return id;
        }

        @Override
        public Integer getQuestionNumber() {
            return questionNumber;
        }

        @Override
        public Integer getAnswerNumber() {
            return answerNumber;
        }

        @Override
        public String getAnswer() {
            return answer;
        }

        @Override
        public Long getAnswerHeaderId() {
            return answerHeaderId;
        }

        @Override
        public Long getQuestionId() {
            return questionId;
        }

        @Override
        public Integer getQuestionSeqId() {
            return questionSeqId;
        }

        public void setQuestionSeqId(Integer questionSeqId) {
            this.questionSeqId = questionSeqId;
        }

        @Override
        public Long getQuestionnaireQuestionsId() {
            return questionnaireQuestionsId;
        }

        @Override
        public List<InternalAnswer> getParentAnswers() {
            return parentAnswers;
        }

        public void setId(Long id) {
            this.id = id;
        }

        public void setQuestionNumber(Integer questionNumber) {
            this.questionNumber = questionNumber;
        }

        public void setAnswerNumber(Integer answerNumber) {
            this.answerNumber = answerNumber;
        }

        public void setAnswer(String answer) {
            this.answer = answer;
        }

        public void setAnswerHeaderId(Long answerHeaderId) {
            this.answerHeaderId = answerHeaderId;
        }

        public void setQuestionId(Long questionId) {
            this.questionId = questionId;
        }

        public void setQuestionnaireQuestionsId(Long questionnaireQuestionsId) {
            this.questionnaireQuestionsId = questionnaireQuestionsId;
        }

        public void setParentAnswers(List<InternalAnswer> parentAnswers) {
            this.parentAnswers = parentAnswers;
        }
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

	@Override
	public List<Resource> getStylesheets() {
		return stylesheets;
	}

	public void setStylesheets(List<Resource> stylesheets) {
		this.stylesheets = stylesheets;
	}

	@Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }
}
