/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.edSF424Supplement40V40.EDSF424Supplement40Document;
import gov.grants.apply.forms.edSF424Supplement40V40.EDSF424Supplement40Document.EDSF424Supplement40;
import gov.grants.apply.forms.edSF424Supplement40V40.EDSF424Supplement40ProjectDirectorDataType;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.org.OrganizationContract;
import org.kuali.coeus.common.api.ynq.YnqConstant;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerHeaderContract;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.propdev.api.s2s.S2SConfigurationService;
import org.kuali.coeus.propdev.api.specialreview.ProposalSpecialReviewContract;
import org.kuali.coeus.propdev.api.specialreview.ProposalSpecialReviewExemptionContract;
import org.kuali.coeus.s2sgen.api.core.AuditError;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;

/**
 * This class is used to generate XML Document object for grants.gov
 * EDSF424SupplementV4.0 This form is generated using XMLBean API's generated
 * by compiling EDSF424SupplementV4.0 schema.
 */
@FormGenerator("EDSF424SupplementV4_0Generator")
public class EDSF424SupplementV4_0Generator extends
        EDSF424SupplementBaseGenerator<EDSF424Supplement40Document> implements S2SFormGeneratorPdfFillable<EDSF424Supplement40Document> {

    protected static final int IS_NOVICE_APPLICANT_QUESTION = -10150;
    protected static final int NIA_PREFERENCE_POINTS_QUESTION = -10151;
    protected static final int IS_SUBJECT_TO_BABAA_QUESTION = -10152;
    protected static final int CONSTRUCTION_BABAA_QUESTION = -10153;
    protected static final int REMODELING_BABAA_QUESTION = -10154;
    protected static final int BROADBAND_BABAA_QUESTION = -10155;
    protected static final int PAGE_NUMBERS_BABAA_QUESTION = -10156;
    protected static final int OPE_IDS_QUESTION = -10157;
    protected static final int NCES_SCHOOL_IDS_QUESTION = -10158;
    protected static final int NCES_LEA_SCHOOL_DISTRICT_IDS_QUESTION = -10159;


    @Value("http://apply.grants.gov/forms/ED_SF424_Supplement_4_0-V4.0")
    private String namespace;

    @Value("ED_SF424_Supplement_4_0-V4.0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/ED_SF424_Supplement-V4.0.fo.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/ED_SF424_Supplement_4_0-V4.0.pdf")
    private Resource pdfForm;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    @Autowired
    @Qualifier("s2SConfigurationService")
    private S2SConfigurationService s2SConfigurationService;

    protected static final String MISSING_PERCENTAGE_EFFORT_ERROR = "The Project Director's (Principal Investigator) Percentage Effort is required.";

    private EDSF424Supplement40Document getEDSF424Supplement() {
        EDSF424Supplement40Document edsf424SupplementDocument = EDSF424Supplement40Document.Factory.newInstance();
        EDSF424Supplement40 edsf424Supplement = EDSF424Supplement40.Factory.newInstance();

        edsf424Supplement.setFormVersion(FormVersion.v4_0.getVersion());
        ProposalPersonContract pi = getS2SProposalPersonService().getPrincipalInvestigator(pdDoc);
        edsf424Supplement.setProjectDirector(getProjectDirector(pi));
        setQuestionnaireAnswers(edsf424Supplement);
        setHumanSubjectsInformation(edsf424Supplement);
        setNarratives(edsf424Supplement);
        edsf424SupplementDocument.setEDSF424Supplement40(edsf424Supplement);
        return edsf424SupplementDocument;
    }

    private EDSF424Supplement40ProjectDirectorDataType getProjectDirector(ProposalPersonContract proposalPerson) {
        EDSF424Supplement40ProjectDirectorDataType projectDirector = EDSF424Supplement40ProjectDirectorDataType.Factory.newInstance();

        if (proposalPerson != null) {
            projectDirector.setName(globLibV20Generator.getHumanNameDataType(proposalPerson));

            if (StringUtils.isNotEmpty(proposalPerson.getOfficePhone())) {
                projectDirector.setPhone(proposalPerson.getOfficePhone());
            }

            if (StringUtils.isNotEmpty(proposalPerson.getEmailAddress())) {
                projectDirector.setEmail(proposalPerson.getEmailAddress());
            }

            if (StringUtils.isNotEmpty(proposalPerson.getFaxNumber())) {
                projectDirector.setFax(proposalPerson.getFaxNumber());
            }

            projectDirector.setAddress(globLibV20Generator.getAddressDataTypeV3(proposalPerson));

            if (proposalPerson.getPercentageEffort() != null) {
                projectDirector.setProjectDirectorLevelOfEffort(proposalPerson.getPercentageEffort().intValue());
            } else {
                getAuditErrors().add(new AuditError(AuditError.NO_FIELD_ERROR_KEY, MISSING_PERCENTAGE_EFFORT_ERROR, AuditError.GG_LINK));
            }
        }

        return projectDirector;
    }

    private void setHumanSubjectsInformation(EDSF424Supplement40 edsf424Supplement) {
        edsf424Supplement.setIsHumanResearch(YesNoDataType.N_NO);
        boolean areSpecialReviewsPresent = Boolean.FALSE;
        for (ProposalSpecialReviewContract specialReview : pdDoc.getDevelopmentProposal().getPropSpecialReviews()) {
            if (specialReview.getSpecialReviewType() != null && specialReview.getSpecialReviewType().getCode().equals(SPECIAL_REVIEW_CODE)) {
                edsf424Supplement.setIsHumanResearch(YesNoDataType.Y_YES);
                if (specialReview.getApprovalType() != null && specialReview.getApprovalType().getCode().equals(APPROVAL_TYPE_CODE)) {
                    edsf424Supplement.setIsHumanResearchExempt(YesNoDataType.Y_YES);
                    areSpecialReviewsPresent = Boolean.TRUE;

                    if (specialReview.getSpecialReviewExemptions() != null && specialReview.getSpecialReviewExemptions().size() > 0) {
                        for (ProposalSpecialReviewExemptionContract exemption : specialReview.getSpecialReviewExemptions()) {
                            switch (exemption.getExemptionType().getCode()) {
                                case "1":
                                    edsf424Supplement.setExemptionNumber1(YesNoDataType.Y_YES);
                                    break;
                                case "2":
                                    edsf424Supplement.setExemptionNumber2(YesNoDataType.Y_YES);
                                    break;
                                case "3":
                                    edsf424Supplement.setExemptionNumber3(YesNoDataType.Y_YES);
                                    break;
                                case "4":
                                    edsf424Supplement.setExemptionNumber4(YesNoDataType.Y_YES);
                                    break;
                                case "5":
                                    edsf424Supplement.setExemptionNumber5(YesNoDataType.Y_YES);
                                    break;
                                case "6":
                                    edsf424Supplement.setExemptionNumber6(YesNoDataType.Y_YES);
                                    break;
                                case "7":
                                    edsf424Supplement.setExemptionNumber7(YesNoDataType.Y_YES);
                                    break;
                                case "8":
                                    edsf424Supplement.setExemptionNumber8(YesNoDataType.Y_YES);
                                    break;
                            }
                        }
                    }
                }
            }
        }
        if (!areSpecialReviewsPresent) {
            populateAssuranceInformation(edsf424Supplement);
        }
    }

    protected void populateAssuranceInformation(EDSF424Supplement40 edsf424Supplement) {
        final OrganizationContract organization = pdDoc.getDevelopmentProposal().getApplicantOrganization().getOrganization();
        if (organization != null) {
            edsf424Supplement.setIsHumanResearchExempt(YesNoDataType.N_NO);
            if (organization.getHumanSubAssurance() != null) {
                edsf424Supplement.setAssuranceNumber(organization.getHumanSubAssurance());
            }
        }
    }

    private void setQuestionnaireAnswers(EDSF424Supplement40 edsf424Supplement) {
        final List<? extends AnswerHeaderContract> answerHeaders = getPropDevQuestionAnswerService().getQuestionnaireAnswerHeaders(pdDoc.getDevelopmentProposal().getProposalNumber(), getNamespace(), getFormName());

        answerHeaders
                .stream()
                .filter(Objects::nonNull)
                .flatMap(answerHeader -> answerHeader.getAnswers().stream())
                .filter(answer -> answer.getAnswer() != null)
                .forEach(answer -> {
                    switch (answer.getQuestionSeqId()) {
                        case IS_NOVICE_APPLICANT_QUESTION:
                            setIsNoviceApplicant(edsf424Supplement, answer.getAnswer());
                            break;
                        case NIA_PREFERENCE_POINTS_QUESTION:
                            edsf424Supplement.setNIAPreferencePoints(Integer.valueOf(answer.getAnswer()));
                            break;
                        case IS_SUBJECT_TO_BABAA_QUESTION:
                            setIsSubjectToBABAAQuestion(edsf424Supplement, answer.getAnswer());
                            break;
                        case CONSTRUCTION_BABAA_QUESTION:
                            setConstructionBABAAQuestion(edsf424Supplement, answer.getAnswer());
                            break;
                        case REMODELING_BABAA_QUESTION:
                            setRemodelingBABAAQuestion(edsf424Supplement, answer.getAnswer());
                            break;
                        case BROADBAND_BABAA_QUESTION:
                            setBroadbandBABAAQuestion(edsf424Supplement, answer.getAnswer());
                            break;
                        case PAGE_NUMBERS_BABAA_QUESTION:
                            edsf424Supplement.setPageNumbers(answer.getAnswer());
                            break;
                        case OPE_IDS_QUESTION:
                            edsf424Supplement.setOPEID(answer.getAnswer());
                            break;
                        case NCES_SCHOOL_IDS_QUESTION:
                            edsf424Supplement.setSchoolID(answer.getAnswer());
                            break;
                        case NCES_LEA_SCHOOL_DISTRICT_IDS_QUESTION:
                            edsf424Supplement.setDistrictID(answer.getAnswer());
                            break;
                    }
                });
    }

    private void setIsNoviceApplicant(EDSF424Supplement40 edsf424Supplement, String answer) {
        if (YnqConstant.YES.code().equals(answer)) {
            edsf424Supplement.setIsNoviceApplicant(YesNoDataType.Y_YES);
        } else if (YnqConstant.NO.code().equals(answer)) {
            edsf424Supplement.setIsNoviceApplicant(YesNoDataType.N_NO);
        } else if (YnqConstant.NA.code().equals(answer)) {
            edsf424Supplement.setNotApplicable(YesNoDataType.Y_YES);
        }
    }

    private void setIsSubjectToBABAAQuestion(EDSF424Supplement40 edsf424Supplement, String answer) {
        if (YnqConstant.YES.code().equals(answer)) {
            edsf424Supplement.setSubjecttoBABAA(YesNoDataType.Y_YES);
        } else if (YnqConstant.NO.code().equals(answer)) {
            edsf424Supplement.setSubjecttoBABAA(YesNoDataType.N_NO);
        }
    }

    private void setConstructionBABAAQuestion(EDSF424Supplement40 edsf424Supplement, String answer) {
        if (YnqConstant.YES.code().equals(answer)) {
            edsf424Supplement.setConstruction(YesNoDataType.Y_YES);
        } else if (YnqConstant.NO.code().equals(answer)) {
            edsf424Supplement.setConstruction(YesNoDataType.N_NO);
        }
    }

    private void setRemodelingBABAAQuestion(EDSF424Supplement40 edsf424Supplement, String answer) {
        if (YnqConstant.YES.code().equals(answer)) {
            edsf424Supplement.setRemodeling(YesNoDataType.Y_YES);
        } else if (YnqConstant.NO.code().equals(answer)) {
            edsf424Supplement.setRemodeling(YesNoDataType.N_NO);
        }
    }

    private void setBroadbandBABAAQuestion(EDSF424Supplement40 edsf424Supplement, String answer) {
        if (YnqConstant.YES.code().equals(answer)) {
            edsf424Supplement.setBroadbandInfrastructure(YesNoDataType.Y_YES);
        } else if (YnqConstant.NO.code().equals(answer)) {
            edsf424Supplement.setBroadbandInfrastructure(YesNoDataType.N_NO);
        }
    }

    protected void setNarratives(EDSF424Supplement40 edsf424Supplement) {
        final Optional<? extends NarrativeContract> edsf424SupplementNarrativeOptional = pdDoc.getDevelopmentProposal().getNarratives().stream()
                .filter(narrative -> narrative.getNarrativeType().getCode() != null && Integer.parseInt(narrative.getNarrativeType().getCode()) == NARRATIVE_TYPE_ED_SF424_SUPPLIMENT)
                .findFirst();

        if (edsf424SupplementNarrativeOptional.isPresent()) {
            final AttachedFileDataType attachment = addAttachedFileType(edsf424SupplementNarrativeOptional.get());

            if (attachment != null) {
                edsf424Supplement.setAttachment(attachment);
            }
        }
    }

    /**
     * This method creates {@link XmlObject} of type
     * {@link EDSF424Supplement40Document} by populating data from the given
     * {@link ProposalDevelopmentDocumentContract}
     *
     * @param proposalDevelopmentDocument for which the {@link XmlObject} needs to be created
     * @return {@link XmlObject} which is generated using the given
     * {@link ProposalDevelopmentDocumentContract}
     */
    @Override
    public EDSF424Supplement40Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        pdDoc = proposalDevelopmentDocument;
        return getEDSF424Supplement();
    }

    public S2SConfigurationService getS2SConfigurationService() {
        return s2SConfigurationService;
    }

    public void setS2SConfigurationService(S2SConfigurationService s2SConfigurationService) {
        this.s2SConfigurationService = s2SConfigurationService;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(EDSF424Supplement40Document form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {
            if (form.getEDSF424Supplement40().getAttachment() != null && a.getContentId().equals(form.getEDSF424Supplement40().getAttachment().getFileLocation().getHref())) {
                return entry("ED_SF424_Supplement_4_0_P2.optionalFile0", a);
            }

            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<EDSF424Supplement40Document> factory() {
        return EDSF424Supplement40Document.Factory;
    }
}
