/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.rrPersonalDataV11.DirectorType;
import gov.grants.apply.forms.rrPersonalDataV11.RRPersonalDataDocument;
import gov.grants.apply.forms.rrPersonalDataV11.RRPersonalDataDocument.RRPersonalData;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Class for generating the XML object for grants.gov RRPersonalDataV1.1. Form is generated using XMLBean classes and is based on
 * RRPersonalData schema.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
@FormGenerator("RRPersonalDataV1_1Generator")
public class RRPersonalDataV1_1Generator extends RRPersonalDataBaseGenerator<RRPersonalDataDocument> {

    @Value("http://apply.grants.gov/forms/RR_PersonalData-V1.1")
    private String namespace;

    @Value("RR_PersonalData-V1.1")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/RR_PersonalData-V1.1.fo.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/RR_PersonalData-V1.1.pdf")
    private Resource pdfForm;

    @Value("157")
    private int sortIndex;

    /**
     * 
     * This method gives the personal information of ProjectDirector and CoProjectDirector
     * 
     * @return rrPersonalDataDocument {@link XmlObject} of type RRPersonalDataDocument.
     */
    private RRPersonalDataDocument getRRPersonalData() {

        RRPersonalDataDocument rrPersonalDataDocument = RRPersonalDataDocument.Factory.newInstance();
        RRPersonalData rrPersonalData = RRPersonalData.Factory.newInstance();
        rrPersonalData.setFormVersion(FormVersion.v1_1.getVersion());
        rrPersonalData.setProjectDirector(getProjectDirectorType());
        rrPersonalData.setCoProjectDirectorArray(getCoProjectDirectoryType());
        rrPersonalDataDocument.setRRPersonalData(rrPersonalData);
        return rrPersonalDataDocument;
    }

    /**
     * 
     * This method is used to get Personal details of Principal Investigator
     * 
     * @return DirectorType principal investigator details.
     */
    private DirectorType getProjectDirectorType() {

        DirectorType directorType = DirectorType.Factory.newInstance();
        ProposalPersonContract PI = s2SProposalPersonService.getPrincipalInvestigator(pdDoc);
        if (PI != null) {
            directorType.setName(globLibV20Generator.getHumanNameDataType(PI));
        }
        return directorType;
    }

    /**
     * 
     * This method is used to get List of Personal details of Co-Investigator
     * 
     * @return DirectorType[] Array of director types.
     */
    private DirectorType[] getCoProjectDirectoryType() {

        List<DirectorType> directorTypeList = new ArrayList<>();
        if (pdDoc.getDevelopmentProposal().getProposalPersons() != null) {
            ProposalPersonContract CoPI = null;
            for (ProposalPersonContract proposalPerson : pdDoc.getDevelopmentProposal().getProposalPersons()) {
                DirectorType coDirectorType = DirectorType.Factory.newInstance();
                if (proposalPerson.getProposalPersonRoleId() != null) {
                    if (KEYPERSON_TYPE_C0_INVESTIGATOR.equals(proposalPerson.getProposalPersonRoleId())) {
                        CoPI = proposalPerson;
                        coDirectorType.setName(globLibV20Generator.getHumanNameDataType(CoPI));
                        directorTypeList.add(coDirectorType);
                    }
                }
            }
        }
        return directorTypeList.toArray(new DirectorType[0]);

    }

    /**
     * This method creates {@link XmlObject} of type {@link RRPersonalDataDocument} by populating data from the given
     * {@link ProposalDevelopmentDocumentContract}
     * 
     * @param proposalDevelopmentDocument for which the {@link XmlObject} needs to be created
     * @return {@link XmlObject} which is generated using the given {@link ProposalDevelopmentDocumentContract}
     */
    @Override
    public RRPersonalDataDocument getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getRRPersonalData();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(RRPersonalDataDocument form, List<AttachmentData> attachments) {
        return new Attachments(Collections.emptyMap(), attachments);
    }

    @Override
    public DocumentFactory<RRPersonalDataDocument> factory() {
        return RRPersonalDataDocument.Factory;
    }
}
