/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.sf424Mandatory30V30.SF424Mandatory30Document;
import gov.grants.apply.forms.sf424Mandatory30V30.SF424Mandatory30Document.SF424Mandatory30;
import gov.grants.apply.forms.sf424Mandatory30V30.SF424Mandatory30Document.SF424Mandatory30.Frequency;
import gov.grants.apply.forms.sf424Mandatory30V30.SF424Mandatory30Document.SF424Mandatory30.SubmissionUnderReview;
import gov.grants.apply.forms.sf424Mandatory30V30.SF424Mandatory30Document.SF424Mandatory30.TypeOfSubmission;
import gov.grants.apply.forms.sf424Mandatory30V30.SF424Mandatory30Document.SF424Mandatory30.Version;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.attachmentsV10.AttachmentGroupMin0Max100DataType;
import gov.grants.apply.system.globalLibraryV20.ApplicantTypeCodeDataType;
import gov.grants.apply.system.globalLibraryV20.OrganizationDataTypeV4;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.org.OrganizationContract;
import org.kuali.coeus.common.api.org.OrganizationYnqContract;
import org.kuali.coeus.common.api.org.type.OrganizationTypeContract;
import org.kuali.coeus.common.api.rolodex.RolodexContract;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerHeaderContract;
import org.kuali.coeus.propdev.api.budget.ProposalDevelopmentBudgetExtContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.location.ProposalSiteContract;
import org.kuali.coeus.propdev.api.s2s.S2SConfigurationService;
import org.kuali.coeus.s2sgen.api.budget.S2SBudgetDto;
import org.kuali.coeus.s2sgen.api.budget.S2SBudgetInfoService;
import org.kuali.coeus.s2sgen.api.core.ConfigurationConstants;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.datetime.S2SDateTimeService;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonDto;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonService;
import org.kuali.coeus.s2sgen.impl.util.FieldValueConstants;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;

@FormGenerator("SF424Mandatory3_0_V3_0Generator")
public class SF424Mandatory3_0_V3_0Generator extends CommonSF424BaseGenerator<SF424Mandatory30Document> implements S2SFormGeneratorPdfFillable<SF424Mandatory30Document> {

    private static final String CONTACT_TYPE_I = "I";
    private static final int PRIMARY_TITLE_MAX_LENGTH = 45;
    private static final String YNQ_STATE_NOT_SELECTED = "Not Selected";
    private static final String YNQ_STATE_NOT_COVERED = "Not Covered";
    private final static String PROPOSAL_YNQ_FEDERAL_DEBTS = "I7";
    private static final String ORGANIZATION_YNQ_ANSWER_YES = "Y";
    private static final String DEBT_EXPLANATION_ATTACHMENT = "136";

    @Value("http://apply.grants.gov/forms/SF424_Mandatory_3_0-V3.0")
    private String namespace;

    @Value("SF424Mandatory3_0_V3_0")
    private String formName;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/SF424_Mandatory_3_0-V3.0.pdf")
    private Resource pdfForm;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    @Autowired
    @Qualifier("s2SConfigurationService")
    protected S2SConfigurationService s2SConfigurationService;

    @Autowired
    @Qualifier("departmentalPersonService")
    protected DepartmentalPersonService departmentalPersonService;

    @Autowired
    @Qualifier("s2SBudgetInfoService")
    protected S2SBudgetInfoService s2sBudgetInfoService;

    @Autowired
    @Qualifier("s2SDateTimeService")
    protected S2SDateTimeService s2SDateTimeService;

    private SF424Mandatory30Document getSF424Mandatory30Document() {
        SF424Mandatory30Document sF424Mandatory30Document = SF424Mandatory30Document.Factory.newInstance();
        sF424Mandatory30Document.setSF424Mandatory30(getSF424Mandatory30());
        return sF424Mandatory30Document;
    }

    private SF424Mandatory30 getSF424Mandatory30() {

        SF424Mandatory30 sF424Mandatory30 = SF424Mandatory30.Factory.newInstance();

        final List<? extends AnswerHeaderContract> answerHeaders = getPropDevQuestionAnswerService().getQuestionnaireAnswerHeaders(pdDoc.getDevelopmentProposal().getProposalNumber());
        setTypeOfSubmission(sF424Mandatory30, answerHeaders);
        setFrequency(sF424Mandatory30, answerHeaders);
        setConsolidatedRequest(sF424Mandatory30, answerHeaders);
        setVersion(sF424Mandatory30);
        setRecievedDate(sF424Mandatory30);
        setApplicantId(sF424Mandatory30);
        setFederalAwardId(sF424Mandatory30);

        final ProposalSiteContract proposalSite = pdDoc.getDevelopmentProposal().getApplicantOrganization();
        final OrganizationContract applicantOrganization = proposalSite.getOrganization();
        setApplicationInformation(sF424Mandatory30, proposalSite);
        setTypeOfApplicant(sF424Mandatory30, applicantOrganization);
        setNameOfFedAgency(sF424Mandatory30);
        setCfda(sF424Mandatory30);
        setTitle(sF424Mandatory30);
        setAreaAffected(sF424Mandatory30, answerHeaders);
        setCongressionalDistricts(sF424Mandatory30, applicantOrganization);
        setFundingPeriods(sF424Mandatory30);
        setEstimatedFunding(sF424Mandatory30);
        setSubmissionReview(sF424Mandatory30, answerHeaders);
        setDelinquencyInfo(sF424Mandatory30, applicantOrganization);
        setSigningInfo(sF424Mandatory30);

        sF424Mandatory30.setFormVersion(FormVersion.v3_0.getVersion());

        return sF424Mandatory30;
    }

    private void setSigningInfo(SF424Mandatory30 sF424Mandatory30) {
        //18 signing this application
        sF424Mandatory30.setApplicationCertification(YesNoDataType.Y_YES);
        final DepartmentalPersonDto aorInfo = departmentalPersonService.getDepartmentalPerson(pdDoc);
        sF424Mandatory30.setAuthorizedRepresentativeName(globLibV20Generator.getHumanNameDataType(aorInfo));
        if (StringUtils.isNotBlank(aorInfo.getPrimaryTitle())) {
            sF424Mandatory30.setAuthorizedRepresentativeTitle(StringUtils.substring(aorInfo.getPrimaryTitle(), 0, PRIMARY_TITLE_MAX_LENGTH));
        }
        if (StringUtils.isNotBlank(aorInfo.getOfficePhone())) {
            sF424Mandatory30.setAuthorizedRepresentativePhoneNumber(aorInfo.getOfficePhone());
        }
        if (StringUtils.isNotBlank(aorInfo.getFaxNumber())) {
            sF424Mandatory30.setAuthorizedRepresentativeFax(aorInfo.getFaxNumber());
        }
        if (StringUtils.isNotBlank(aorInfo.getEmailAddress())) {
            sF424Mandatory30.setAuthorizedRepresentativeEmail(aorInfo.getEmailAddress());
        }

        sF424Mandatory30.setAuthorizedRepresentativeSignature(aorInfo.getFullName());
        sF424Mandatory30.setAuthorizedRepresentativeSignatureDate(aorInfo.getSubmittedDate());
    }

    private void setDelinquencyInfo(SF424Mandatory30 sF424Mandatory30, OrganizationContract applicantOrganization) {
        //17 Is The Applicant Delinquent
        if (applicantOrganization != null) {
            final Optional<YesNoDataType.Enum> yesNo = applicantOrganization.getOrganizationYnqs().stream()
                    .filter(orgYnq -> PROPOSAL_YNQ_FEDERAL_DEBTS.equals(orgYnq.getQuestionId()))
                    .map(OrganizationYnqContract::getAnswer)
                    .filter(StringUtils::isNotBlank)
                    .map(answer -> ORGANIZATION_YNQ_ANSWER_YES.equals(answer) ? YesNoDataType.Y_YES : YesNoDataType.N_NO)
                    .findFirst();
            sF424Mandatory30.setDelinquentOnFederalDebt(yesNo.orElse(YesNoDataType.N_NO));
        }

        final AttachedFileDataType[] attachedFileDataTypes = getAttachedFileDataTypes(DEBT_EXPLANATION_ATTACHMENT);
        if (ArrayUtils.isNotEmpty(attachedFileDataTypes)) {
            sF424Mandatory30.setDelinquentFederalDebtExplanation("See attachment for explanation");
            AttachmentGroupMin0Max100DataType attachments = AttachmentGroupMin0Max100DataType.Factory
                    .newInstance();
            attachments.setAttachedFileArray(attachedFileDataTypes);
            sF424Mandatory30.setAttachments(attachments);
        }
    }

    private void setSubmissionReview(SF424Mandatory30 sF424Mandatory30, List<? extends AnswerHeaderContract> answerHeaders) {
        //16 IS SUBMISSION SUBJECT TO REVIEW
        final String dateAvail = getAnswer(129, answerHeaders);
        if ("Y".equals(dateAvail)) {
            sF424Mandatory30.setSubmissionUnderReview(SubmissionUnderReview.STATE_REVIEW_AVAILABLE);
            final String reviewDate = getAnswer(130, answerHeaders);
            if (StringUtils.isNotBlank(reviewDate)) {

                sF424Mandatory30.setStateReviewAvailableDate(s2SDateTimeService.convertDateStringToCalendar(reviewDate));
            }
        } else if ("N".equals(dateAvail)){
            final String reviewData = getAnswer(131, answerHeaders);
            if (YNQ_STATE_NOT_SELECTED.equals(reviewData)) {
                sF424Mandatory30.setSubmissionUnderReview(SubmissionUnderReview.STATE_REVIEW_NOT_SELECTED);
            } else if (YNQ_STATE_NOT_COVERED.equals(reviewData)) {
                sF424Mandatory30.setSubmissionUnderReview(SubmissionUnderReview.STATE_REVIEW_NOT_NEEDED);
            }
        }
    }

    private void setEstimatedFunding(SF424Mandatory30 sF424Mandatory30)  {
        //15 Estimated Funding
        final ProposalDevelopmentBudgetExtContract budget = getS2SCommonBudgetService().getBudget(pdDoc.getDevelopmentProposal());
        if (budget != null) {
            final S2SBudgetDto budgetSummary = getS2sBudgetInfoService().getBudgetInfo(pdDoc);
            sF424Mandatory30.setEstimatedFundingFederal(budgetSummary.getCumTotalCosts().bigDecimalValue());
            sF424Mandatory30.setEstimatedFundingMatch(budgetSummary.getCumTotalCostSharing().bigDecimalValue());
        } else {
            sF424Mandatory30.setEstimatedFundingFederal(BigDecimal.ZERO);
            sF424Mandatory30.setEstimatedFundingMatch(BigDecimal.ZERO);
        }
    }

    private void setFundingPeriods(SF424Mandatory30 sF424Mandatory30) {
        //14 Funding Period
        sF424Mandatory30.setFundingPeriodStartDate(DateUtils.toCalendar(pdDoc.getDevelopmentProposal().getRequestedStartDateInitial()));
        sF424Mandatory30.setFundingPeriodEndDate(DateUtils.toCalendar(pdDoc.getDevelopmentProposal().getRequestedEndDateInitial()));
    }

    private void setCongressionalDistricts(SF424Mandatory30 sF424Mandatory30, OrganizationContract applicantOrganization) {
        //13 Congressional Districts
        if (applicantOrganization != null) {
            final String congressionalDistrict = StringUtils.isBlank(applicantOrganization.getCongressionalDistrict()) ?
                    FieldValueConstants.VALUE_UNKNOWN : StringUtils.substring(applicantOrganization.getCongressionalDistrict(), 0, CONGRESSIONAL_DISTRICT_MAX_LENGTH);
            sF424Mandatory30.setCongressionalDistrictApplicant(congressionalDistrict);
        }

        final ProposalSiteContract perfOrganization = pdDoc.getDevelopmentProposal().getPerformingOrganization();
        if (perfOrganization != null) {
            final String congDistrictProject = StringUtils.isBlank(perfOrganization.getFirstCongressionalDistrictName()) ?
                    FieldValueConstants.VALUE_UNKNOWN : StringUtils.substring(perfOrganization.getFirstCongressionalDistrictName(), 0, CONGRESSIONAL_DISTRICT_MAX_LENGTH);
            sF424Mandatory30.setCongressionalDistrictProject(congDistrictProject);
        }
    }

    private void setAreaAffected(SF424Mandatory30 sF424Mandatory30, List<? extends AnswerHeaderContract> answerHeaders) {
        //12 Areas Affected by Funding
        final String affectedAreasOfFunding = getAnswer(-10140, answerHeaders);
        if (StringUtils.isNotBlank(affectedAreasOfFunding) && "Y".equals(affectedAreasOfFunding)) {
            final String affectedAreasOfFundingDesc = getAnswer(-10141, answerHeaders);
            if (StringUtils.isNotBlank(affectedAreasOfFundingDesc)) {
                sF424Mandatory30.setAreasAffectedByFunding(affectedAreasOfFundingDesc);
            }
        }
    }

    private void setTitle(SF424Mandatory30 sF424Mandatory30) {
        //11 Descriptive Title of Applicant's Project
        if (StringUtils.isNotBlank(pdDoc.getDevelopmentProposal().getTitle())) {
            sF424Mandatory30.setDescriptiveProjectTitle(pdDoc.getDevelopmentProposal().getTitle());
        }
    }

    private void setCfda(SF424Mandatory30 sF424Mandatory30) {
        final String cfdaNumber = getFirstCfdaNumber();
        if (cfdaNumber != null) {
            sF424Mandatory30.setCFDANumber(cfdaNumber);
        }
    }

    private void setNameOfFedAgency(SF424Mandatory30 sF424Mandatory30) {
        //9 Name of Federal Agency
        sF424Mandatory30.setAgencyName(getFederalAgencyName());
    }

    private void setTypeOfApplicant(SF424Mandatory30 sF424Mandatory30, OrganizationContract applicantOrganization) {
        //8a TYPE OF APPLICANT
        if (applicantOrganization != null) {
            final List<? extends OrganizationTypeContract> organizationTypes = applicantOrganization.getOrganizationTypes();
            final List<Integer> organizationTypeCodes = organizationTypes.stream()
                    .map(organizationType -> organizationType.getOrganizationTypeList().getCode())
                    .collect(Collectors.toList());

            for (int organizationTypeCode : organizationTypeCodes)         {
                if (1 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.C_CITY_OR_TOWNSHIP_GOVERNMENT);
                    break;
                } else if (2 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.A_STATE_GOVERNMENT);
                    break;
                } else if (3 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.X_OTHER_SPECIFY);
                    sF424Mandatory30.setOtherTypeApplicant("Federal Government");
                    break;
                } else if (4 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.M_NONPROFIT_WITH_501_C_3_IRS_STATUS_OTHER_THAN_INSTITUTION_OF_HIGHER_EDUCATION);
                    break;
                } else if (5 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.N_NONPROFIT_WITHOUT_501_C_3_IRS_STATUS_OTHER_THAN_INSTITUTION_OF_HIGHER_EDUCATION);
                    break;
                } else if (6 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.Q_FOR_PROFIT_ORGANIZATION_OTHER_THAN_SMALL_BUSINESS);
                    break;
                } else if (7 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.X_OTHER_SPECIFY);
                    break;
                } else if (8 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.I_INDIAN_NATIVE_AMERICAN_TRIBAL_GOVERNMENT_FEDERALLY_RECOGNIZED);
                    break;
                } else if (9 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.P_INDIVIDUAL);
                    break;
                } else if (10 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.O_PRIVATE_INSTITUTION_OF_HIGHER_EDUCATION);
                    break;
                } else if (11 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.R_SMALL_BUSINESS);
                    break;
                } else if (14 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.X_OTHER_SPECIFY);
                    sF424Mandatory30.setOtherTypeApplicant("Socially and Economically Disadvantaged");
                    break;
                } else if (15 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.X_OTHER_SPECIFY);
                    sF424Mandatory30.setOtherTypeApplicant("Women owned");
                    break;
                } else if (21 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.H_PUBLIC_STATE_CONTROLLED_INSTITUTION_OF_HIGHER_EDUCATION);
                    break;
                } else if (22 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.B_COUNTY_GOVERNMENT);
                    break;
                } else if (23 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.D_SPECIAL_DISTRICT_GOVERNMENT);
                    break;
                } else if (24 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.G_INDEPENDENT_SCHOOL_DISTRICT);
                    break;
                } else if (25 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.L_PUBLIC_INDIAN_HOUSING_AUTHORITY);
                    break;
                } else if (26 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.J_INDIAN_NATIVE_AMERICAN_TRIBAL_GOVERNMENT_OTHER_THAN_FEDERALLY_RECOGNIZED);
                    break;
                } else if (100 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.E_REGIONAL_ORGANIZATION);
                    break;
                } else if (101 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.F_U_S_TERRITORY_OR_POSSESSION);
                    break;
                } else if (102 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.K_INDIAN_NATIVE_AMERICAN_TRIBALLY_DESIGNATED_ORGANIZATION);
                    break;
                } else if (103 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.S_HISPANIC_SERVING_INSTITUTION);
                    break;
                } else if (104 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.T_HISTORICALLY_BLACK_COLLEGES_AND_UNIVERSITIES_HBC_US);
                    break;
                } else if (105 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.U_TRIBALLY_CONTROLLED_COLLEGES_AND_UNIVERSITIES_TCC_US);
                    break;
                } else if (106 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.V_ALASKA_NATIVE_AND_NATIVE_HAWAIIAN_SERVING_INSTITUTIONS);
                    break;
                } else if (107 == organizationTypeCode) {
                    sF424Mandatory30.setApplicantType(ApplicantTypeCodeDataType.W_NON_DOMESTIC_NON_US_ENTITY);
                    break;
                }
            }
        }
    }

    private void setApplicationInformation(SF424Mandatory30 sF424Mandatory30, ProposalSiteContract proposalSite) {
        OrganizationContract applicantOrganization = proposalSite.getOrganization();
        //7 APPLICANT INFORMATION
        final OrganizationDataTypeV4 organization = OrganizationDataTypeV4.Factory.newInstance();
        if (applicantOrganization != null) {
            if (StringUtils.isNotBlank(proposalSite.getLocationName())) {
                organization.setOrganizationName(StringUtils.substring(proposalSite.getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
            }
            if (StringUtils.isNotBlank(applicantOrganization.getFederalEmployerId())) {
                organization.setEIN(applicantOrganization.getFederalEmployerId());
            }
            if (StringUtils.isNotBlank(applicantOrganization.getUei())) {
                organization.setSAMUEI(applicantOrganization.getUei());
            }
        }

        final RolodexContract applicantOrganizationRolodex = proposalSite.getRolodex();
        if (applicantOrganizationRolodex != null) {
            organization.setAddress(globLibV20Generator.getAddressDataTypeV3(proposalSite));
        }
        sF424Mandatory30.setOrganization(organization);

        if (CONTACT_TYPE_I.equals(s2SConfigurationService.getValueAsString(ConfigurationConstants.PROPOSAL_CONTACT_TYPE))) {
            if (applicantOrganizationRolodex != null) {
                sF424Mandatory30.setContactName(globLibV20Generator.getHumanNameDataType(applicantOrganizationRolodex));
                if (StringUtils.isNotBlank(applicantOrganizationRolodex.getPhoneNumber())) {
                    sF424Mandatory30.setContactPhoneNumber(applicantOrganizationRolodex.getPhoneNumber());
                }
                if (StringUtils.isNotBlank(applicantOrganizationRolodex.getFaxNumber())) {
                    sF424Mandatory30.setContactFax(applicantOrganizationRolodex.getFaxNumber());
                }
                if(StringUtils.isNotBlank(applicantOrganizationRolodex.getEmailAddress())) {
                    sF424Mandatory30.setContactEmail(applicantOrganizationRolodex.getEmailAddress());
                }
            }
        } else {
            // contact will come from unit or unit_administrators
            final DepartmentalPersonDto depPerson = departmentalPersonService.getContactPerson(pdDoc);
            if (depPerson != null) {
                sF424Mandatory30.setContactName(globLibV20Generator.getHumanNameDataType(depPerson));
                if (StringUtils.isNotEmpty(depPerson.getPrimaryTitle())) {
                    sF424Mandatory30.setContactTitle(StringUtils.substring(depPerson.getPrimaryTitle(), 0, PRIMARY_TITLE_MAX_LENGTH));
                }
                if (StringUtils.isNotBlank(depPerson.getOfficePhone())) {
                    sF424Mandatory30.setContactPhoneNumber(depPerson.getOfficePhone());
                }
                if (StringUtils.isNotEmpty(depPerson.getFaxNumber())) {
                    sF424Mandatory30.setContactFax(depPerson.getFaxNumber());
                }
                if (StringUtils.isNotEmpty(depPerson.getEmailAddress())) {
                    sF424Mandatory30.setContactEmail(depPerson.getEmailAddress());
                }
            }
        }
    }

    private void setFederalAwardId(SF424Mandatory30 sF424Mandatory30) {
        //4b Federal Award Identifier
        if (StringUtils.isNotBlank(pdDoc.getDevelopmentProposal().getSponsorProposalNumber())) {
            sF424Mandatory30.setFederalAwardID(StringUtils.left(pdDoc.getDevelopmentProposal().getSponsorProposalNumber().trim(), FED_AWARD_ID_MAX_LENGTH));
        }
    }

    private void setApplicantId(SF424Mandatory30 sF424Mandatory30) {
        //3 Applicant Identifier
        sF424Mandatory30.setApplicantID(pdDoc.getDevelopmentProposal().getProposalNumber());
    }

    private void setRecievedDate(SF424Mandatory30 sF424Mandatory30) {
        //2 Received Date
        sF424Mandatory30.setReceivedDate(DateUtils.toCalendar(pdDoc.getDevelopmentProposal().getRequestedStartDateInitial()));
    }

    private void setVersion(SF424Mandatory30 sF424Mandatory30) {
        //1d Version
        final String proposalTypeCode = pdDoc.getDevelopmentProposal().getProposalType().getCode();
        if (s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_NEW).contains(proposalTypeCode)) {
            sF424Mandatory30.setVersion(Version.INITIAL);
        } else if (s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_RESUBMISSION).contains(proposalTypeCode)) {
            sF424Mandatory30.setVersion(Version.RESUBMISSION);
        } else if (s2SConfigurationService.getValuesFromCommaSeparatedParam(ConfigurationConstants.PROPOSAL_TYPE_CODE_REVISION).contains(proposalTypeCode)) {
            sF424Mandatory30.setVersion(Version.REVISION);
        }
    }

    private void setConsolidatedRequest(SF424Mandatory30 sF424Mandatory30, List<? extends AnswerHeaderContract> answerHeaders) {
        //1c Consolidated Application/Plan/Funding Request
        final String consolidated = getAnswer(-10138, answerHeaders);
        if (StringUtils.isNotBlank(consolidated)) {
            final YesNoDataType.Enum consolidatedEnum = "Y".equals(consolidated) ? YesNoDataType.Y_YES : YesNoDataType.N_NO;
            sF424Mandatory30.setConsolidatedRequest(consolidatedEnum);
            if (consolidatedEnum.equals(YesNoDataType.Y_YES)) {
                final String consolidatedExplanation = getAnswer(-10139, answerHeaders);
                if (StringUtils.isNotBlank(consolidatedExplanation)) {
                    sF424Mandatory30.setConsolidatedRequestExplanation(consolidatedExplanation);
                }
            }
        }
    }

    private void setFrequency(SF424Mandatory30 sF424Mandatory30, List<? extends AnswerHeaderContract> answerHeaders) {
        //1b Frequency
        final String frequency = getAnswer(-10136, answerHeaders);
        if (StringUtils.isNotBlank(frequency)) {
            final Frequency.Enum frequencyEnum = Frequency.Enum.forString(frequency);
            sF424Mandatory30.setFrequency(frequencyEnum);

            if (frequencyEnum.equals(Frequency.OTHER)) {
                final String frequencyOtherDesc = getAnswer(-10137, answerHeaders);
                if (StringUtils.isNotBlank(frequencyOtherDesc)) {
                    sF424Mandatory30.setFrequencyOtherDescription(frequencyOtherDesc);
                }
            }
        }
    }

    private void setTypeOfSubmission(SF424Mandatory30 sF424Mandatory30, List<? extends AnswerHeaderContract> answerHeaders) {
        //1a Type of Submission
        final String typeOfSubmission = getAnswer(-10134, answerHeaders);
        if (StringUtils.isNotBlank(typeOfSubmission)) {
            final TypeOfSubmission.Enum typeOfSubmissionEnum = TypeOfSubmission.Enum.forString(typeOfSubmission);
            sF424Mandatory30.setTypeOfSubmission(typeOfSubmissionEnum);

            if (typeOfSubmissionEnum.equals(TypeOfSubmission.OTHER)) {
                final String typeOfSubmissionOtherDesc = getAnswer(-10135, answerHeaders);
                if (StringUtils.isNotBlank(typeOfSubmissionOtherDesc)) {
                    sF424Mandatory30.setTypeOfSubmissionOtherDescription(typeOfSubmissionOtherDesc);
                }
            }
        }
    }

    @Override
    public SF424Mandatory30Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;

        return getSF424Mandatory30Document();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    public S2SConfigurationService getS2SConfigurationService() {
        return s2SConfigurationService;
    }

    public void setS2SConfigurationService(S2SConfigurationService s2SConfigurationService) {
        this.s2SConfigurationService = s2SConfigurationService;
    }

    public DepartmentalPersonService getDepartmentalPersonService() {
        return departmentalPersonService;
    }

    public void setDepartmentalPersonService(DepartmentalPersonService departmentalPersonService) {
        this.departmentalPersonService = departmentalPersonService;
    }

    public S2SBudgetInfoService getS2sBudgetInfoService() {
        return s2sBudgetInfoService;
    }

    public void setS2sBudgetInfoService(S2SBudgetInfoService s2sBudgetInfoService) {
        this.s2sBudgetInfoService = s2sBudgetInfoService;
    }

    public S2SDateTimeService getS2SDateTimeService() {
        return s2SDateTimeService;
    }

    public void setS2SDateTimeService(S2SDateTimeService s2SDateTimeService) {
        this.s2SDateTimeService = s2SDateTimeService;
    }

    @Override
    public Attachments getMappedAttachments(SF424Mandatory30Document form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {

            final AttachedFileDataType congressionalDistricts = form.getSF424Mandatory30().getAdditionalCongressionalDistricts();
            if (congressionalDistricts != null && a.getContentId().equals(congressionalDistricts.getFileLocation().getHref())) {
                return entry("SF424_Mandatory_3_0_P2.optionalFile0", a);
            }

            final AttachmentGroupMin0Max100DataType supportingDocuments = form.getSF424Mandatory30().getAttachments();
            if (supportingDocuments != null && supportingDocuments.getAttachedFileList() != null) {
                final List<AttachedFileDataType> atts = supportingDocuments.getAttachedFileList();
                for (int i = 0; i < atts.size(); i++) {
                    final AttachedFileDataType att = atts.get(i);
                    if (att != null && a.getContentId().equals(att.getFileLocation().getHref())){
                        return entry("SF424_Mandatory_3_0_P3.optionalFile" + i, a);
                    }
                }
            }

            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<SF424Mandatory30Document> factory() {
        return SF424Mandatory30Document.Factory;
    }

    @Override
    public boolean supportsXslTransform() {
        return false;
    }
}
