/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.sf42430V30.SF42430Document;
import gov.grants.apply.forms.sf42430V30.SF42430Document.SF42430;
import gov.grants.apply.forms.sf42430V30.SF42430Document.SF42430.ApplicationType;
import gov.grants.apply.forms.sf42430V30.SF42430Document.SF42430.RevisionType;
import gov.grants.apply.forms.sf42430V30.SF42430Document.SF42430.StateReview;
import gov.grants.apply.forms.sf42430V30.SF42430Document.SF42430.SubmissionType;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.attachmentsV10.AttachmentGroupMin0Max100DataType;
import gov.grants.apply.system.globalLibraryV20.ApplicantTypeCodeDataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.org.OrganizationContract;
import org.kuali.coeus.common.api.org.OrganizationYnqContract;
import org.kuali.coeus.common.api.org.type.OrganizationTypeContract;
import org.kuali.coeus.common.api.unit.UnitContract;
import org.kuali.coeus.common.budget.api.income.BudgetProjectIncomeContract;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemCalculatedAmountContract;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemContract;
import org.kuali.coeus.common.budget.api.period.BudgetPeriodContract;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.budget.ProposalDevelopmentBudgetExtContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.location.ProposalSiteContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.propdev.api.s2s.S2sOpportunityContract;
import org.kuali.coeus.propdev.api.s2s.S2sSubmissionTypeContract;
import org.kuali.coeus.s2sgen.api.core.ConfigurationConstants;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonDto;
import org.kuali.coeus.s2sgen.impl.person.S2sDivisionService;
import org.kuali.coeus.s2sgen.impl.util.FieldValueConstants;
import org.kuali.coeus.sys.api.model.ScaleTwoDecimal;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;

@FormGenerator("SF424V3_0Generator")
public class SF424V3_0Generator extends SF424BaseGenerator<SF42430Document> implements S2SFormGeneratorPdfFillable<SF42430Document> {

    private DepartmentalPersonDto aorInfo = null;
    private String applicantTypeOtherSpecify = null;
    private String stateReviewDate = null;
    private String strReview = null;
    private static final String ORGANIZATION_YNQ_ANSWER_YES = "Y";

    private static final int AREAS_AFFECTED_ATTACHMENT = 135;
    private static final int DEBT_EXPLANATION_ATTACHMENT = 136;
    private static final int ADDITIONAL_PROJECT_TITLE_ATTACHMENT = 137;
    private static final int ADDITIONAL_CONGRESSIONAL_DISTRICTS_ATTACHMENT = 138;

    @Value("http://apply.grants.gov/forms/SF424_3_0-V3.0")
    private String namespace;

    @Value("SF424_3_0-V3.0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/SF424_3_0-V3.0.fo.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/SF424_3_0-V3.0.pdf")
    private Resource pdfForm;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    @Autowired
    @Qualifier("s2sDivisionService")
    protected S2sDivisionService s2sDivisionService;

    private SF42430Document getSF42430Doc() {
        SF42430Document sf42430Document = SF42430Document.Factory.newInstance();
        sf42430Document.setSF42430(getSF42430());
        return sf42430Document;
    }

    private SF42430 getSF42430() {

        SF42430 sf424V30 = SF42430.Factory.newInstance();
        sf424V30.setFormVersion(FormVersion.v3_0.getVersion());
        boolean hasBudgetLineItem = false;
        S2sOpportunityContract s2sOpportunity = pdDoc.getDevelopmentProposal().getS2sOpportunity();
        if (s2sOpportunity != null && s2sOpportunity.getS2sSubmissionType() != null) {
            S2sSubmissionTypeContract submissionType = s2sOpportunity.getS2sSubmissionType();
            SubmissionType.Enum subEnum = SubmissionType.Enum.forInt(Integer.parseInt(submissionType.getCode()));
            sf424V30.setSubmissionType(subEnum);
            ApplicationType.Enum applicationTypeEnum = null;
            if (pdDoc.getDevelopmentProposal().getProposalType() != null) {
                String proposalTypeCode = pdDoc.getDevelopmentProposal().getProposalType().getCode();
                if (doesParameterContainCode(ConfigurationConstants.PROPOSAL_TYPE_CODE_NEW, proposalTypeCode)) {
			        applicationTypeEnum = ApplicationType.NEW;
                } else if (doesParameterContainCode(ConfigurationConstants.PROPOSAL_TYPE_CODE_RESUBMISSION, proposalTypeCode)) {
					applicationTypeEnum = ApplicationType.REVISION;
                } else if (doesParameterContainCode(ConfigurationConstants.PROPOSAL_TYPE_CODE_RENEWAL, proposalTypeCode)) {
                    applicationTypeEnum = ApplicationType.CONTINUATION;
                } else if (doesParameterContainCode(ConfigurationConstants.PROPOSAL_TYPE_CODE_CONTINUATION, proposalTypeCode)) {
                    applicationTypeEnum = ApplicationType.CONTINUATION;
                } else if (doesParameterContainCode(ConfigurationConstants.PROPOSAL_TYPE_CODE_REVISION, proposalTypeCode)) {
                    applicationTypeEnum = ApplicationType.REVISION;
                }
            }
            sf424V30.setApplicationType(applicationTypeEnum);

            String revisionType = s2sOpportunity.getS2sRevisionType() != null ? s2sOpportunity.getS2sRevisionType().getCode() : null;
            if (revisionType != null) {
                RevisionType.Enum revType = null;
                if (revisionType.equals(INCREASE_AWARD_CODE)) {                    
                    revType = RevisionType.A_INCREASE_AWARD;
                    
                } else if (revisionType.equals(DECREASE_AWARD_CODE)) {                    
                    revType = RevisionType.B_DECREASE_AWARD;
                    
                } else if (revisionType.equals(INCREASE_DURATION_CODE)) {
                    revType = RevisionType.C_INCREASE_DURATION;
                    
                } else if (revisionType.equals(DECREASE_DURATION_CODE)) {
                    revType = RevisionType.D_DECREASE_DURATION;
                    
                } else if (revisionType.equals(INCREASE_AWARD_DECREASE_DURATION_CODE)) {
                    revType = RevisionType.AD_INCREASE_AWARD_DECREASE_DURATION;
                    
                } else if (revisionType.equals(INCREASE_AWARD_INCREASE_DURATION_CODE)) {
                    revType = RevisionType.AC_INCREASE_AWARD_INCREASE_DURATION;
                    
                } else if (revisionType.equals(DECREASE_AWARD_DECREASE_DURATION_CODE)) {
                    revType = RevisionType.BD_DECREASE_AWARD_DECREASE_DURATION;
                    
                } else if (revisionType.equals(DECREASE_AWARD_INCREASE_DURATION_CODE)) {
                    revType = RevisionType.BC_DECREASE_AWARD_INCREASE_DURATION;
                    
                } else if (revisionType.equals(OTHER_SPECIFY_CODE)) {
                    revType = RevisionType.E_OTHER_SPECIFY;
                }

                if (revType != null) {
                    sf424V30.setRevisionType(revType);
                }
                if (revisionType.startsWith(REVISIONCODE_STARTS_WITH_E)) {
                    sf424V30.setRevisionOtherSpecify(s2sOpportunity.getRevisionOtherDescription());
                }
            }
        }
        sf424V30.setDateReceived(aorInfo.getSubmittedDate());
        sf424V30.setApplicantID(pdDoc.getDevelopmentProposal().getProposalNumber());
		String federalId = getSubmissionInfoService().getFederalId(pdDoc.getDevelopmentProposal().getProposalNumber());
		if (federalId != null) {
        	sf424V30.setFederalAwardIdentifier(StringUtils.left(federalId.trim(), FED_AWARD_ID_MAX_LENGTH));
		}

		ProposalSiteContract proposalSite = pdDoc.getDevelopmentProposal().getApplicantOrganization();
        OrganizationContract organization = proposalSite.getOrganization();
        if (organization != null) {
            sf424V30.setOrganizationName(StringUtils.substring(proposalSite.getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
            sf424V30.setDUNSNumber(organization.getDunsNumber());
            sf424V30.setOrganizationAffiliation(StringUtils.substring(proposalSite.getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
        } else {
            sf424V30.setOrganizationName(null);
            sf424V30.setDUNSNumber(null);
        }
        sf424V30.setEmployerTaxpayerIdentificationNumber(getEmployerId());
        sf424V30.setApplicant(globLibV20Generator.getAddressDataTypeV3(proposalSite));
        final UnitContract leadUnit = pdDoc.getDevelopmentProposal().getOwnedByUnit();
        if (leadUnit != null) {
            String departmentName = leadUnit.getUnitName();

            if (departmentName != null) {
                sf424V30.setDepartmentName(StringUtils.substring(departmentName, 0, DEPARTMENT_NAME_MAX_LENGTH));
            }

            final String divisionName = s2sDivisionService.getDivisionFromUnit(leadUnit.getUnitNumber());
            if (divisionName != null) {
                sf424V30.setDivisionName(divisionName);
            }
        }
        ProposalPersonContract personInfo = s2SProposalPersonService.getPrincipalInvestigator(pdDoc);
        if (personInfo != null) {
            sf424V30.setContactPerson(globLibV20Generator.getHumanNameDataType(personInfo));
            if (personInfo.getDirectoryTitle() != null) {
            	String directoryTitle = personInfo.getDirectoryTitle();
            	directoryTitle = directoryTitle.length() > 45 ? directoryTitle.substring(0,44) : directoryTitle;
                sf424V30.setTitle(directoryTitle);
            }
            sf424V30.setPhoneNumber(personInfo.getOfficePhone());
            if (StringUtils.isNotEmpty(personInfo.getFaxNumber())) {
                sf424V30.setFax(personInfo.getFaxNumber());
            }
            sf424V30.setEmail(personInfo.getEmailAddress());
        } else {
            sf424V30.setPhoneNumber(null);
            sf424V30.setEmail(null);
        }

        setApplicatTypeCodes(sf424V30);

        if (pdDoc.getDevelopmentProposal().getSponsor() != null) {
            sf424V30.setAgencyName(getFederalAgencyName());
        }
        final String cfdaNumber = getFirstCfdaNumber();
        if (cfdaNumber != null) {
            sf424V30.setCFDANumber(cfdaNumber);
        }
        if (pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle() != null) {
            String announcementTitle;
            if (pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle().length() > PROGRAM_ANNOUNCEMENT_TITLE_LENGTH) {
                announcementTitle = pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle().substring(0, PROGRAM_ANNOUNCEMENT_TITLE_LENGTH);
            } else {
                announcementTitle = pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle();
            }
            sf424V30.setCFDAProgramTitle(announcementTitle);
        }
        if (pdDoc.getDevelopmentProposal().getS2sOpportunity() != null) {
        	sf424V30.setFundingOpportunityNumber(pdDoc.getDevelopmentProposal()
					.getS2sOpportunity().getOpportunityId());
			if (pdDoc.getDevelopmentProposal().getS2sOpportunity()
					.getOpportunityTitle() != null) {
				sf424V30.setFundingOpportunityTitle(pdDoc
						.getDevelopmentProposal().getS2sOpportunity()
						.getOpportunityTitle());
			}
			if (pdDoc.getDevelopmentProposal().getS2sOpportunity().getCompetitionId() != null) {
                sf424V30.setCompetitionIdentificationNumber(pdDoc.getDevelopmentProposal().getS2sOpportunity().getCompetitionId());
            }
        } else {
            sf424V30.setFundingOpportunityTitle(null);
        }
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            if (narrative.getNarrativeType().getCode() != null
                    && Integer.parseInt(narrative.getNarrativeType().getCode()) == AREAS_AFFECTED_ATTACHMENT) {
                AttachedFileDataType attachedFileDataType = addAttachedFileType(narrative);
                if (attachedFileDataType != null) {
                    sf424V30.setAreasAffected(attachedFileDataType);
                    break;
                }
            }
        }
        sf424V30.setProjectTitle(pdDoc.getDevelopmentProposal().getTitle());
        AttachmentGroupMin0Max100DataType attachedFileMin0Max100 = AttachmentGroupMin0Max100DataType.Factory.newInstance();
        attachedFileMin0Max100.setAttachedFileArray(getAttachedFileDataTypes());
        sf424V30.setAdditionalProjectTitle(attachedFileMin0Max100);
        String congressionalDistrict = organization.getCongressionalDistrict() == null ? FieldValueConstants.VALUE_UNKNOWN : organization
                .getCongressionalDistrict();
        if (congressionalDistrict.length() > CONGRESSIONAL_DISTRICT_MAX_LENGTH) {
            sf424V30.setCongressionalDistrictApplicant(congressionalDistrict.substring(0, CONGRESSIONAL_DISTRICT_MAX_LENGTH));
        } else {
            sf424V30.setCongressionalDistrictApplicant(congressionalDistrict);
        }
        ProposalSiteContract perfOrganization = pdDoc.getDevelopmentProposal().getPerformingOrganization();
        if (perfOrganization != null) {
            String congDistrictProject = perfOrganization.getFirstCongressionalDistrictName() == null ? FieldValueConstants.VALUE_UNKNOWN
                    : perfOrganization.getFirstCongressionalDistrictName();
            if (congDistrictProject.length() > CONGRESSIONAL_DISTRICT_MAX_LENGTH) {
                sf424V30.setCongressionalDistrictProgramProject(congDistrictProject.substring(0, CONGRESSIONAL_DISTRICT_MAX_LENGTH));
            } else {
                sf424V30.setCongressionalDistrictProgramProject(congDistrictProject);
            }
        }
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            if (narrative.getNarrativeType().getCode() != null
                    && Integer.parseInt(narrative.getNarrativeType().getCode()) == ADDITIONAL_CONGRESSIONAL_DISTRICTS_ATTACHMENT) {
                AttachedFileDataType attachedFileDataType = addAttachedFileType(narrative);
                if (attachedFileDataType != null) {
                	sf424V30.setAdditionalCongressionalDistricts(attachedFileDataType);
                	break;
                }
            }
        }
        if (pdDoc.getDevelopmentProposal().getRequestedStartDateInitial() != null) {
            sf424V30.setProjectStartDate(s2SDateTimeService.convertDateToCalendar(pdDoc.getDevelopmentProposal().getRequestedStartDateInitial()));
        } else {
            sf424V30.setProjectStartDate(null);
        }
        if (pdDoc.getDevelopmentProposal().getRequestedEndDateInitial() != null) {
            sf424V30.setProjectEndDate(s2SDateTimeService.convertDateToCalendar(pdDoc.getDevelopmentProposal().getRequestedEndDateInitial()));
        } else {
            sf424V30.setProjectEndDate(null);
        }
        ProposalDevelopmentBudgetExtContract budget = getS2SCommonBudgetService().getBudget(pdDoc.getDevelopmentProposal());

        if (budget != null) {
            if (budget.getTotalCost() != null) {
                sf424V30.setFederalEstimatedFunding(budget.getTotalCost().bigDecimalValue());
            }
            ScaleTwoDecimal costSharingAmount = ScaleTwoDecimal.ZERO;

            for (BudgetPeriodContract budgetPeriod : budget.getBudgetPeriods()) {
                for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {
                    hasBudgetLineItem = true;
                    if (budget.getSubmitCostSharingFlag() && lineItem.getSubmitCostSharingFlag()) {
                        costSharingAmount =  costSharingAmount.add(lineItem.getCostSharingAmount());
                        List<? extends BudgetLineItemCalculatedAmountContract> calculatedAmounts = lineItem.getBudgetLineItemCalculatedAmounts();
                        for (BudgetLineItemCalculatedAmountContract budgetLineItemCalculatedAmount : calculatedAmounts) {
                            costSharingAmount =  costSharingAmount.add(budgetLineItemCalculatedAmount.getCalculatedCostSharing());
                        }                        
                    }
                }
            }
            if (!hasBudgetLineItem && budget.getSubmitCostSharingFlag()) {
                costSharingAmount = budget.getCostSharingAmount();      
            }
            sf424V30.setApplicantEstimatedFunding(costSharingAmount.bigDecimalValue());
            BigDecimal projectIncome = BigDecimal.ZERO;
            for (BudgetProjectIncomeContract budgetProjectIncome : budget.getBudgetProjectIncomes()) {
                projectIncome = projectIncome.add(budgetProjectIncome.getProjectIncome().bigDecimalValue());
            }
            sf424V30.setProgramIncomeEstimatedFunding(projectIncome);

            ScaleTwoDecimal totalEstimatedAmount = ScaleTwoDecimal.ZERO;
            if (budget.getTotalCost() != null) {
                totalEstimatedAmount = totalEstimatedAmount.add(budget.getTotalCost());
            }
            totalEstimatedAmount = totalEstimatedAmount.add(costSharingAmount);
            totalEstimatedAmount = totalEstimatedAmount.add(new ScaleTwoDecimal(projectIncome));
            sf424V30.setTotalEstimatedFunding(totalEstimatedAmount.bigDecimalValue());
        } else {
            sf424V30.setFederalEstimatedFunding(BigDecimal.ZERO);
            sf424V30.setApplicantEstimatedFunding(BigDecimal.ZERO);
            sf424V30.setProgramIncomeEstimatedFunding(BigDecimal.ZERO);
            sf424V30.setTotalEstimatedFunding(BigDecimal.ZERO);
        }
        sf424V30.setStateEstimatedFunding(BigDecimal.ZERO);
        sf424V30.setLocalEstimatedFunding(BigDecimal.ZERO);
        sf424V30.setOtherEstimatedFunding(BigDecimal.ZERO);
        sf424V30.setStateReview(getStateReviewCode());
        if (strReview != null && strReview.equals(STATE_REVIEW_YES)) {
            Calendar reviewDate;
            reviewDate = s2SDateTimeService.convertDateStringToCalendar(stateReviewDate);
            sf424V30.setStateReviewAvailableDate(reviewDate);
        }
        YesNoDataType.Enum yesNo = YesNoDataType.N_NO;
        OrganizationContract applicantOrganization = pdDoc.getDevelopmentProposal().getApplicantOrganization().getOrganization();
        if (applicantOrganization != null) {
            for (OrganizationYnqContract orgYnq : applicantOrganization.getOrganizationYnqs()) {
                if (orgYnq.getQuestionId() != null && orgYnq.getQuestionId().equals(PROPOSAL_YNQ_FEDERAL_DEBTS)) {
                    String orgYnqanswer = orgYnq.getAnswer();
                    if (orgYnqanswer != null) {
                        if (orgYnqanswer.equalsIgnoreCase(ORGANIZATION_YNQ_ANSWER_YES)) {
                            yesNo = YesNoDataType.Y_YES;
                        } else {
                            yesNo = YesNoDataType.N_NO;
                        }
                    }
                }
            }
        }
        sf424V30.setDelinquentFederalDebt(yesNo);
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            if (narrative.getNarrativeType().getCode() != null
                    && Integer.parseInt(narrative.getNarrativeType().getCode()) == DEBT_EXPLANATION_ATTACHMENT) {
                AttachedFileDataType attachedFileDataType = addAttachedFileType(narrative);
                if (attachedFileDataType != null) {
                    sf424V30.setDebtExplanation(attachedFileDataType);
                    break;
                }
            }
        }
        
        sf424V30.setCertificationAgree(YesNoDataType.Y_YES);
        sf424V30.setAuthorizedRepresentative(globLibV20Generator.getHumanNameDataType(aorInfo));
        if (aorInfo.getPrimaryTitle() != null) {
            sf424V30.setAuthorizedRepresentativeTitle(StringUtils.substring(aorInfo.getPrimaryTitle(), 0, PRIMARY_TITLE_MAX_LENGTH));
        } else {
            sf424V30.setAuthorizedRepresentativeTitle(null);
        }
        sf424V30.setAuthorizedRepresentativePhoneNumber(aorInfo.getOfficePhone());
        sf424V30.setAuthorizedRepresentativeEmail(aorInfo.getEmailAddress());
        if (StringUtils.isNotEmpty(aorInfo.getFaxNumber())) {
            sf424V30.setAuthorizedRepresentativeFax(aorInfo.getFaxNumber());
        }
        sf424V30.setAORSignature(aorInfo.getFullName());
        sf424V30.setDateSigned(aorInfo.getSubmittedDate());
        return sf424V30;
    }

    private void setApplicatTypeCodes(SF42430 sf424V30) {
        OrganizationContract organization = pdDoc.getDevelopmentProposal().getApplicantOrganization().getOrganization();
        List<? extends OrganizationTypeContract> organizationTypes = organization.getOrganizationTypes();
        if (organizationTypes.isEmpty()) {
            sf424V30.setApplicantTypeCode1(null);
            return;
        }
        for (int i = 0; i < organizationTypes.size() && i < 3; i++) {
            OrganizationTypeContract orgType = organizationTypes.get(i);
            ApplicantTypeCodeDataType.Enum applicantTypeCode = ApplicantTypeCodeDataType.X_OTHER_SPECIFY;
            switch (orgType.getOrganizationTypeList().getCode()) {
                case 1:
                    applicantTypeCode = ApplicantTypeCodeDataType.C_CITY_OR_TOWNSHIP_GOVERNMENT;
                    break;
                case 2:
                    applicantTypeCode = ApplicantTypeCodeDataType.A_STATE_GOVERNMENT;
                    break;
                case 3:
                    applicantTypeCode = ApplicantTypeCodeDataType.X_OTHER_SPECIFY;
                    applicantTypeOtherSpecify = "Federal Government";
                    break;
                case 4:
                    applicantTypeCode = ApplicantTypeCodeDataType.M_NONPROFIT_WITH_501_C_3_IRS_STATUS_OTHER_THAN_INSTITUTION_OF_HIGHER_EDUCATION;
                    break;
                case 5:
                    applicantTypeCode = ApplicantTypeCodeDataType.N_NONPROFIT_WITHOUT_501_C_3_IRS_STATUS_OTHER_THAN_INSTITUTION_OF_HIGHER_EDUCATION;
                    break;
                case 6:
                    applicantTypeCode = ApplicantTypeCodeDataType.Q_FOR_PROFIT_ORGANIZATION_OTHER_THAN_SMALL_BUSINESS;
                    break;
                case 7:
                    applicantTypeCode = ApplicantTypeCodeDataType.X_OTHER_SPECIFY;
                    break;
                case 8:
                    applicantTypeCode = ApplicantTypeCodeDataType.I_INDIAN_NATIVE_AMERICAN_TRIBAL_GOVERNMENT_FEDERALLY_RECOGNIZED;
                    break;
                case 9:
                    applicantTypeCode = ApplicantTypeCodeDataType.P_INDIVIDUAL;
                    break;
                case 10:
                    applicantTypeCode = ApplicantTypeCodeDataType.O_PRIVATE_INSTITUTION_OF_HIGHER_EDUCATION;
                    break;
                case 11:
                    applicantTypeCode = ApplicantTypeCodeDataType.R_SMALL_BUSINESS;
                    break;
                case 14:
                    applicantTypeCode = ApplicantTypeCodeDataType.X_OTHER_SPECIFY;
                    applicantTypeOtherSpecify = "Socially and Economically Disadvantaged";
                    break;
                case 15:
                    applicantTypeCode = ApplicantTypeCodeDataType.X_OTHER_SPECIFY;
                    applicantTypeOtherSpecify = "Women owned";
                    break;
                case 21:
                    applicantTypeCode = ApplicantTypeCodeDataType.H_PUBLIC_STATE_CONTROLLED_INSTITUTION_OF_HIGHER_EDUCATION;
                    break;
                case 22:
                    applicantTypeCode = ApplicantTypeCodeDataType.B_COUNTY_GOVERNMENT;
                    break;
                case 23:
                    applicantTypeCode = ApplicantTypeCodeDataType.D_SPECIAL_DISTRICT_GOVERNMENT;
                    break;
                case 24:
                    applicantTypeCode = ApplicantTypeCodeDataType.G_INDEPENDENT_SCHOOL_DISTRICT;
                    break;
                case 25:
                    applicantTypeCode = ApplicantTypeCodeDataType.L_PUBLIC_INDIAN_HOUSING_AUTHORITY;
                    break;
                case 26:
                    applicantTypeCode = ApplicantTypeCodeDataType.J_INDIAN_NATIVE_AMERICAN_TRIBAL_GOVERNMENT_OTHER_THAN_FEDERALLY_RECOGNIZED;
                    break;
                case 100:
                    applicantTypeCode = ApplicantTypeCodeDataType.E_REGIONAL_ORGANIZATION;
                    break;
                case 101:
                    applicantTypeCode = ApplicantTypeCodeDataType.F_U_S_TERRITORY_OR_POSSESSION;
                    break;
                case 102:
                    applicantTypeCode = ApplicantTypeCodeDataType.K_INDIAN_NATIVE_AMERICAN_TRIBALLY_DESIGNATED_ORGANIZATION;
                    break;
                case 103:
                    applicantTypeCode = ApplicantTypeCodeDataType.S_HISPANIC_SERVING_INSTITUTION;
                    break;
                case 104:
                    applicantTypeCode = ApplicantTypeCodeDataType.T_HISTORICALLY_BLACK_COLLEGES_AND_UNIVERSITIES_HBC_US;
                    break;
                case 105:
                    applicantTypeCode = ApplicantTypeCodeDataType.U_TRIBALLY_CONTROLLED_COLLEGES_AND_UNIVERSITIES_TCC_US;
                    break;
                case 106:
                    applicantTypeCode = ApplicantTypeCodeDataType.V_ALASKA_NATIVE_AND_NATIVE_HAWAIIAN_SERVING_INSTITUTIONS;
                    break;
                case 107:
                    applicantTypeCode = ApplicantTypeCodeDataType.W_NON_DOMESTIC_NON_US_ENTITY;
                    break;
            }
            switch (i) {
                case 0:
                    sf424V30.setApplicantTypeCode1(applicantTypeCode);
                    break;
                case 1:
                    sf424V30.setApplicantTypeCode2(applicantTypeCode);
                    break;
                case 2:
                    sf424V30.setApplicantTypeCode3(applicantTypeCode);
                    break;
            }
            if (applicantTypeOtherSpecify != null) {
                sf424V30.setApplicantTypeOtherSpecify(applicantTypeOtherSpecify);
            }
        }
    }

    private StateReview.Enum getStateReviewCode() {
        Map<String, String> eoStateReview = getEOStateReview(pdDoc);
        StateReview.Enum stateType = null;
        String stateReviewData;
        strReview = eoStateReview.get(YNQ_ANSWER);
        if (strReview != null) {
            if (strReview.equals(STATE_REVIEW_YES)) {
                stateType = StateReview.A_THIS_APPLICATION_WAS_MADE_AVAILABLE_TO_THE_STATE_UNDER_THE_EXECUTIVE_ORDER_12372_PROCESS_FOR_REVIEW_ON;
            } else if (strReview.equals(STATE_REVIEW_NO)) {
                stateReviewData = eoStateReview.get(YNQ_STATE_REVIEW_DATA);
                if (YNQ_STATE_NOT_COVERED.equals(stateReviewData)) {
                    stateType = StateReview.C_PROGRAM_IS_NOT_COVERED_BY_E_O_12372;
                } else if (YNQ_STATE_NOT_SELECTED.equals(stateReviewData)) {
                    stateType = StateReview.B_PROGRAM_IS_SUBJECT_TO_E_O_12372_BUT_HAS_NOT_BEEN_SELECTED_BY_THE_STATE_FOR_REVIEW;
                }
            }
        }
        if (eoStateReview.get(YNQ_REVIEW_DATE) != null) {
            stateReviewDate = eoStateReview.get(YNQ_REVIEW_DATE);
        }
        return stateType;
    }

    private AttachedFileDataType[] getAttachedFileDataTypes() {
        List<AttachedFileDataType> attachedFileDataTypeList = new ArrayList<>();
        AttachedFileDataType attachedFileDataType;
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            if (narrative.getNarrativeType().getCode() != null
                    && Integer.parseInt(narrative.getNarrativeType().getCode()) == ADDITIONAL_PROJECT_TITLE_ATTACHMENT) {
                attachedFileDataType = addAttachedFileType(narrative);
                if (attachedFileDataType != null) {
                	attachedFileDataTypeList.add(attachedFileDataType);
                }
            }
        }
        return attachedFileDataTypeList.toArray(new AttachedFileDataType[0]);
    }

    @Override
    public SF42430Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        aorInfo = departmentalPersonService.getDepartmentalPerson(pdDoc);
        return getSF42430Doc();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    public S2sDivisionService getS2sDivisionService() {
        return s2sDivisionService;
    }

    public void setS2sDivisionService(S2sDivisionService s2sDivisionService) {
        this.s2sDivisionService = s2sDivisionService;
    }

    @Override
    public Attachments getMappedAttachments(SF42430Document form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {

            final AttachedFileDataType areasAffected = form.getSF42430().getAreasAffected();
            if (areasAffected != null && a.getContentId().equals(areasAffected.getFileLocation().getHref())) {
                return entry("SF424_3_0_P2.optionalFile1", a);
            }

            final AttachedFileDataType congressionalDistricts = form.getSF42430().getAdditionalCongressionalDistricts();
            if (congressionalDistricts != null && a.getContentId().equals(congressionalDistricts.getFileLocation().getHref())) {
                return entry("SF424_3_0_P3.optionalFile0", a);
            }

            final AttachedFileDataType debtExplanation = form.getSF42430().getDebtExplanation();
            if (debtExplanation != null && a.getContentId().equals(debtExplanation.getFileLocation().getHref())) {
                return entry("SF424_3_0_P3.optionalFile1", a);
            }

            final AttachmentGroupMin0Max100DataType additionalProjectTitle = form.getSF42430().getAdditionalProjectTitle();
            if (additionalProjectTitle != null && additionalProjectTitle.getAttachedFileList() != null) {
                final List<AttachedFileDataType> atts = additionalProjectTitle.getAttachedFileList();
                for (int i = 0; i < atts.size(); i++) {
                    final AttachedFileDataType att = atts.get(i);
                    if (att != null && a.getContentId().equals(att.getFileLocation().getHref())){
                        return entry("SF424_3_0_P2.optionalFile0" + i, a);
                    }
                }
            }

            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<SF42430Document> factory() {
        return SF42430Document.Factory;
    }
}
