/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.hash;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.xml.security.algorithms.MessageDigestAlgorithm;
import org.apache.xml.security.c14n.CanonicalizationException;
import org.apache.xml.security.c14n.Canonicalizer;
import org.apache.xml.security.c14n.InvalidCanonicalizerException;
import org.apache.xml.security.signature.XMLSignatureException;
import org.apache.xml.security.utils.Base64;
import org.apache.xml.security.utils.DigesterOutputStream;
import org.kuali.coeus.s2sgen.api.core.InfrastructureConstants;
import org.kuali.coeus.s2sgen.api.core.S2SException;
import org.kuali.coeus.s2sgen.impl.util.XPathExecutor;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

import java.io.IOException;

/**
 * This is a convenience object that simplifies the hashing processing to one
 * method call.
 * 
 * @author David Wong
 */
final class GrantApplicationHash {

	private static final Logger LOG = LogManager.getLogger(GrantApplicationHash.class.getName());
	private static final String FORMS_XPATH = "//*[local-name(.) = 'Forms' and namespace-uri(.) = 'http://apply.grants.gov/system/MetaGrantApplication']";


	private static java.security.MessageDigest MESSAGE_DIGESTER = null;

	static {
		org.apache.xml.security.Init.init();
		try {
			MESSAGE_DIGESTER = java.security.MessageDigest.getInstance(InfrastructureConstants.HASH_ALGORITHM);
		} catch (Exception ex) {
			LOG.error(
					"Unable to get instance of java.security.MessageDigester",
					ex);
		}
	}

	/**
	 * Added private constructor to prevent creation by user.
	 */
	private GrantApplicationHash() {
		throw new UnsupportedOperationException("do not call");
	}

	/**
	 * Computes the hash value for the Grants.gov application XML.
	 * 
	 * @param xml
	 *            The Grants.gov application XML.
	 * @return The SHA-1 hash value of &lt;grant:forms&gt; tag inside the
	 *         application XML.
	 * @throws S2SException
	 *             When the XML cannot be parsed.
	 */
	static String computeGrantFormsHash(String xml) throws S2SException {
		final XPathExecutor xpath;
        try {
            xpath = new XPathExecutor(xml);
            return _hash(xpath);
        }catch (Exception e) {
			LOG.error(e.getMessage(), e);
            throw new S2SException(e.getMessage(),e);
        }
	}

	/**
	 * Computes the hash of an binary attachment.
	 *
	 * @return The SHA-1 hash value of the attachment byte array.
	 */
	static String computeAttachmentHash(byte[] attachment) {

		byte[] rawDigest = MESSAGE_DIGESTER.digest(attachment);

		return Base64.encode(rawDigest);

	}

	private static String _hash(XPathExecutor xpath)
			throws XMLSignatureException,
			InvalidCanonicalizerException, CanonicalizationException {
		Node formsNode = xpath.getNode(FORMS_XPATH);
		try (DigesterOutputStream digester = _createDigesterOutputStream(xpath
				.getDoc())) {
			Canonicalizer canonicalizer = Canonicalizer
					.getInstance(Canonicalizer.ALGO_ID_C14N_EXCL_OMIT_COMMENTS);

			canonicalizer.canonicalizeSubtree(formsNode, digester);
			byte[] hash = digester.getDigestValue();
			return Base64.encode(hash);
		} catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

	private static DigesterOutputStream _createDigesterOutputStream(Document doc)
			throws XMLSignatureException {
		DigesterOutputStream stream = null;
		if (doc != null) {
			stream = new DigesterOutputStream(MessageDigestAlgorithm
					.getInstance(doc,
							MessageDigestAlgorithm.ALGO_ID_DIGEST_SHA1));
		}
		return stream;
	}
}
