/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package org.kuali.coeus.s2sgen.impl.util;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.TransformerFactory;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

/**
 * Utility class to help avoid security vulnerabilities with various XML libraries.
 */
public class SafeXmlUtils {

    private static final Logger LOG = LogManager.getLogger(SafeXmlUtils.class);
    public static final String DISALLOW_DOCTYPE_DECL = "http://apache.org/xml/features/disallow-doctype-decl";
    public static final String EXTERNAL_GENERAL_ENTITIES = "http://xml.org/sax/features/external-general-entities";
    public static final String EXTERNAL_PARAMETER_ENTITIES = "http://xml.org/sax/features/external-parameter-entities";
    public static final String LOAD_EXTERNAL_DTD = "http://apache.org/xml/features/nonvalidating/load-external-dtd";

    private SafeXmlUtils() {
        throw new UnsupportedOperationException("do not call");
    }


    /**
     * Creates a {@link DocumentBuilderFactory} that configures certain features to avoid XXE attacks as documented:
     * https://cheatsheetseries.owasp.org/cheatsheets/XML_External_Entity_Prevention_Cheat_Sheet.html
     */
    public static DocumentBuilderFactory safeDocumentBuilderFactory() {
        final DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        try {
            dbf.setFeature(DISALLOW_DOCTYPE_DECL, true);
        } catch (ParserConfigurationException e) {
            LOG.warn(e.getMessage(), e);
        }
        try {
            dbf.setFeature(EXTERNAL_GENERAL_ENTITIES, false);
        } catch (ParserConfigurationException e) {
            LOG.warn(e.getMessage(), e);
        }
        try {
            dbf.setFeature(EXTERNAL_PARAMETER_ENTITIES, false);
        } catch (ParserConfigurationException e) {
            LOG.warn(e.getMessage(), e);
        }
        try {
            dbf.setFeature(LOAD_EXTERNAL_DTD, false);
        } catch (ParserConfigurationException e) {
            LOG.warn(e.getMessage(), e);
        }
        dbf.setXIncludeAware(false);
        dbf.setExpandEntityReferences(false);

        return dbf;
    }

    /**
     * Creates a {@link TransformerFactory} that configures certain attributes to avoid XXE attacks as documented:
     * https://cheatsheetseries.owasp.org/cheatsheets/XML_External_Entity_Prevention_Cheat_Sheet.html
     */
    public static TransformerFactory safeTransformerFactory() {
        final TransformerFactory tf = TransformerFactory.newInstance();

        tf.setAttribute(XMLConstants.ACCESS_EXTERNAL_DTD, "");
        tf.setAttribute(XMLConstants.ACCESS_EXTERNAL_STYLESHEET, "");

        return tf;
    }

    public static SAXParserFactory safeSAXParserFactory() throws SAXNotSupportedException, SAXNotRecognizedException, ParserConfigurationException {
        final SAXParserFactory spf = SAXParserFactory.newInstance();
        spf.setFeature(EXTERNAL_GENERAL_ENTITIES, false);
        spf.setFeature(EXTERNAL_PARAMETER_ENTITIES, false);
        spf.setFeature(LOAD_EXTERNAL_DTD, false);

        return spf;
    }

    public static Validator safeValidator(Schema schema) throws SAXNotRecognizedException, SAXNotSupportedException {
        final Validator validator = schema.newValidator();
        validator.setProperty(XMLConstants.ACCESS_EXTERNAL_DTD, "");
        validator.setProperty(XMLConstants.ACCESS_EXTERNAL_SCHEMA, "");
        return validator;
    }

    public static SchemaFactory safeSchemaFactory() throws SAXNotRecognizedException, SAXNotSupportedException {
        SchemaFactory factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
        factory.setProperty(XMLConstants.ACCESS_EXTERNAL_DTD, "");
        factory.setProperty(XMLConstants.ACCESS_EXTERNAL_SCHEMA, "");
        return factory;
    }
}
