/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.sf424Short20V20.SF424Short20Document;
import gov.grants.apply.forms.sf424Short20V20.SF424Short20Document.SF424Short20;
import gov.grants.apply.system.globalLibraryV20.ApplicantTypeCodeDataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.org.OrganizationContract;
import org.kuali.coeus.common.api.org.type.OrganizationTypeContract;
import org.kuali.coeus.common.api.sponsor.SponsorContract;
import org.kuali.coeus.propdev.api.abstrct.ProposalAbstractContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.location.ProposalSiteContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.propdev.api.s2s.S2sOpportunityContract;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonDto;
import org.kuali.coeus.s2sgen.impl.util.FieldValueConstants;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.Collections;
import java.util.List;

@FormGenerator("SF424ShortV2_0Generator")
public class SF424ShortV2_0Generator extends SF424BaseGenerator<SF424Short20Document> implements S2SFormGeneratorPdfFillable<SF424Short20Document> {

    private static final String ABSTRACT_TYPE_PROJECT_DESCRIPTION = "1";
    private static final int PROGRAM_ANNOUNCEMENT_TITLE_MAX_LENGTH = 120;
    private static final int OPPORTUNITY_ID_MAX_LENGTH = 40;
    private static final int ABSTRACT_TYPE_CODE_MAX_LENGTH = 1000;
    private static final int OFFICE_PHONE_MAX_LENGTH = 25;
    private static final int EMAIL_ADDRESS_MAX_LENGTH = 60;
    private static final int FAX_NUMBER_MAX_LENGTH = 25;

    @Value("http://apply.grants.gov/forms/SF424_Short_2_0-V2.0")
    private String namespace;

    @Value("SF424_Short-V2.0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/SF424_Short-V2.0.fo.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/SF424_Short_2_0-V2.0.pdf")
    private Resource pdfForm;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    private SF424Short20Document getsf424ShortDocument() {

        SF424Short20Document sf424ShortDocument = SF424Short20Document.Factory.newInstance();
        SF424Short20 sf424Short = SF424Short20.Factory.newInstance();
        sf424Short.setFormVersion(FormVersion.v2_0.getVersion());

        // Set default values for mandatory fields
        sf424Short.setAgencyName("");
        sf424Short.setFundingOpportunityNumber("");
        sf424Short.setFundingOpportunityTitle("");
        sf424Short.setOrganizationName("");
        sf424Short.setEmployerTaxpayerIdentificationNumber("");
        sf424Short.setProjectDescription("");
        sf424Short.setAuthorizedRepresentativeTitle("");
        sf424Short.setAuthorizedRepresentativePhoneNumber("");
        sf424Short.setAuthorizedRepresentativeEmail("");

        SponsorContract sponsor = pdDoc.getDevelopmentProposal().getPrimeSponsor();
        if (pdDoc.getDevelopmentProposal().getSponsor() != null && pdDoc.getDevelopmentProposal().getSponsor().getSponsorName() != null) {
            if (pdDoc.getDevelopmentProposal().getSponsor().getSponsorName().length() > SPONSOR_NAME_MAX_LENGTH) {
                sf424Short.setAgencyName(pdDoc.getDevelopmentProposal().getSponsor().getSponsorName().substring(0, SPONSOR_NAME_MAX_LENGTH));
            }
            else {
                sf424Short.setAgencyName(pdDoc.getDevelopmentProposal().getSponsor().getSponsorName());
            }
        }
        else if (sponsor.getSponsorName() != null) {
            if (sponsor.getSponsorName().length() > SPONSOR_NAME_MAX_LENGTH) {
                sf424Short.setAgencyName(sponsor.getSponsorName().substring(0, SPONSOR_NAME_MAX_LENGTH));
            }
            else {
                sf424Short.setAgencyName(sponsor.getSponsorName());
            }
        }

        final String cfdaNumber = getFirstCfdaNumber();
        if (cfdaNumber != null) {
            sf424Short.setCFDANumber(cfdaNumber);
        }

        if (pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle() != null) {
            if (pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle().length() > PROGRAM_ANNOUNCEMENT_TITLE_MAX_LENGTH) {
                sf424Short.setCFDAProgramTitle(pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle().substring(0,
                        PROGRAM_ANNOUNCEMENT_TITLE_MAX_LENGTH));
            }
            else {
                sf424Short.setCFDAProgramTitle(pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle());
            }
        }

        S2sOpportunityContract s2sOpportunity = pdDoc.getDevelopmentProposal().getS2sOpportunity();
        if (s2sOpportunity != null) {
            if (s2sOpportunity.getOpportunityId().length() > OPPORTUNITY_ID_MAX_LENGTH) {
                sf424Short.setFundingOpportunityNumber(s2sOpportunity.getOpportunityId().substring(0, OPPORTUNITY_ID_MAX_LENGTH));
            }
            else {
                sf424Short.setFundingOpportunityNumber(s2sOpportunity.getOpportunityId());
            }
            if (s2sOpportunity.getOpportunityTitle() != null) {
                sf424Short.setFundingOpportunityTitle(s2sOpportunity.getOpportunityTitle());
            }
        }

        ProposalSiteContract proposalSite = pdDoc.getDevelopmentProposal().getApplicantOrganization();
        OrganizationContract organization = proposalSite.getOrganization();
        if (proposalSite.getLocationName() != null) {
            sf424Short.setOrganizationName(StringUtils.substring(proposalSite.getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
        }

        if (proposalSite != null) {
            sf424Short.setAddress(globLibV20Generator.getAddressDataTypeV3(proposalSite));
        }
        List<? extends OrganizationTypeContract> organizationTypes = organization
				.getOrganizationTypes();
		ApplicantTypeCodeDataType.Enum applicantTypeCode = getApplicantType(
				organizationTypes, APPLICANT_TYPE_1_INDEX);
		if (applicantTypeCode != null) {
			sf424Short.setApplicantTypeCode1(applicantTypeCode);
		}
		ApplicantTypeCodeDataType.Enum applicantTypeCode2 = getApplicantType(
				organizationTypes, APPLICANT_TYPE_2_INDEX);
		if (applicantTypeCode2 != null) {
			sf424Short.setApplicantTypeCode2(applicantTypeCode2);
		}
		ApplicantTypeCodeDataType.Enum applicantTypeCode3 = getApplicantType(
				organizationTypes, APPLICANT_TYPE_3_INDEX);
		if (applicantTypeCode3 != null) {
            sf424Short.setApplicantTypeCode3(applicantTypeCode3);
        }

        sf424Short.setEmployerTaxpayerIdentificationNumber(getEmployerId());
        sf424Short.setDUNSNumber(organization.getDunsNumber());
        String congressionalDistrict = organization.getCongressionalDistrict() == null ? FieldValueConstants.VALUE_UNKNOWN : organization
                .getCongressionalDistrict();
        if (congressionalDistrict.length() > CONGRESSIONAL_DISTRICT_MAX_LENGTH) {
            sf424Short.setCongressionalDistrictApplicant(congressionalDistrict.substring(0, CONGRESSIONAL_DISTRICT_MAX_LENGTH));
        }
        else {
            sf424Short.setCongressionalDistrictApplicant(congressionalDistrict);
        }

        sf424Short.setProjectTitle(pdDoc.getDevelopmentProposal().getTitle());

        for (ProposalAbstractContract proposalAbstract : pdDoc.getDevelopmentProposal().getProposalAbstracts()) {
            if (proposalAbstract.getAbstractType().getCode() != null
                    && proposalAbstract.getAbstractType().getCode().equals(ABSTRACT_TYPE_PROJECT_DESCRIPTION)) {
                if (proposalAbstract.getAbstractDetails().length() > ABSTRACT_TYPE_CODE_MAX_LENGTH) {
                    sf424Short.setProjectDescription(proposalAbstract.getAbstractDetails().substring(0,
                            ABSTRACT_TYPE_CODE_MAX_LENGTH));
                }
                else {
                    sf424Short.setProjectDescription(proposalAbstract.getAbstractDetails());
                }
                break;
            }
        }

        sf424Short.setProjectStartDate(s2SDateTimeService.convertDateToCalendar(pdDoc.getDevelopmentProposal().getRequestedStartDateInitial()));
        sf424Short.setProjectEndDate(s2SDateTimeService.convertDateToCalendar(pdDoc.getDevelopmentProposal().getRequestedEndDateInitial()));

        ProposalPersonContract pi = s2SProposalPersonService.getPrincipalInvestigator(pdDoc);
        sf424Short.setProjectDirectorGroup(globLibV20Generator.getContactPersonDataTypeV3(pi));

        // Rolodex is set to getOrganization.getRolodex above
        sf424Short.setContactPersonGroup(globLibV20Generator.getContactPersonDataTypeV3(pdDoc));

        // value is hard coded
        sf424Short.setApplicationCertification(YesNoDataType.Y_YES);

        DepartmentalPersonDto aorInfo = departmentalPersonService.getDepartmentalPerson(pdDoc);
        sf424Short.setDateReceived(aorInfo.getSubmittedDate());
        sf424Short.setAuthorizedRepresentative(globLibV20Generator.getHumanNameDataType(aorInfo));

        if (aorInfo.getPrimaryTitle() != null) {
            sf424Short.setAuthorizedRepresentativeTitle(StringUtils.substring(aorInfo.getPrimaryTitle(), 0, PRIMARY_TITLE_MAX_LENGTH));
        }

        if (aorInfo.getOfficePhone() != null) {
            if (aorInfo.getOfficePhone().length() > OFFICE_PHONE_MAX_LENGTH) {
                sf424Short.setAuthorizedRepresentativePhoneNumber(aorInfo.getOfficePhone().substring(0, OFFICE_PHONE_MAX_LENGTH));
            }
            else {
                sf424Short.setAuthorizedRepresentativePhoneNumber(aorInfo.getOfficePhone());
            }
        }

        if (aorInfo.getEmailAddress() != null) {
            if (aorInfo.getEmailAddress().length() > EMAIL_ADDRESS_MAX_LENGTH) {
                sf424Short.setAuthorizedRepresentativeEmail(aorInfo.getEmailAddress().substring(0, EMAIL_ADDRESS_MAX_LENGTH));
            }
            else {
                sf424Short.setAuthorizedRepresentativeEmail(aorInfo.getEmailAddress());
            }
        }

        if (StringUtils.isNotEmpty(aorInfo.getFaxNumber())) {
            if (aorInfo.getFaxNumber().length() > FAX_NUMBER_MAX_LENGTH) {
                sf424Short.setAuthorizedRepresentativeFaxNumber(aorInfo.getFaxNumber().substring(0, FAX_NUMBER_MAX_LENGTH));
            }
            else {
                sf424Short.setAuthorizedRepresentativeFaxNumber(aorInfo.getFaxNumber());
            }
        }
        sf424Short.setAuthorizedRepresentativeSignature(aorInfo.getFullName());
        sf424Short.setAuthorizedRepresentativeDateSigned(aorInfo.getSubmittedDate());
        sf424ShortDocument.setSF424Short20(sf424Short);
        return sf424ShortDocument;
    }

    @Override
    public SF424Short20Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getsf424ShortDocument();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(SF424Short20Document form, List<AttachmentData> attachments) {
        return new Attachments(Collections.emptyMap(), attachments);
    }

    @Override
    public DocumentFactory<SF424Short20Document> factory() {
        return SF424Short20Document.Factory;
    }
}
