/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.sflllV11.AwardeeDataType.Address;
import gov.grants.apply.forms.sflllV11.LobbyingActivitiesDisclosureDocument;
import gov.grants.apply.forms.sflllV11.LobbyingActivitiesDisclosureDocument.LobbyingActivitiesDisclosure;
import gov.grants.apply.forms.sflllV11.LobbyingActivitiesDisclosureDocument.LobbyingActivitiesDisclosure.*;
import gov.grants.apply.forms.sflllV11.LobbyingActivitiesDisclosureDocument.LobbyingActivitiesDisclosure.IndividualsPerformingServices.Individual;
import gov.grants.apply.forms.sflllV11.LobbyingActivitiesDisclosureDocument.LobbyingActivitiesDisclosure.ReportEntity.Prime;
import gov.grants.apply.forms.sflllV11.ReportDataType;
import gov.grants.apply.forms.sflllV11.ReportEntityDataType;
import gov.grants.apply.system.globalLibraryV20.HumanNameDataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.org.OrganizationContract;
import org.kuali.coeus.common.api.sponsor.SponsorContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.location.ProposalSiteContract;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonDto;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.List;

@FormGenerator("SFLLLV1_1Generator")
public class SFLLLV1_1Generator extends SFLLLBaseGenerator<LobbyingActivitiesDisclosureDocument> {

    private DepartmentalPersonDto aorInfo;

    @Value("http://apply.grants.gov/forms/SFLLL-V1.1")
    private String namespace;

    @Value("SFLLL-V1.1")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/SFLLL-V1.1.fo.xsl")
    private List<Resource> stylesheets;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    private LobbyingActivitiesDisclosureDocument getLobbyingActivitiesDisclosure() {

        LobbyingActivitiesDisclosureDocument lobbyinADDocument = LobbyingActivitiesDisclosureDocument.Factory.newInstance();
        LobbyingActivitiesDisclosure lobbyingDisclosure = LobbyingActivitiesDisclosure.Factory.newInstance();
        lobbyingDisclosure.setFormVersion(FormVersion.v1_1.getVersion());
        lobbyingDisclosure.setFederalActionType(FederalActionType.GRANT);
        lobbyingDisclosure.setFederalActionStatus(FederalActionStatus.BID_OFFER);
        lobbyingDisclosure.setReportType(ReportDataType.INITIAL_FILING);
        lobbyingDisclosure.setReportEntity(getReportEntity());
        lobbyingDisclosure.setFederalAgencyDepartment("");
        SponsorContract sponsor = pdDoc.getDevelopmentProposal().getSponsor();
        if (sponsor != null) {
            if (sponsor.getSponsorName() != null) {
                if (sponsor.getSponsorName().length() > SPONSOR_NAME_MAX_LENGTH) {
                    lobbyingDisclosure.setFederalAgencyDepartment(sponsor.getSponsorName().substring(0, SPONSOR_NAME_MAX_LENGTH));
                }
                else {
                    lobbyingDisclosure.setFederalAgencyDepartment(sponsor.getSponsorName());
                }
            }
        }
        else {
            String primeSponsorCode = pdDoc.getDevelopmentProposal().getPrimeSponsor().getSponsorCode();
            if (primeSponsorCode != null) {
                SponsorContract primeSponsor = pdDoc.getDevelopmentProposal().getPrimeSponsor();
                if (primeSponsor.getSponsorName() != null) {
                    if (primeSponsor.getSponsorName().length() > SPONSOR_NAME_MAX_LENGTH) {
                        lobbyingDisclosure.setFederalAgencyDepartment(primeSponsor.getSponsorName().substring(0, SPONSOR_NAME_MAX_LENGTH));
                    }
                    else {
                        lobbyingDisclosure.setFederalAgencyDepartment(primeSponsor.getSponsorName());
                    }
                }
            }
        }
        lobbyingDisclosure.setFederalProgramName(getFedProgramName());
        lobbyingDisclosure.setLobbyingRegistrant(getLobbyingRegistrant());
        lobbyingDisclosure.setIndividualsPerformingServices(getIndividualsPerformingServices());
        lobbyingDisclosure.setSignatureBlock(getSignatureBlock());
        lobbyinADDocument.setLobbyingActivitiesDisclosure(lobbyingDisclosure);
        return lobbyinADDocument;
    }

    private SignatureBlock getSignatureBlock() {

        SignatureBlock signatureBlock = SignatureBlock.Factory.newInstance();
        signatureBlock.setName(globLibV20Generator.getHumanNameDataType(aorInfo));
        if (aorInfo.getPrimaryTitle() != null) {
            signatureBlock.setTitle(StringUtils.substring(aorInfo.getPrimaryTitle(), 0, PRIMARY_TITLE_MAX_LENGTH));
        }
        signatureBlock.setSignature(aorInfo.getFullName());

        signatureBlock.setSignedDate(aorInfo.getSubmittedDate());
        return signatureBlock;
    }

    private LobbyingRegistrant getLobbyingRegistrant() {

        LobbyingRegistrant lobbyingRegistrant = LobbyingRegistrant.Factory.newInstance();
        lobbyingRegistrant.setOrganizationName(NOT_APPLICABLE);
        return lobbyingRegistrant;
    }

    private IndividualsPerformingServices getIndividualsPerformingServices() {

        // we don't have info for Individuals Performing Services. This is a required field, so we set it to N/A and add 1 element
        // to array
        IndividualsPerformingServices individualServices = IndividualsPerformingServices.Factory.newInstance();
        Individual individual = Individual.Factory.newInstance();
        HumanNameDataType humanName = HumanNameDataType.Factory.newInstance();
        humanName.setFirstName(NOT_APPLICABLE);
        humanName.setLastName(NOT_APPLICABLE);
        Individual[] individualArray = new Individual[1];
        individual.setName(humanName);
        individualArray[0] = individual;
        individualServices.setIndividualArray(individualArray);
        return individualServices;
    }

    /**
     * 
     * This method returns FederalProgramName informations including ProgramAnnouncementTitle and CFDANumber for the
     * FederalProgramName.
     * 
     * @return FederalProgramName Federal Program details.
     */
    private FederalProgramName getFedProgramName() {

        FederalProgramName federalProgramName = FederalProgramName.Factory.newInstance();                  
        if (pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle() != null) {
            String announcementTitle;
            if (pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle()
                    .length() > PROGRAM_ANNOUNCEMENT_TITLE_MAX_LENGTH) {
                announcementTitle = pdDoc.getDevelopmentProposal()
                        .getProgramAnnouncementTitle().substring(0,PROGRAM_ANNOUNCEMENT_TITLE_MAX_LENGTH);
            } else {
                announcementTitle = pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle();
            }
            federalProgramName.setFederalProgramDescription(announcementTitle);
        }
        final String cfdaNumber = getFirstCfdaNumber();
        if (cfdaNumber != null) {
            federalProgramName.setCFDANumber(cfdaNumber);
        }
        return federalProgramName;
    }

    private ReportEntity getReportEntity() {

        ReportEntity reportEntity = ReportEntity.Factory.newInstance();
        reportEntity.setReportEntityType(ReportEntityDataType.PRIME);
        reportEntity.setReportEntityIsPrime(YesNoDataType.Y_YES);
        Prime prime = Prime.Factory.newInstance();
        ProposalSiteContract proposalSite = pdDoc.getDevelopmentProposal().getApplicantOrganization();
        OrganizationContract organization = proposalSite.getOrganization();

        if (organization != null) {
            if (proposalSite.getLocationName() != null) {
                prime.setOrganizationName(StringUtils.substring(proposalSite.getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
            }
            if (organization.getCongressionalDistrict() != null) {
                if (organization.getCongressionalDistrict().length() > CONGRESSIONAL_DISTRICT_MAX_LENGTH) {
                    prime.setCongressionalDistrict(organization.getCongressionalDistrict().substring(0, CONGRESSIONAL_DISTRICT_MAX_LENGTH));
                }
                else {
                    prime.setCongressionalDistrict(organization.getCongressionalDistrict());
                }
            }
        }
        prime.setReportEntityType(ReportEntityDataType.PRIME);
        Address address = Address.Factory.newInstance();

        if (proposalSite != null) {
            if (proposalSite.getAddressLine1() != null) {
                if (proposalSite.getAddressLine1().length() > ADDRESS_LINE1_MAX_LENGTH) {
                    address.setStreet1(proposalSite.getAddressLine1().substring(0, ADDRESS_LINE1_MAX_LENGTH));
                }
                else {
                    address.setStreet1(proposalSite.getAddressLine1());
                }
            }
            if (proposalSite.getAddressLine2() != null) {
                if (proposalSite.getAddressLine2().length() > ADDRESS_LINE2_MAX_LENGTH) {
                    address.setStreet2(proposalSite.getAddressLine2().substring(0, ADDRESS_LINE2_MAX_LENGTH));
                }
                else {
                    address.setStreet2(proposalSite.getAddressLine2());
                }
            }
            if (proposalSite.getCity() != null) {
                if (proposalSite.getCity().length() > CITY_MAX_LENGTH) {
                    address.setCity(proposalSite.getCity().substring(0, CITY_MAX_LENGTH));
                }
                else {
                    address.setCity(proposalSite.getCity());
                }
            }
            globLibV20Generator.getStateCodeDataType(proposalSite.getCountryCode(), proposalSite.getState())
                    .ifPresent(address::setState);

            address.setZipPostalCode(proposalSite.getPostalCode());
        }
        prime.setAddress(address);
        reportEntity.setPrime(prime);
        return reportEntity;
    }

    @Override
    public LobbyingActivitiesDisclosureDocument getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        aorInfo = departmentalPersonService.getDepartmentalPerson(pdDoc);
        return getLobbyingActivitiesDisclosure();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public DocumentFactory<LobbyingActivitiesDisclosureDocument> factory() {
        return LobbyingActivitiesDisclosureDocument.Factory;
    }
}
