/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
/*
 Accessibility: Improve screen reader labels for
 - Credit Allocation table (row label + column header for each input/value)
 - Budget Periods & Totals table (column header for each cell, plus Totals)

 Vanilla JS only (no jQuery). Safe with KRAD partial refresh.
*/
(function(){
  "use strict";

  // ------------------------
  // Small, explicit helpers
  // ------------------------
  function trimText(s){
    return (s || "").replace(/\u00A0|&nbsp;/g, " ").replace(/\*/g, "").replace(/\s+/g, " ").trim();
  }
  function nodeName(el){ return (el && el.nodeName ? el.nodeName : "").toLowerCase(); }
  function isFormControl(el){ var n = nodeName(el); return n === "input" || n === "select" || n === "textarea"; }
  function hasClass(el, cls){ return !!(el && (" "+(el.className||"")+" ").indexOf(" "+cls+" ") >= 0); }
  function qsa(root, sel){ return Array.prototype.slice.call((root || document).querySelectorAll(sel)); }
  function qf(root, sel){ return (root || document).querySelector(sel); }
  function getAttr(el, name){ return el ? el.getAttribute(name) : null; }
  function setAttr(el, name, val){ if (el) el.setAttribute(name, val); }
  function removeAttr(el, name){ if (el) el.removeAttribute(name); }
  function getReadableText(el){
    if (!el) return "";
    var aria = getAttr(el, "aria-label");
    if (aria && aria.length) return trimText(aria);
    return trimText(el.textContent || "");
  }

  // ------------------------
  // Header mapping (per table)
  // ------------------------
  function buildHeaderMap(table){
    var map = [];
    var row = qf(table, "thead tr");
    if (!row) return map;
    var ths = qsa(row, "th");
    for (var i=0; i<ths.length; i++){
      var th = ths[i];
      var isAction = hasClass(th, "uif-collection-column-action");
      var labelEl = qf(th, "label");
      var text = getReadableText(labelEl || th);
      if (!th.id) { th.id = "a11y-th-" + i + "-" + Math.floor(Math.random()*1e6); }
      map.push({ id: th.id, text: trimText(text), isAction: isAction });
    }
    return map;
  }
  function headerForDataIndex(headerMap, tdDataIdx){
    var count = -1;
    for (var i=0; i<headerMap.length; i++){
      if (headerMap[i].isAction) continue;
      count++;
      if (count === tdDataIdx) return headerMap[i];
    }
    return { id: null, text: "" };
  }

  // ------------------------
  // Labeling helpers
  // ------------------------
  function composeLabel(base, el){
    var v = "";
    if (isFormControl(el)) v = trimText(el.value || "");
    else v = trimText(el.textContent || "");
    if (!v) return base;
    var b = trimText(base);
    // Avoid double colon when base already ends with ':'
    if (b && /[:：]$/.test(b)) return b + " " + v;
    return b ? (b + ": " + v) : v;
  }

  function labelFocusablesInCell(td, base){
    var list = qsa(td, "input,select,textarea,[tabindex]");
    if (!list.length) return false;
    for (var i=0; i<list.length; i++){
      var el = list[i];
      removeAttr(el, "aria-describedby");
      removeAttr(el, "aria-labelledby");
      setAttr(el, "aria-label", composeLabel(base, el));
      if (isFormControl(el)){
        el.addEventListener("input", function(){ setAttr(this, "aria-label", composeLabel(base, this)); });
        el.addEventListener("change", function(){ setAttr(this, "aria-label", composeLabel(base, this)); });
      }
    }
    if (td.hasAttribute("tabindex")){
      removeAttr(td, "aria-labelledby");
      setAttr(td, "aria-label", base);
    }
    return true;
  }

  function labelBudgetTotals(table, headerMap){
    var headers = [];
    for (var i=0; i<headerMap.length; i++){
      if (!headerMap[i].isAction) headers.push(headerMap[i].text);
    }
    var tfoot = qf(table, "tfoot");
    if (!tfoot) return;
    var rows = qsa(tfoot, "tr");
    for (var r=0; r<rows.length; r++){
      var cells = qsa(rows[r], "th,td");
      for (var c=0; c<cells.length; c++){
        var header = trimText(headers[c] || "");
        if (!header) continue;
        var valEl = qf(cells[c], '[data-role="totalValue"], .uif-message');
        if (valEl){
          var t = trimText(valEl.textContent || "");
          setAttr(valEl, "aria-label", header + " - Total" + (t ? ": " + t : ""));
        }
      }
    }
  }

  // ------------------------
  // Credit Allocation
  // ------------------------
  function enhanceCreditAllocation(root){
    var tables = qsa(root, "table.credit-allocation");
    for (var t=0; t<tables.length; t++){
      var table = tables[t];
      if (nodeName(table) !== "table") continue;
      var headerMap = buildHeaderMap(table);
      var rows = qsa(table, "tbody tr");
      for (var r=0; r<rows.length; r++){
        var tds = qsa(rows[r], "td");
        if (!tds.length) continue;
        // Row label from first cell
        var rowLabelEl = qf(tds[0], "[aria-label]") || qf(tds[0], "span,label,div");
        var rowLabel = trimText(getReadableText(rowLabelEl));
        var dataIdx = -1;
        for (var c=0; c<tds.length; c++){
          if (c === 0) continue; // skip corner cell
          var td = tds[c];
          if (hasClass(td, "uif-collection-column-action")) continue;
          dataIdx++;
          // In Credit Allocation the first TH is a blank corner; shift header mapping by +1
          var hdr = headerForDataIndex(headerMap, dataIdx + 1);
          var colHeader = trimText(hdr.text || "");
          var base = trimText(rowLabel && colHeader ? (rowLabel + " - " + colHeader) : (rowLabel || colHeader));
          if (!base) continue;
          if (!labelFocusablesInCell(td, base)){
            var val = qf(td, '[data-role="totalValue"], .uif-message, span,label,div');
            if (val){ removeAttr(val, "aria-labelledby"); setAttr(val, "aria-label", composeLabel(base, val)); }
          }
        }
      }
    }
  }

  // ------------------------
  // Budget Periods & Totals
  // ------------------------
  function isBudgetPeriodsTable(table){
    if (!hasClass(table, "uif-tableCollectionLayout")) return false;
    // Editable mode (inputs present)
    if (qsa(table,'tbody input[name^="budget.budgetPeriods["], tbody select[name^="budget.budgetPeriods["], tbody textarea[name^="budget.budgetPeriods["]').length > 0) return true;
    // View-only mode detection by headers
    var row = qf(table, 'thead tr'); if (!row) return false;
    var heads = qsa(row, 'th').map(function(th){ return trimText(getReadableText(th)); }).filter(Boolean);
    var cues = ['Period Start', 'Period End', 'Direct Cost', 'F&A Cost', 'Total Sponsor Cost', 'Cost Sharing'];
    var hits = 0;
    for (var i=0; i<cues.length; i++){
      for (var j=0; j<heads.length; j++){
        if (heads[j].indexOf(cues[i]) !== -1){ hits++; break; }
      }
    }
    return hits >= 2;
  }

  function enhanceBudgetPeriods(root){
    var tables = qsa(root, 'table.uif-tableCollectionLayout');
    for (var t=0; t<tables.length; t++){
      var table = tables[t];
      if (!isBudgetPeriodsTable(table)) continue;
      var headerMap = buildHeaderMap(table);
      var rows = qsa(table, 'tbody tr');
      for (var r=0; r<rows.length; r++){
        var tds = qsa(rows[r], 'td');
        var dataIdx = -1;
        for (var c=0; c<tds.length; c++){
          var td = tds[c];
          if (hasClass(td, 'uif-collection-column-action')) continue;
          dataIdx++;
          var hdr = headerForDataIndex(headerMap, dataIdx);
          var colHeader = trimText(hdr.text || '');
          if (!colHeader) continue;
          var base = colHeader;
          if (!labelFocusablesInCell(td, base)){
            var val = qf(td, '[data-role="totalValue"], .uif-message, span,label,div');
            if (val){ removeAttr(val, 'aria-labelledby'); setAttr(val, 'aria-label', composeLabel(base, val)); }
          }
        }
      }
      labelBudgetTotals(table, headerMap);
    }
  }

  function enhanceAll(root){
    var scope = (root && root.nodeType) ? root : document;
    enhanceBudgetPeriods(scope);
    enhanceCreditAllocation(scope);
  }

  // ------------------------
  // Bootstrapping & dynamic updates
  // ------------------------
  if (document.readyState === 'loading'){
    document.addEventListener('DOMContentLoaded', function(){ enhanceAll(document); });
  } else {
    enhanceAll(document);
  }

  // KRAD partial refresh hook (safe-guarded)
  try {
    if (window.kradVariables && window.kradVariables.PAGE_LOAD_EVENT && window.jQuery){
      window.jQuery(document).on(window.kradVariables.PAGE_LOAD_EVENT, function (event) {
        enhanceAll(event && event.target ? event.target : document);
      });
    }
  } catch (e) { /* ignore */ }

  // MutationObserver fallback (when content is injected and jQuery/KRAD not available yet)
  try {
    var mo = new MutationObserver(function(mutations){
      var needs = false;
      for (var i=0; i<mutations.length && !needs; i++){
        var m = mutations[i];
        for (var j=0; j<m.addedNodes.length; j++){
          var n = m.addedNodes[j];
          if (n.nodeType !== 1) continue;
          if ((n.matches && (n.matches('table.uif-tableCollectionLayout') || n.matches('table.credit-allocation'))) ||
              (n.querySelector && (n.querySelector('table.uif-tableCollectionLayout') || n.querySelector('table.credit-allocation')))){
            needs = true; break;
          }
        }
      }
      if (needs) enhanceAll(document);
    });
    mo.observe(document.body, { childList: true, subtree: true });
  } catch (e) { /* ignore */ }

  // Strong focus override: on focus inside Budget Periods, force aria-label to "<Header>: <value>"
  document.addEventListener('focusin', function(e){
    var target = e.target;
    var td = target && target.closest ? target.closest('td') : null;
    if (!td) return;
    var table = td.closest ? td.closest('table') : null;
    if (!table || !isBudgetPeriodsTable(table)) return;

    // compute dataIdx skipping action cols
    var dataIdx = -1; var found = false;
    var cells = td.parentElement ? td.parentElement.querySelectorAll('td') : [];
    for (var i=0; i<cells.length; i++){
      var c = cells[i];
      if (!hasClass(c, 'uif-collection-column-action')) dataIdx++;
      if (c === td){ found = true; break; }
    }
    if (!found || dataIdx < 0) return;

    var map = buildHeaderMap(table);
    var hdr = headerForDataIndex(map, dataIdx);
    var headerText = trimText(hdr.text || '');
    if (!headerText) return;

    // If a focusable wrapper (span/div) is used, avoid forcing form roles; rely on aria-label only
    if (!isFormControl(target)){
      // Remove misleading roles that might alter announcement
      // Do not set textbox/button/etc. to prevent SR from saying "edit"
      // If role is explicitly "group" or was set to "textbox" before, drop it
      var role = getAttr(target, 'role');
      if (role === 'group' || role === 'textbox') {
        removeAttr(target, 'role');
      }
    }

    removeAttr(target, 'aria-labelledby');
    setAttr(target, 'aria-label', composeLabel(headerText, target));
  }, true);
})();
