/*-
 * #%L
 * %%
 * Copyright (C) 2019 - 2020 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package org.kuali.research.gg.fetcher.fetch

import org.apache.logging.log4j.kotlin.Logging
import org.kuali.research.gg.fetcher.fetch.search.SearchParams
import org.kuali.research.gg.fetcher.fetch.search.SearchResults
import org.kuali.research.gg.fetcher.fetch.status.StatusResults
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.core.io.Resource
import org.springframework.stereotype.Service
import org.springframework.web.client.RestTemplate

@Service
class GrantsGovFetchServiceImpl(@Autowired val restTemplate: RestTemplate) : GrantsGovFetchService, Logging {

    override fun search(params: SearchParams): SearchResults = restTemplate.postForEntity(FORMS_SEARCH_ENDPOINT, params, SearchResults::class.java).body!!
    override fun status(): StatusResults = restTemplate.getForEntity(FORMS_STATUS_ENDPOINT, StatusResults::class.java).body!!

    override fun fetchPdf(formName: String, formVersion: String, formId: String): Map<String, Resource> {
        val (names, versions) = getNamesVersions(formName, formVersion)
        val cp = cartesianProduct(names, versions)

        val restResource = fetch("$PDF_DOWNLOAD_ENDPOINT$formId")
        val restResources = toMap(cp.map { (name, version) -> createPdfName(name, version) to restResource })
        val pathResources = toMap(cp.map { (name, version) -> fetchPdf(createPdfName(name, version)) })

        return restResources + pathResources
    }

    override fun fetchPdf(resourceName: String): Pair<String, Resource?> = resourceName to fetch("$PDF_BASE_URL$resourceName")

    override fun fetchDat(formName: String, formVersion: String, formId: String): Map<String, Resource> {
        val (names, versions) = getNamesVersions(formName, formVersion)
        val cp = cartesianProduct(names, versions)

        val restResource = fetch("$DAT_DOWNLOAD_ENDPOINT$formId")
        val restResources = toMap(cp.map { (name, version) -> createDatName(name, version, formId) to restResource })
        val pathResources = toMap(cp.map { (name, version) -> fetchDat(createDatName(name, version, formId)) })

        return restResources + pathResources
    }

    override fun fetchDat(resourceName: String): Pair<String, Resource?> = resourceName to fetch("$DAT_BASE_URL$resourceName")

    override fun fetchInstructions(formName: String, formVersion: String, formId: String): Map<String, Resource> {
        val (names, versions) = getNamesVersions(formName, formVersion)
        val cp = cartesianProduct(names, versions)

        val restResource = fetch("$INSTRUCTIONS_DOWNLOAD_ENDPOINT$formId")
        val restResources = toMap(cp.map { (name, version) -> createInstructionsName(name, version) to restResource })
        val pathResources = toMap(cp.map { (name, version) -> fetchInstructions(createInstructionsName(name, version)) })

        return restResources + pathResources
    }

    override fun fetchInstructions(resourceName: String): Pair<String, Resource?> = resourceName to fetch("$INSTRUCTIONS_BASE_URL$resourceName")

    override fun fetchSchema(formName: String, formVersion: String, formId: String): Map<String, Resource> {
        val (names, versions) = getNamesVersions(formName, formVersion)
        val cp = cartesianProduct(names, versions)

        val restResource = fetch("$SCHEMA_DOWNLOAD_ENDPOINT$formId")
        val restResources = toMap(cp.map { (name, version) -> createSchemaName(name, version) to restResource })
        val pathResources = toMap(cp.map { (name, version) -> fetchSchema(createSchemaName(name, version)) })

        return restResources + pathResources
    }

    override fun fetchSchema(resourceName: String): Pair<String, Resource?> = resourceName to fetch("$SCHEMA_BASE_URL$resourceName")

    override fun fetchNihXsl(formName: String, formVersion: String, formId: String): Map<String, Resource> {
        val (names, versions) = getNamesVersions(formName, formVersion)
        val cp = cartesianProduct(names, versions)

        return toMap(cp.map { (name, version) -> fetchNihXsl(createNihXslName(name, version)) })
    }

    override fun fetchNihXsl(resourceName: String): Pair<String, Resource?> = resourceName to fetch("$NIH_XSL_BASE_URL$resourceName")

    override fun fetchXsl(formName: String, formVersion: String, formId: String): Map<String, Resource> {
        val (names, versions) = getNamesVersions(formName, formVersion)
        val cp = cartesianProduct(names, versions)

        return toMap(cp.map { (name, version) -> fetchXsl(createXslName(name, version)) })
    }

    override fun fetchXsl(resourceName: String): Pair<String, Resource?> = resourceName to fetch("$XSL_BASE_URL$resourceName")

    private fun fetch(url: String): Resource? = try {
            with(restTemplate.getForEntity(url, Resource::class.java).body, {
                return if (this == null || !this.exists() || this.contentLength() <= 0) {
                    logger.info("Resource not found at: $url")
                    null
                } else {
                    val contents = this.inputStream.readAllBytes()
                    if (contents == null || contents.isEmpty()) {
                        logger.info("Resource not found at: $url")
                        null
                    } else {
                        val contentsStr = String(contents)
                        if (contentsStr.contains("<!DOCTYPE html>") || contentsStr.contains("ErrorDetail.jsp") || contentsStr.contains("FormNamespaceRDDL.jsp")) {
                            logger.info("Resource not found at: $url")
                            null
                        } else {
                            logger.info("Resource found at: $url")
                            this
                        }
                    }
                }
            })
        } catch (e: Exception) {
            logger.info("Resource not found at: $url")
            null
        }

    fun cartesianProduct(names: Set<String>, versions: Set<String>): Set<Pair<String, String>> {
        val product = mutableSetOf<Pair<String, String>>()

        names.forEach { name ->
            versions.forEach { version ->
                product += (name to version)
            }
        }

        return product
    }

    private fun toMap(l: List<Pair<String, Resource?>>): Map<String, Resource> = l.filterNot { (_, v) -> v == null }.map { (k, v) -> k to v!! }.toMap()

    private fun getNamesVersions(formName: String, formVersion: String) = cleanSplit(formName, ",") to cleanSplit(formVersion, ",")

    private fun cleanSplit(string: String, delim: String): Set<String> = string.split(delim.toRegex()).map { it.trim() }.filterNot { it.isBlank() }.toSet()

    private fun createPdfName(formName: String, formVersion: String): String = "$formName-V$formVersion.pdf"
    private fun createDatName(formName: String, formVersion: String, formId: String): String = formName + "-V" + formVersion + "_F" + formId + ".xls"
    private fun createInstructionsName(formName: String, formVersion: String): String = "$formName-V$formVersion-Instructions.pdf"
    private fun createSchemaName(formName: String, formVersion: String): String = "$formName-V$formVersion.xsd"
    private fun createNihXslName(formName: String, formVersion: String): String = "$formName-V$formVersion.xsl"
    private fun createXslName(formName: String, formVersion: String): String = "$formName-V$formVersion.fo.xsl"

    companion object {
        private const val FORMS_SEARCH_ENDPOINT = "https://www.grants.gov/grantsws/rest/forms/search/"
        private const val FORMS_STATUS_ENDPOINT = "https://www.grants.gov/grantsws/rest/forms/report/status/"
        private const val PDF_DOWNLOAD_ENDPOINT = "https://www.grants.gov/grantsws/rest/forms/download/pdf/"
        private const val INSTRUCTIONS_DOWNLOAD_ENDPOINT = "https://www.grants.gov/grantsws/rest/forms/download/instructions/"
        private const val SCHEMA_DOWNLOAD_ENDPOINT = "https://www.grants.gov/grantsws/rest/forms/download/schema/"
        private const val DAT_DOWNLOAD_ENDPOINT = "https://www.grants.gov/grantsws/rest/forms/download/dat/"
        private const val PDF_BASE_URL: String = "https://apply07.grants.gov/apply/forms/sample/"
        private const val DAT_BASE_URL: String = "https://apply07.grants.gov/apply/forms/sample/"
        private const val INSTRUCTIONS_BASE_URL: String = "https://apply07.grants.gov/apply/forms/instructions/"
        private const val SCHEMA_BASE_URL: String = "https://apply07.grants.gov/apply/forms/schemas/"
        private const val NIH_XSL_BASE_URL: String = "https://grants.nih.gov/grants/ElectronicReceipt/files/"
        private const val XSL_BASE_URL: String = "https://apply07.grants.gov/apply/forms/fo/"
    }
}
